/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2007 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.webui.jsf.component.customizers;

import java.awt.Color;
import java.awt.Component;
import java.awt.event.ItemEvent;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.ResourceBundle;
import com.sun.webui.jsf.model.Option;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.util.EventObject;
import javax.swing.DefaultCellEditor;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.JToggleButton;
import javax.swing.ListSelectionModel;
import javax.swing.UIManager;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;
import javax.faces.model.SelectItem;
import javax.swing.AbstractCellEditor;


/**
 * A panel for editing default data beans that support selector-based components,
 * such as listboxes and dropdowns. 
 *
 * @author gjmurphy
 */

public class OptionsListPanel extends JPanel {
    
    static ResourceBundle bundle = 
        ResourceBundle.getBundle(OptionsListPanel.class.getPackage().getName() + ".Bundle-DT"); //NOI18N
    
    private ItemsTableModel tableModel;
    
    private boolean acceptsMultipleChoice;
    
    /**
     * Create a new panel for editing select items, for a component that
     * supports only single selection (i.e. the component has no
     * <code>multiple</code> property or its equivalent).
     */
    public OptionsListPanel() {
        this(false);
    }
    
    /**
     * Create a new panel for editing select items. If the component to which
     * the items are assigned supports multiple selection (has a 
     * <code>multiple</code> property), specify <code>true</code> for the
     * <code>acceptsMultipleChoice</code> parameter.
     */
    public OptionsListPanel(boolean acceptsMultipleChoice) {
        this.tableModel = new ItemsTableModel();
        this.acceptsMultipleChoice = acceptsMultipleChoice;
        initComponents();
        
        // Set customized cell editor for the Selected column
        SelectedCellEditor selectedCellEditor = new SelectedCellEditor( selectValuesCheckBox,  multipleChoiceCheckBox );
        dataTable.setDefaultEditor( Boolean.class, selectedCellEditor );
    }
    
    /** This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        dataPane = new javax.swing.JScrollPane();
        dataTable = new JTable(tableModel);

        // Only one row selectable at a time
        dataTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        // Create a custom renderer for column headers
        DefaultTableCellRenderer renderer = new DefaultTableCellRenderer();
        renderer.setHorizontalAlignment(javax.swing.JLabel.CENTER);
        renderer.setBackground(new Color(212, 208, 200));
        dataTable.getTableHeader().setDefaultRenderer(renderer);

        // Create a default cell editor for String values that forces a stop
        // editing event whenever focus is lost, and that has same background
        // color as renderer
        JTextField textField = new JTextField();
        final TableCellEditor stringEditor =new CellEditor(dataTable, textField);
        dataTable.setDefaultEditor(String.class, stringEditor);

        dataTable.putClientProperty("terminateEditOnFocusLost", Boolean.TRUE); //NOI18N

        // Single click to start editing
        ((DefaultCellEditor)dataTable.getDefaultEditor(String.class)).setClickCountToStart( 1 );

        // Create a default cell renderer for String values that consistently renders
        // background colors.
        dataTable.setDefaultRenderer(String.class, new HomogonousCellRenderer());

        // Set a renderer for the "selected" column that uses
        // radio buttons if in single select mode, and checkboxes if
        // in multiple select mode.
        TableCellRenderer booleanRenderer = new SelectedCellRenderer();
        dataTable.setDefaultRenderer(Boolean.class, booleanRenderer);

        buttonsPanel = new javax.swing.JPanel();
        newButton = new javax.swing.JButton();
        deleteButton = new javax.swing.JButton();
        upButton = new javax.swing.JButton();
        downButton = new javax.swing.JButton();
        checkBoxesPanel = new javax.swing.JPanel();
        selectValuesCheckBox = new javax.swing.JCheckBox();
        multipleChoiceCheckBox = new javax.swing.JCheckBox();

        setLayout(new java.awt.GridBagLayout());

        dataPane.setEnabled(false);
        dataTable.setAutoResizeMode(javax.swing.JTable.AUTO_RESIZE_ALL_COLUMNS);
        dataPane.setViewportView(dataTable);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 10, 0);
        add(dataPane, gridBagConstraints);

        buttonsPanel.setLayout(new java.awt.GridLayout(4, 1, 0, 5));

        newButton.setMnemonic(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("NEW_BUTTON_MNEMONIC").charAt(0));
        newButton.setText(bundle.getString("OptionsListPanel.button.new"));
        newButton.setLabel(bundle.getString("OptionsListPanel.button.new"));
        newButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                newActionPerformed(evt);
            }
        });

        buttonsPanel.add(newButton);
        newButton.getAccessibleContext().setAccessibleName(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.button.new"));
        newButton.getAccessibleContext().setAccessibleDescription(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.button.new"));

        deleteButton.setMnemonic(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("DELETE_BUTTON_MNEMONIC").charAt(0));
        deleteButton.setText(bundle.getString("OptionsListPanel.button.delete"));
        deleteButton.setLabel(bundle.getString("OptionsListPanel.button.delete"));
        deleteButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                deleteActionPerformed(evt);
            }
        });

        buttonsPanel.add(deleteButton);
        deleteButton.getAccessibleContext().setAccessibleName(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.button.delete"));
        deleteButton.getAccessibleContext().setAccessibleDescription(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.button.delete"));

        upButton.setMnemonic(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("UP_BUTTON_MNEMONIC").charAt(0));
        upButton.setText(bundle.getString("OptionsListPanel.button.up"));
        upButton.setLabel(bundle.getString("OptionsListPanel.button.up"));
        upButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                upActionPerformed(evt);
            }
        });

        buttonsPanel.add(upButton);
        upButton.getAccessibleContext().setAccessibleName(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.button.up"));
        upButton.getAccessibleContext().setAccessibleDescription(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.button.up"));

        downButton.setMnemonic(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("DOWN_BUTTON_MNEMONIC").charAt(0));
        downButton.setText(bundle.getString("OptionsListPanel.button.down"));
        downButton.setLabel(bundle.getString("OptionsListPanel.button.down"));
        downButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                downActionPerformed(evt);
            }
        });

        buttonsPanel.add(downButton);
        downButton.getAccessibleContext().setAccessibleName(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.button.down"));
        downButton.getAccessibleContext().setAccessibleDescription(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.button.down"));

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(10, 6, 0, 10);
        add(buttonsPanel, gridBagConstraints);

        checkBoxesPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.LEFT, 6, 0));

        selectValuesCheckBox.setMnemonic(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("SELECT_VALUES_CHECKBOX_MNEMONIC").charAt(0));
        selectValuesCheckBox.setLabel(bundle.getString("OptionsListPanel.checkbox.select.label"));
        selectValuesCheckBox.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                selectValuesCheckBoxStateChanged(evt);
            }
        });
        selectValuesCheckBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                selectValuesCheckBoxActionPerformed(evt);
            }
        });

        checkBoxesPanel.add(selectValuesCheckBox);
        selectValuesCheckBox.getAccessibleContext().setAccessibleName(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.checkbox.select.label"));
        selectValuesCheckBox.getAccessibleContext().setAccessibleDescription(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.checkbox.select.label"));

        multipleChoiceCheckBox.setMnemonic(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("MULTIPLE_CHOICE_CHECKBOX_MNEMONIC").charAt(0));
        multipleChoiceCheckBox.setEnabled(acceptsMultipleChoice && this.selectValuesCheckBox.isSelected());
        multipleChoiceCheckBox.setLabel(bundle.getString("OptionsListPanel.checkbox.multiple.label"));
        multipleChoiceCheckBox.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                multipleChoiceCheckBoxStateChanged(evt);
            }
        });

        checkBoxesPanel.add(multipleChoiceCheckBox);
        multipleChoiceCheckBox.getAccessibleContext().setAccessibleName(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.checkbox.multiple.label"));
        multipleChoiceCheckBox.getAccessibleContext().setAccessibleDescription(java.util.ResourceBundle.getBundle("com/sun/webui/jsf/component/customizers/Bundle-DT").getString("OptionsListPanel.checkbox.multiple.label"));

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        add(checkBoxesPanel, gridBagConstraints);

    }
    // </editor-fold>//GEN-END:initComponents

    private void selectValuesCheckBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_selectValuesCheckBoxActionPerformed
// TODO add your handling code here:
    }//GEN-LAST:event_selectValuesCheckBoxActionPerformed

    /**
     * When focus moves from table to a button, make sure that editor for string
     * cell is done editing (if it was in edit mode), to avoid problems when rows
     * are moved or added, and to make sure any edited data is saved in the 
     * correct row.
     */
    /**
     * Listen for mouse clicks on the table. When a mouse click occurs over a cell
     * in the "selected" column, update the cell value in the table model. This
     * listener is necessary because a custom renderer is used for the column, and
     * JTable does not pass mouse events to listeners on cell renderer components.
     */
    /**
     * Listen for a change in the value of the multipleChoice check box. If value
     * is toggled, the state of the table model must be changed to allow or disallow
     * multiple selections, and the state of the custom renderer for the selection
     * column must be changed to use the appropriate component (radioButton for
     * single selection, checkbox for multiple).
     */
    private void multipleChoiceCheckBoxStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_multipleChoiceCheckBoxStateChanged
        ItemsTableModel tableModel = (ItemsTableModel)dataTable.getModel();
        if (evt.getStateChange() == ItemEvent.SELECTED) {
            // Multiple-select mode
            tableModel.setMultipleSelect(true);
            ((SelectedCellRenderer) dataTable.getDefaultRenderer(Boolean.class)).setMultipleChoice(true);
        } else {
            // Single-select mode
            tableModel.setMultipleSelect(false);
            ((SelectedCellRenderer) dataTable.getDefaultRenderer(Boolean.class)).setMultipleChoice(false);
        }
        SelectItem[] items = tableModel.getItems();
        for( int i = 0; i < items.length; i++) {
            tableModel.fireTableCellUpdated(i, ItemsTableModel.SELECT_COLUMN);
        }
    }//GEN-LAST:event_multipleChoiceCheckBoxStateChanged

    /**
     * Listen for a change of value in the checkbox that indicates that the user
     * does or does not wish to preselect items. If the user chooses to select values,
     * then the select column is enabled; otherewise it is disabled.
     */
    private void selectValuesCheckBoxStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_selectValuesCheckBoxStateChanged
        ItemsTableModel tableModel = (ItemsTableModel)dataTable.getModel();
        int selectedRow = dataTable.getSelectedRow();
        if (evt.getStateChange() == ItemEvent.SELECTED) {
            tableModel.setItemsSelectable(true);
            this.adjustColumnWidth(dataTable, ItemsTableModel.SELECT_COLUMN);
            dataTable.revalidate();
            if (acceptsMultipleChoice) {
                this.multipleChoiceCheckBox.setEnabled(true);
            }
        } else {
            this.multipleChoiceCheckBox.setEnabled(false);
            tableModel.setItemsSelectable(false);
        }
        dataTable.changeSelection(selectedRow, 0, false, false);
    }//GEN-LAST:event_selectValuesCheckBoxStateChanged

    private void downActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_downActionPerformed
        ItemsTableModel tableModel = (ItemsTableModel)dataTable.getModel();
        if (dataTable.getSelectedRowCount() > 0) {
            int[] selectedRows = dataTable.getSelectedRows();
            tableModel.moveRowDown(selectedRows[0]);
            dataTable.setModel(tableModel);
            dataTable.changeSelection(selectedRows[0] + 1, 0, false, false);
        }
    }//GEN-LAST:event_downActionPerformed

    private void upActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_upActionPerformed
        ItemsTableModel tableModel = (ItemsTableModel)dataTable.getModel();
        if (dataTable.getSelectedRowCount() > 0) {
            int[] selectedRows = dataTable.getSelectedRows();
            tableModel.moveRowUp(selectedRows[0]);
            dataTable.setModel(tableModel);
            dataTable.changeSelection(selectedRows[0] - 1, 0, false, false);
        }
    }//GEN-LAST:event_upActionPerformed

    private void deleteActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_deleteActionPerformed
        ItemsTableModel tableModel = (ItemsTableModel)dataTable.getModel();
        int numRows = tableModel.getRowCount();
        if (dataTable.getSelectedRowCount() > 0) {
            int[] selectedRows = dataTable.getSelectedRows();
            boolean lastRowRemoved = (selectedRows[0]==(numRows-1)) ? true : false;
            if (tableModel.canRemoveRow(selectedRows[0])) {
                tableModel.removeRow(selectedRows[0]);
                dataTable.setModel(tableModel);
                dataTable.changeSelection(selectedRows[0], 0, false, false);
            }
            
            // If this is not the last row we removed, have the next row selected
            if( tableModel.getRowCount() == 0 )
                return;
            else {
                if( lastRowRemoved ) {
                    // Have the last row selected again
                    dataTable.changeSelection(tableModel.getRowCount()-1, 0, false, false);
                } else {
                    // Have the same row index selected again
                    dataTable.changeSelection(selectedRows [0], 0, false, false);
                }
            }
        }
    }//GEN-LAST:event_deleteActionPerformed

    private void newActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_newActionPerformed
        ItemsTableModel tableModel = (ItemsTableModel)dataTable.getModel();
        if (tableModel.canAddRow()) {
            tableModel.addRow();
            dataTable.setModel(tableModel);
            dataTable.changeSelection(tableModel.getRowCount() - 1, 0, false, false);
            dataTable.requestFocusInWindow();
            
            // Let's not call this. This makes the new row acting very weird. -dongmei
            //dataTable.editCellAt(tableModel.getRowCount() - 1, ItemsTableModel.DISPLAY_COLUMN);
        }
    }//GEN-LAST:event_newActionPerformed

    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel buttonsPanel;
    private javax.swing.JPanel checkBoxesPanel;
    private javax.swing.JScrollPane dataPane;
    private javax.swing.JTable dataTable;
    private javax.swing.JButton deleteButton;
    private javax.swing.JButton downButton;
    private javax.swing.JCheckBox multipleChoiceCheckBox;
    private javax.swing.JButton newButton;
    private javax.swing.JCheckBox selectValuesCheckBox;
    private javax.swing.JButton upButton;
    // End of variables declaration//GEN-END:variables
    
    public void setItems(SelectItem[] items) {
        this.tableModel.setItems(items);    
    }
    
    public SelectItem[] getItems() {
        return this.tableModel.getItems();
    }
    
    /**
     * Setter for property <code>options</code>. If component uses items of
     * type {@link com.sun.webui.jsf.model.Option}, use this method to set them.
     */
    public void setOptions(Option[] options) {
        this.setItems((SelectItem[]) options);
        
        // Have the first row selected by default
        if( options.length > 0 )
            dataTable.changeSelection(0,0,false,false);
    }
    
    /**
     * Getter for property <code>options</code>.
     */
    public Option[] getOptions() {
        SelectItem[] items = getItems();
        Option[] options = new Option[items.length];
        for (int i = 0; i < items.length; i++)
            options[i] = new Option(items[i].getValue(), items[i].getLabel());
        return options;
    }


    /**
     * Returns true if the user has chosen to be able to select multiple items.
     * This value should correspond to the component's <code>multiple</code>
     * property (or equivalent). If the component has no <code>multiple</code>
     * property, this options list should have been constructed with 
     * <code>acceptsMultipleChoice</code> set to <code>false</code>, in which
     * case, this method will always return false.
     */
    public boolean isMultipleChoice() {
        return this.multipleChoiceCheckBox.getSelectedObjects() == null ? false : true;
    }

    /**
     * Set to true if the component is configured to select multiple items.
     * This value should correspond to the component's <code>multiple</code>
     * property (or equivalent).
     */
    public void setMultipleChoice(boolean multipleChoice) {
        this.multipleChoiceCheckBox.setSelected(multipleChoice);
    }
    
    /**
     * Returns true if this component is currently configured to edit the selected
     * state of its options.
     */
    public boolean isValueSelecting() {
        return this.selectValuesCheckBox.isSelected();
    }
    
    /**
     * Set to true to allow this compoment to edit the selected state of its
     * options. The user can override this setting by clicking on the \
     * selectValues checkbox.
     */
    public void setValueSelecting(boolean valueSelecting) {
        this.selectValuesCheckBox.setSelected(valueSelecting);
    }

    /**
     * Getter for property <code>selected</code>. If panel was configured to
     * accept multiple selections, and if {@link #isMultipleChoice} returns true,
     * then this method will return an array of objects. Otherwise, it will
     * return a scalar object. Returns <code>null</code> if no items were 
     * selected.
     */
    public Object getSelectedValues() {
        Object[] selectedValues = this.tableModel.getSelectedValues();
        if (selectedValues.length == 0)
            return null;
        return isMultipleChoice() ? selectedValues : selectedValues[0];
    }

    /**
     * Setter for property <code>selected</code>. If panel was configured to
     * accept multiple selections, and if {@link #isMultipleChoice} returns true,
     * then this method will check to see if the argument is an array of objects,
     * and if so, select all items with values corresponding to each of the
     * array objects. Set to null to clear all selections. If a value specified
     * does not correspond to a value returned by one of the items or options
     * getValue() method, it is ignored.
     */
    public void setSelectedValues(Object selectedValues)  {
        if (selectedValues == null) {
            this.tableModel.setSelectedValues(new Object[0]);
        } else if (isMultipleChoice() && selectedValues instanceof Object[]) {
            this.tableModel.setSelectedValues((Object[]) selectedValues);
        } else {
            this.tableModel.setSelectedValues(new Object[]{selectedValues});
        }
    }
    
    // Set prefered width of column to the width of its header. This is a
    // stop-gap. For non-boolean columns, should calculate widths of all rows,
    // and take maximum value.
    private void adjustColumnWidth(JTable table, int columnIndex) {
        TableColumnModel columnModel = table.getColumnModel();
        TableModel tableModel = table.getModel();
        TableColumn column = columnModel.getColumn(columnIndex);
        // Calculate width of header
        TableCellRenderer headerRenderer = column.getHeaderRenderer();
        if (headerRenderer == null)
            headerRenderer = table.getTableHeader().getDefaultRenderer();
        Component headerComponent = headerRenderer.getTableCellRendererComponent(
                table, column.getHeaderValue(), false, false, 0, columnIndex);
        int maxWidth = headerComponent.getPreferredSize().width;
        column.setPreferredWidth(maxWidth);
    }
    

    /**
     * Table model for a list of select items. Display and value properties 
     * are handled directly, in the first two columns. If displayed, the selected
     * property is in the third column. Editing actions on the selected column
     * are handled by listening for mouse click events on the column renderer.
     */
    static class ItemsTableModel extends AbstractTableModel {
        
        static final int DISPLAY_COLUMN = 0;
        static final int VALUE_COLUMN = 1;
        static final int SELECT_COLUMN = 2;
        
        int columnCount;
        boolean multipleSelect;
        ArrayList rows;
        
        ItemsTableModel() {
            rows = new ArrayList();
            columnCount = 2;
        }
        
        public SelectItem[] getItems() {
            SelectItem[] items = new SelectItem[rows.size()];
            for (int i = 0; i < rows.size(); i++) {
                items[i] = new SelectItem();
                items[i].setLabel(((ItemRowModel) rows.get(i)).getLabel());
                items[i].setValue(((ItemRowModel) rows.get(i)).getValue());
            }
            return items;
        }
        
        public void setItems(SelectItem[] items) {
            rows.clear();
            if (items != null) {
                for (int i = 0; i < items.length; i++) {
                    rows.add(new ItemRowModel(items[i].getLabel(), items[i].getValue()));
                }
            }
            this.fireTableDataChanged();
        }
        
        public void setItemsSelectable(boolean selectItems) {
            int newColumnCount = selectItems ? 3 : 2;
            if (newColumnCount != columnCount) {
                columnCount = newColumnCount;
                this.fireTableStructureChanged();
            }
        }
        
        public boolean isItemsSelectable() {
            return columnCount == 3 ? true : false;
        }

        public void setMultipleSelect(boolean multipleSelect) {
            if (!multipleSelect && this.multipleSelect) {
                int r = -1;
                for (int i = rows.size() - 1; i >= 0; i--) {
                    if (((ItemRowModel)rows.get(i)).isSelected()) {
                        ((ItemRowModel)rows.get(i)).setSelected(false);
                        this.fireTableCellUpdated(i, SELECT_COLUMN);
                        r = i;
                    }
                }
                if (r >= 0) {
                    ((ItemRowModel)rows.get(r)).setSelected(true);
                    this.fireTableCellUpdated(r, SELECT_COLUMN);
                }
            }
            this.multipleSelect = multipleSelect;
        }
        
        public void setSelectedValues(Object[] values) {
            ArrayList valuesList = new ArrayList(Arrays.asList(values));
            for (int i = 0; i < rows.size(); i++) {
                if (valuesList.contains(((ItemRowModel) rows.get(i)).getValue()))
                    ((ItemRowModel) rows.get(i)).setSelected(true);
                else
                    ((ItemRowModel) rows.get(i)).setSelected(false);
            }
        }
        
        public Object[] getSelectedValues() {
            ArrayList values = new ArrayList();
            for (int i = 0; i < rows.size(); i++) {
                if (((ItemRowModel) rows.get(i)).isSelected())
                    values.add(((ItemRowModel) rows.get(i)).getValue());
            }
            return values.toArray(new Object[values.size()]);
        }

        public Class getColumnClass(int columnIndex) {
            if (columnIndex == DISPLAY_COLUMN)
                return String.class;
            if (columnIndex == SELECT_COLUMN)
                return Boolean.class;
            if (rows.size() > 0)
                return ((ItemRowModel) rows.get(0)).getValue().getClass();
            return String.class;
        }

        public String getColumnName(int columnIndex) {
            if (columnIndex == DISPLAY_COLUMN)
                return bundle.getString("OptionsListPanel.table.column.display");
            else if (columnIndex == VALUE_COLUMN)
                return bundle.getString("OptionsListPanel.table.column.value");
            else if (columnIndex == SELECT_COLUMN)
                return bundle.getString("OptionsListPanel.table.column.selected");
            return null;
        }

        public int getColumnCount() {
            return columnCount;
        }

        public int getRowCount() {
            return rows.size();
        }
        
        public boolean canRemoveRow(int index) {
            return index >= 0 && index < rows.size();
        }

        public boolean removeRow(int index) {
            if (canRemoveRow(index)) {
                rows.remove(index);
                this.fireTableRowsDeleted(index, index);
                return true;
            }
            return false;
        }
        
        public boolean removeAllRows() {
            if( rows.isEmpty() )
                return true;
            
            int numRows = rows.size();
            rows.clear();
            this.fireTableRowsDeleted(0, numRows);
            
            return true;
        }

        public void setValueAt(Object newValue, int rowIndex, int columnIndex) {
            if (columnIndex == DISPLAY_COLUMN) {
                ((ItemRowModel) rows.get(rowIndex)).setLabel((String) newValue);
            } else if (columnIndex == VALUE_COLUMN) {
                ((ItemRowModel) rows.get(rowIndex)).setValue(newValue);
            } else if (columnIndex == SELECT_COLUMN) {
                if (!multipleSelect && Boolean.TRUE.equals(newValue)) {
                    for (int i = 0; i < rows.size(); i++) {
                        if (((ItemRowModel) rows.get(i)).isSelected()) {
                            ((ItemRowModel) rows.get(i)).setSelected(false);
                            this.fireTableCellUpdated(i, columnIndex);
                        }
                    }
                }
                ((ItemRowModel) rows.get(rowIndex)).setSelected(Boolean.TRUE.equals(newValue) ? true : false);
            }
            this.fireTableCellUpdated(rowIndex, columnIndex);
        }

        public Object getValueAt(int rowIndex, int columnIndex) {
            if (columnIndex == DISPLAY_COLUMN)
                return ((ItemRowModel) rows.get(rowIndex)).getLabel();
            else if (columnIndex == VALUE_COLUMN)
                return ((ItemRowModel) rows.get(rowIndex)).getValue();
            else if (columnIndex == SELECT_COLUMN)
                return ((ItemRowModel) rows.get(rowIndex)).isSelected() ? Boolean.TRUE : Boolean.FALSE;
            return null;
        }

        public boolean canAddRow() {
            return true;
        }

        public void addRow() {
            rows.add(new ItemRowModel(bundle.getString("OptionsListPanel.default.display"),
                    bundle.getString("OptionsListPanel.default.value")));
            this.fireTableRowsInserted(rows.size() - 1, rows.size() - 1);
        }

        public boolean canMoveRowUp(int index) {
            return index > 0 ? true : false;
        }

        public boolean canMoveRowDown(int index) {
            return index < rows.size() - 1 ? true : false;
        }
        
        public void moveRowUp(int index) {
            if (canMoveRowUp(index)) {
                Object row = rows.remove(index);
                rows.add(index - 1, row);
                this.fireTableDataChanged();
            }
        }
      
        public void moveRowDown(int index) {
            if (canMoveRowDown(index)) {
                Object row = rows.remove(index);
                rows.add(index + 1, row);
                this.fireTableDataChanged();
            }
        }

        public boolean isCellEditable(int rowIndex, int columnIndex) {
            return true;
        }
        
    }
    
    
    static class ItemRowModel {
        
        ItemRowModel(String label, Object value) {
            this.setLabel(label);
            this.setValue(value);
        }
        
        private String label;

        public String getLabel() {
            return this.label;
        }

        public void setLabel(String label) {
            this.label = label;
        }

        private Object value;

        public Object getValue() {
            return this.value;
        }

        public void setValue(Object value) {
            this.value = value;
        }

        private boolean selected;

        public boolean isSelected() {
            return this.selected;
        }

        public void setSelected(boolean selected) {
            this.selected = selected;
        }
        
    }
    
    /** A cell renderer for the "selected" column. By default, radio
     * buttons are used to render cell values. If set to multiple choice
     * mode, check boxes are used.
     */
    class SelectedCellRenderer extends DefaultTableCellRenderer {
        
        Color SELECTION_BACKGROUND =
                UIManager.getDefaults().getColor("TextField.selectionBackground");
        
        JCheckBox checkBox;
        JRadioButton radioButton;
        JToggleButton currentComponent;
        
        SelectedCellRenderer() {
            checkBox = new JCheckBox();
            checkBox.setHorizontalAlignment(JCheckBox.CENTER);
            checkBox.setBackground(Color.WHITE);
            radioButton = new JRadioButton();
            radioButton.setHorizontalAlignment(JRadioButton.CENTER);
            radioButton.setBackground(Color.WHITE);
            currentComponent = radioButton;
        }
        
        public void setMultipleChoice(boolean multipleChoice) {
            if (multipleChoice)
                currentComponent = checkBox;
            else
                currentComponent = radioButton;
        }
        
        public boolean isMultipleChoice() {
            return currentComponent == checkBox ? true : false;
        }

        public Component getTableCellRendererComponent(JTable table, Object value,
                boolean isSelected, boolean hasFocus, int row, int column) {
            if (isSelected)
                currentComponent.setBackground(SELECTION_BACKGROUND);
            else
                currentComponent.setBackground(Color.WHITE);
            if (Boolean.TRUE.equals(value)) {
                currentComponent.setSelected(true);
            } else {
                currentComponent.setSelected(false);
            }
            return currentComponent;
        }
        
    }
    
    /**
     * A delegating renderer class that consistently sets the background color
     * of cells to reflect "selected" and "unselected" states.
     */
    class HomogonousCellRenderer extends DefaultTableCellRenderer {
        
        Color SELECTION_BACKGROUND =
                UIManager.getDefaults().getColor("TextField.selectionBackground");
        
        public Component getTableCellRendererComponent(JTable table, Object value,
                boolean isSelected, boolean hasFocus, int row, int column) {
            Component c = super.getTableCellRendererComponent(table,  value, isSelected, hasFocus, row, column);
            if (isSelected)
                c.setBackground(SELECTION_BACKGROUND);
            else
                c.setBackground(Color.WHITE);
            return c;
        }
    }
    
    /**
     * A JTextField cell editor by default all the text is selected
     */
    class CellEditor extends DefaultCellEditor  {
        
        private JTable table;
        
        public CellEditor( JTable table, JTextField component ) {
            super( component );
            this.table = table;
        }
    
        // This method is called when a cell value is edited by the user.
        public Component getTableCellEditorComponent(JTable table, Object value,
                boolean isSelected, int rowIndex, int vColIndex) {
    
            // Configure the component with the specified value
            ((JTextField)super.getComponent()).setText((String)value);
            
            if (isSelected) {
                ((JTextField)super.getComponent()).selectAll();
                table.repaint();
            }
    
            // Return the configured component
            return super.getComponent();
        }
    }
    
    // Cutomized Cell Editor for the Selected column
    class SelectedCellEditor extends AbstractCellEditor implements TableCellEditor, ActionListener {
        
        // These are the components that will handle the editing of the cell value
        JRadioButton radioBtn;
        JCheckBox checkbox;
        
        JCheckBox selectValuesCheckBox, multipleChoiceCheckBox;
        
        public SelectedCellEditor( JCheckBox selectValuesCheckBox, JCheckBox multipleChoiceCheckBox ) {
            this.selectValuesCheckBox = selectValuesCheckBox;
            this.multipleChoiceCheckBox = multipleChoiceCheckBox;  
            
            radioBtn = new JRadioButton();
            checkbox = new JCheckBox();
            radioBtn.setHorizontalAlignment( JRadioButton.CENTER );
            checkbox.setHorizontalAlignment(JCheckBox.CENTER);
            
            radioBtn.addActionListener( (ActionListener)this );
            checkbox.addActionListener( (ActionListener)this );
            
        }
    
        // This method is called when a cell value is edited by the user.
        public Component getTableCellEditorComponent(JTable table, Object value,
                boolean isSelected, int rowIndex, int vColIndex) {
    
            if( selectValuesCheckBox.isSelected() && multipleChoiceCheckBox.isSelected() ) {
                // Must use JCheckBox as the cell editor
                checkbox.setSelected( ((Boolean)value).booleanValue());
                return checkbox;
                
            } else {
                
                radioBtn.setSelected( ((Boolean)value).booleanValue() );
                return radioBtn;
            }
        }
    
        // This method is called when editing is completed.
        // It must return the new value to be stored in the cell.
        public Object getCellEditorValue() {
            
            if( selectValuesCheckBox.isSelected() && multipleChoiceCheckBox.isSelected() ) {
                // Must use JCheckBox as the cell editor
                return new Boolean( checkbox.isSelected() );
                
            } else
                return new Boolean( radioBtn.isSelected() );
        }
        
        public boolean isCellEditable(EventObject evt) {
            if (evt instanceof MouseEvent) {
                
                // 1 click to activate the editor
                int clickCount = 1;
                return ((MouseEvent)evt).getClickCount() >= clickCount;
            }
            return true;
        }
        
        
        public void actionPerformed(ActionEvent e) {
           // Editing ends when an action is performed
           fireEditingStopped();
        }
    }

}
