use std::marker::PhantomData;
use std::mem::MaybeUninit;
use std::ptr::null;

use crate::Boolean;
use crate::Context;
use crate::Integer;
use crate::Local;
use crate::Script;
use crate::String;
use crate::ToLocal;
use crate::Value;

/// The origin, within a file, of a script.
#[repr(C)]
pub struct ScriptOrigin<'sc>([usize; 7], PhantomData<&'sc ()>);

extern "C" {
  fn v8__Script__Compile(
    context: *mut Context,
    source: *mut String,
    origin: *const ScriptOrigin,
  ) -> *mut Script;
  fn v8__Script__Run(this: &mut Script, context: *mut Context) -> *mut Value;

  fn v8__ScriptOrigin__CONSTRUCT(
    buf: &mut MaybeUninit<ScriptOrigin>,
    resource_name: *mut Value,
    resource_line_offset: *mut Integer,
    resource_column_offset: *mut Integer,
    resource_is_shared_cross_origin: *mut Boolean,
    script_id: *mut Integer,
    source_map_url: *mut Value,
    resource_is_opaque: *mut Boolean,
    is_wasm: *mut Boolean,
    is_module: *mut Boolean,
  );
}

impl Script {
  /// A shorthand for ScriptCompiler::Compile().
  pub fn compile<'sc>(
    scope: &mut impl ToLocal<'sc>,
    mut context: Local<Context>,
    mut source: Local<String>,
    origin: Option<&ScriptOrigin>,
  ) -> Option<Local<'sc, Script>> {
    // TODO: use the type system to enforce that a Context has been entered.
    // TODO: `context` and `source` probably shouldn't be mut.
    let ptr = unsafe {
      v8__Script__Compile(
        &mut *context,
        &mut *source,
        origin.map(|r| r as *const _).unwrap_or(null()),
      )
    };
    unsafe { scope.to_local(ptr) }
  }

  /// Runs the script returning the resulting value. It will be run in the
  /// context in which it was created (ScriptCompiler::CompileBound or
  /// UnboundScript::BindToCurrentContext()).
  pub fn run<'sc>(
    &mut self,
    scope: &mut impl ToLocal<'sc>,
    mut context: Local<Context>,
  ) -> Option<Local<'sc, Value>> {
    unsafe { scope.to_local(v8__Script__Run(self, &mut *context)) }
  }
}

/// The origin, within a file, of a script.
impl<'sc> ScriptOrigin<'sc> {
  #[allow(clippy::too_many_arguments)]
  pub fn new(
    mut resource_name: Local<'sc, Value>,
    mut resource_line_offset: Local<'sc, Integer>,
    mut resource_column_offset: Local<'sc, Integer>,
    mut resource_is_shared_cross_origin: Local<'sc, Boolean>,
    mut script_id: Local<'sc, Integer>,
    mut source_map_url: Local<'sc, Value>,
    mut resource_is_opaque: Local<'sc, Boolean>,
    mut is_wasm: Local<'sc, Boolean>,
    mut is_module: Local<'sc, Boolean>,
  ) -> Self {
    unsafe {
      let mut buf = std::mem::MaybeUninit::<ScriptOrigin>::uninit();
      v8__ScriptOrigin__CONSTRUCT(
        &mut buf,
        &mut *resource_name,
        &mut *resource_line_offset,
        &mut *resource_column_offset,
        &mut *resource_is_shared_cross_origin,
        &mut *script_id,
        &mut *source_map_url,
        &mut *resource_is_opaque,
        &mut *is_wasm,
        &mut *is_module,
      );
      buf.assume_init()
    }
  }
}
