package io.quarkus.paths;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.fail;

import io.quarkus.fs.util.ZipUtils;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashSet;
import java.util.Set;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

public class JarPathTreeTest {

    private static final String BASE_DIR = "paths/directory-path-tree";

    private static Path root;

    @BeforeAll
    public static void staticInit() throws Exception {
        final URL url = Thread.currentThread().getContextClassLoader().getResource(BASE_DIR + "/root");
        if (url == null) {
            throw new IllegalStateException("Failed to locate " + BASE_DIR + " on the classpath");
        }
        final Path rootDir = Path.of(url.toURI()).toAbsolutePath();
        if (!Files.exists(rootDir)) {
            throw new IllegalStateException("Failed to locate " + rootDir);
        }

        root = rootDir.getParent().resolve("root.jar");
        ZipUtils.zip(rootDir, root);
    }

    @Test
    public void acceptExistingPath() throws Exception {
        final PathTree tree = PathTree.ofDirectoryOrArchive(root);
        tree.accept("README.md", visit -> {
            assertThat(visit).isNotNull();
            assertThat(visit.getRelativePath("/")).isEqualTo("README.md");
            assertThat(visit.getPath()).exists();
            assertThat(visit.getRoot()).isEqualTo(root);
            try {
                assertThat(Files.readString(visit.getPath())).isEqualTo("test readme");
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        });
    }

    @Test
    public void acceptNonExistentPath() throws Exception {
        final PathTree tree = PathTree.ofDirectoryOrArchive(root);
        tree.accept("non-existent", visit -> {
            assertThat(visit).isNull();
        });
    }

    @Test
    public void acceptAbsolutePath() throws Exception {
        final PathTree tree = PathTree.ofDirectoryOrArchive(root);
        try {
            tree.accept("/README.md", visit -> {
                fail("Absolute paths aren't allowed");
            });
        } catch (IllegalArgumentException e) {
            //expected
        }
    }

    @Test
    public void acceptAbsolutePathOutsideTree() throws Exception {
        final PathTree tree = PathTree.ofDirectoryOrArchive(root);
        final Path absolute = root.getParent().resolve("external.txt");
        assertThat(absolute).exists();
        try {
            tree.accept(absolute.toString(), visit -> {
                fail("Absolute paths aren't allowed");
            });
        } catch (IllegalArgumentException e) {
            //expected
        }
    }

    @Test
    public void walk() throws Exception {
        final PathTree tree = PathTree.ofDirectoryOrArchive(root);

        final Set<String> visited = new HashSet<>();
        final PathVisitor visitor = new PathVisitor() {
            @Override
            public void visitPath(PathVisit visit) {
                visited.add(visit.getRelativePath("/"));
            }
        };
        tree.walk(visitor);

        assertThat(visited).isEqualTo(Set.of(
                "",
                "README.md",
                "src",
                "src/main",
                "src/main/java",
                "src/main/java/Main.java"));
    }
}
