﻿// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

using System;
using System.Collections.Generic;
using System.Collections;
using System.Text;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Globalization;
using System.Linq;

using Microsoft.Build.Framework;
using Microsoft.Build.Shared;
using System.Runtime.InteropServices;

namespace Microsoft.Build.Utilities
{
    /// <summary>
    /// Enumeration to express the type of executable being wrapped by Tracker.exe
    /// </summary>
    public enum ExecutableType
    {
        /// <summary>
        /// 32-bit native executable
        /// </summary>
        Native32Bit = 0,

        /// <summary>
        /// 64-bit native executable 
        /// </summary>
        Native64Bit = 1,

        /// <summary>
        /// A managed executable without a specified bitness
        /// </summary>
        ManagedIL = 2,

        /// <summary>
        /// A managed executable specifically marked as 32-bit
        /// </summary>
        Managed32Bit = 3,

        /// <summary>
        /// A managed executable specifically marked as 64-bit
        /// </summary>
        Managed64Bit = 4,

        /// <summary>
        /// Use the same bitness as the currently running executable. 
        /// </summary>
        SameAsCurrentProcess = 5
    }

    /// <summary>
    /// This class contains utility functions to encapsulate launching and logging for the Tracker
    /// </summary>
    public static class FileTracker
    {
        #region Static Member Data
        // The default path to temp, used to create explicitly short and long paths
        private static string s_tempPath = Path.GetDirectoryName(Path.GetTempPath());

        // The short path to temp
        private static string s_tempShortPath = FileUtilities.EnsureTrailingSlash(NativeMethodsShared.GetShortFilePath(s_tempPath).ToUpperInvariant());

        // The long path to temp
        private static string s_tempLongPath = FileUtilities.EnsureTrailingSlash(NativeMethodsShared.GetLongFilePath(s_tempPath).ToUpperInvariant());

        // The path to ApplicationData (is equal to %USERPROFILE%\Application Data folder in Windows XP and %USERPROFILE%\AppData\Roaming in Vista and later)
        private static string s_applicationDataPath = FileUtilities.EnsureTrailingSlash(Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData).ToUpperInvariant());

        // The path to LocalApplicationData (is equal to %USERPROFILE%\Local Settings\Application Data folder in Windows XP and %USERPROFILE%\AppData\Local in Vista and later).
        private static string s_localApplicationDataPath = FileUtilities.EnsureTrailingSlash(Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData).ToUpperInvariant());

        // The path to the LocalLow folder. In Vista and later, user application data is organized across %USERPROFILE%\AppData\LocalLow,  %USERPROFILE%\AppData\Local (%LOCALAPPDATA%) 
        // and %USERPROFILE%\AppData\Roaming (%APPDATA%). The LocalLow folder is not present in XP.
        private static string s_localLowApplicationDataPath = FileUtilities.EnsureTrailingSlash(Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.UserProfile), "AppData\\LocalLow").ToUpperInvariant());

        // The path to the common Application Data, which is also used by some programs (e.g. antivirus) that we wish to ignore.
        // Is equal to C:\Documents and Settings\All Users\Application Data on XP, and C:\ProgramData on Vista+.
        // But for backward compatibility, the paths "C:\Documents and Settings\All Users\Application Data" and "C:\Users\All Users\Application Data" are still accessible via Junction point on Vista+.
        // Thus this list is created to store all possible common application data paths to cover more cases as possible.
        private static List<string> s_commonApplicationDataPaths;

        // The name of the standalone tracker tool.
        private static string s_TrackerFilename = "Tracker.exe";

        // The name of the assembly that is injected into the executing process
        private static string s_FileTrackerFilename = "FileTracker.dll";
        #endregion

        #region Static constructor

        static FileTracker()
        {
            s_commonApplicationDataPaths = new List<string>();

            string defaultCommonApplicationDataPath = FileUtilities.EnsureTrailingSlash(Environment.GetFolderPath(Environment.SpecialFolder.CommonApplicationData).ToUpperInvariant());
            s_commonApplicationDataPaths.Add(defaultCommonApplicationDataPath);

            string defaultRootDirectory = Path.GetPathRoot(defaultCommonApplicationDataPath);
            string alternativeCommonApplicationDataPath1 = FileUtilities.EnsureTrailingSlash(Path.Combine(defaultRootDirectory, @"Documents and Settings\All Users\Application Data").ToUpperInvariant());

            if (!alternativeCommonApplicationDataPath1.Equals(defaultCommonApplicationDataPath, StringComparison.Ordinal))
            {
                s_commonApplicationDataPaths.Add(alternativeCommonApplicationDataPath1);
            }

            string alternativeCommonApplicationDataPath2 = FileUtilities.EnsureTrailingSlash(Path.Combine(defaultRootDirectory, @"Users\All Users\Application Data").ToUpperInvariant());

            if (!alternativeCommonApplicationDataPath2.Equals(defaultCommonApplicationDataPath, StringComparison.Ordinal))
            {
                s_commonApplicationDataPaths.Add(alternativeCommonApplicationDataPath2);
            }
        }

        #endregion

        #region Native method wrappers

        /// <summary>
        /// Stops tracking file accesses.  
        /// </summary>
        public static void EndTrackingContext()
        {
            InprocTrackingNativeMethods.EndTrackingContext();
        }

        /// <summary>
        /// Resume tracking file accesses in the current tracking context. 
        /// </summary>
        public static void ResumeTracking()
        {
            InprocTrackingNativeMethods.ResumeTracking();
        }

        /// <summary>
        /// Set the global thread count, and assign that count to the current thread. 
        /// </summary>
        public static void SetThreadCount(int threadCount)
        {
            InprocTrackingNativeMethods.SetThreadCount(threadCount);
        }

        /// <summary>
        /// Starts tracking file accesses. 
        /// </summary>
        /// <param name="intermediateDirectory">The directory into which to write the tracking log files</param>
        /// <param name="taskName">The name of the task calling this function, used to determine the 
        /// names of the tracking log files</param>
        public static void StartTrackingContext(string intermediateDirectory, string taskName)
        {
            InprocTrackingNativeMethods.StartTrackingContext(intermediateDirectory, taskName);
        }

        /// <summary>
        /// Starts tracking file accesses, using the rooting marker in the response file provided.  To 
        /// automatically generate a response file given a rooting marker, call 
        /// FileTracker.CreateRootingMarkerResponseFile. 
        /// </summary>
        /// <param name="intermediateDirectory">The directory into which to write the tracking log files</param>
        /// <param name="taskName">The name of the task calling this function, used to determine the 
        /// names of the tracking log files</param>
        public static void StartTrackingContextWithRoot(string intermediateDirectory, string taskName, string rootMarkerResponseFile)
        {
            InprocTrackingNativeMethods.StartTrackingContextWithRoot(intermediateDirectory, taskName, rootMarkerResponseFile);
        }

        /// <summary>
        /// Stop tracking file accesses and get rid of the current tracking contexts. 
        /// </summary>
        public static void StopTrackingAndCleanup()
        {
            InprocTrackingNativeMethods.StopTrackingAndCleanup();
        }

        /// <summary>
        /// Temporarily suspend tracking of file accesses in the current tracking context. 
        /// </summary>
        public static void SuspendTracking()
        {
            InprocTrackingNativeMethods.SuspendTracking();
        }

        /// <summary>
        /// Write tracking logs for all contexts and threads. 
        /// </summary>
        /// <param name="intermediateDirectory">The directory into which to write the tracking log files</param>
        /// <param name="taskName">The name of the task calling this function, used to determine the 
        /// names of the tracking log files</param>
        [SuppressMessage("Microsoft.Naming", "CA1702:CompoundWordsShouldBeCasedCorrectly", MessageId = "TLogs", Justification = "Has now shipped as public API; plus it's unclear whether 'Tlog' or 'TLog' is the preferred casing")]
        public static void WriteAllTLogs(string intermediateDirectory, string taskName)
        {
            InprocTrackingNativeMethods.WriteAllTLogs(intermediateDirectory, taskName);
        }

        /// <summary>
        /// Write tracking logs corresponding to the current tracking context.  
        /// </summary>
        /// <param name="intermediateDirectory">The directory into which to write the tracking log files</param>
        /// <param name="taskName">The name of the task calling this function, used to determine the 
        /// names of the tracking log files</param>
        [SuppressMessage("Microsoft.Naming", "CA1702:CompoundWordsShouldBeCasedCorrectly", MessageId = "TLogs", Justification = "Has now shipped as public API; plus it's unclear whether 'Tlog' or 'TLog' is the preferred casing")]
        public static void WriteContextTLogs(string intermediateDirectory, string taskName)
        {
            InprocTrackingNativeMethods.WriteContextTLogs(intermediateDirectory, taskName);
        }

        #endregion // Native method wrappers

        #region Methods

        /// <summary>
        /// Test to see if the specified file is excluded from tracked dependencies
        /// </summary>
        /// <param name="fileName">
        /// Full path of the file to test
        /// </param>
        public static bool FileIsExcludedFromDependencies(string fileName)
        {
            bool exclude = true;

            // UNDONE: This check means that we cannot incremental build projects
            // that exist under the following directories on XP:
            // %USERPROFILE%\Application Data
            // %USERPROFILE%\Local Settings\Application Data
            //
            // and the following directories on Vista:
            // %USERPROFILE%\AppData\Local
            // %USERPROFILE%\AppData\LocalLow
            // %USERPROFILE%\AppData\Roaming

            // We don't want to be including these as dependencies or outputs:
            // 1. Files under %USERPROFILE%\Application Data in XP and %USERPROFILE%\AppData\Roaming in Vista and later.
            // 2. Files under %USERPROFILE%\Local Settings\Application Data in XP and %USERPROFILE%\AppData\Local in Vista and later.
            // 3. Files under %USERPROFILE%\AppData\LocalLow in Vista and later.
            // 4. Files that are in the TEMP directory (Since on XP, temp files are not
            //    located under AppData, they would not be compacted out correctly otherwise).
            // 5. Files under the common ("All Users") Application Data location -- C:\Documents and Settings\All Users\Application Data 
            //    on XP and either C:\Users\All Users\Application Data or C:\ProgramData on Vista+

            exclude = FileTracker.FileIsUnderPath(fileName, s_applicationDataPath) ||
                      FileTracker.FileIsUnderPath(fileName, s_localApplicationDataPath) ||
                      FileTracker.FileIsUnderPath(fileName, s_localLowApplicationDataPath) ||
                      FileTracker.FileIsUnderPath(fileName, s_tempShortPath) ||
                      FileTracker.FileIsUnderPath(fileName, s_tempLongPath) ||
                      s_commonApplicationDataPaths.Any(p => FileTracker.FileIsUnderPath(fileName, p));

            return exclude;
        }

        /// <summary>
        /// Test to see if the specified file is under the specified path
        /// </summary>
        /// <param name="fileName">
        /// Full path of the file to test
        /// </param>
        /// <param name="filePath">
        /// Is the file under this full path?
        /// </param>
        public static bool FileIsUnderPath(string fileName, string path)
        {
            // UNDONE: Get the long file path for the entry
            // This is an incredibly expensive operation. The tracking log
            // as written by CL etc. does not contain short paths
            // fileDirectory = NativeMethods.GetFullLongFilePath(fileDirectory);

            // Ensure that the path has a trailing slash that we are checking under
            // By default the paths that we check for most often will have, so this will
            // return fast and not allocate memory in the process
            path = FileUtilities.EnsureTrailingSlash(path);

            // Is the fileName under the filePath?
            return String.Compare(fileName, 0, path, 0, path.Length, StringComparison.OrdinalIgnoreCase) == 0;
        }

        /// <summary>
        /// Construct a rooting marker string from the ITaskItem array of primary sources.
        /// </summary>
        /// <param name="sources">
        /// ITaskItem array of primary sources.
        /// </param>
        public static string FormatRootingMarker(ITaskItem source)
        {
            return FormatRootingMarker(new ITaskItem[] { source }, null);
        }

        /// <summary>
        /// Construct a rooting marker string from the ITaskItem array of primary sources.
        /// </summary>
        /// <param name="sources">
        /// ITaskItem array of primary sources.
        /// </param>
        public static string FormatRootingMarker(ITaskItem source, ITaskItem output)
        {
            return FormatRootingMarker(new ITaskItem[] { source }, new ITaskItem[] { output });
        }

        /// <summary>
        /// Construct a rooting marker string from the ITaskItem array of primary sources.
        /// </summary>
        /// <param name="sources">
        /// ITaskItem array of primary sources.
        /// </param>
        public static string FormatRootingMarker(ITaskItem[] sources)
        {
            return FormatRootingMarker(sources, null);
        }

        /// <summary>
        /// Construct a rooting marker string from the ITaskItem array of primary sources.
        /// </summary>
        /// <param name="sources">
        /// ITaskItem array of primary sources.
        /// </param>
        public static string FormatRootingMarker(ITaskItem[] sources, ITaskItem[] outputs)
        {
            ErrorUtilities.VerifyThrowArgumentNull(sources, "sources");

            ArrayList rootSources = new ArrayList();
            StringBuilder rootSourcesList = new StringBuilder();
            int builderLength = 0;

            foreach (ITaskItem source in sources)
            {
                rootSources.Add(FileUtilities.NormalizePath(source.ItemSpec).ToUpperInvariant());
            }

            if (outputs != null)
            {
                foreach (ITaskItem output in outputs)
                {
                    rootSources.Add(FileUtilities.NormalizePath(output.ItemSpec).ToUpperInvariant());
                }
            }

            rootSources.Sort(StringComparer.OrdinalIgnoreCase);

            foreach (string source in rootSources)
            {
                rootSourcesList.Append(source);
                rootSourcesList.Append('|');
            }

            builderLength = rootSourcesList.Length - 1;
            if (builderLength < 0)
            {
                builderLength = 0;
            }

            return rootSourcesList.ToString(0, builderLength);
        }

        /// <summary>
        /// Given a set of source files in the form of ITaskItem, creates a temporary response
        /// file containing the rooting marker that corresponds to those sources. 
        /// </summary>
        /// <param name="rootMarker">The rooting marker to put in the response file.</param>
        /// <returns>The response file path.</returns>
        public static string CreateRootingMarkerResponseFile(ITaskItem[] sources)
        {
            return CreateRootingMarkerResponseFile(FormatRootingMarker(sources));
        }

        /// <summary>
        /// Given a rooting marker, creates a temporary response file with that rooting marker 
        /// in it.
        /// </summary>
        /// <param name="rootMarker">The rooting marker to put in the response file.</param>
        /// <returns>The response file path.</returns>
        public static string CreateRootingMarkerResponseFile(string rootMarker)
        {
            string trackerResponseFile = FileUtilities.GetTemporaryFile(".rsp");
            File.WriteAllText(trackerResponseFile, "/r \"" + rootMarker + "\"", Encoding.Unicode);

            return trackerResponseFile;
        }

        /// <summary>
        /// Prepends the path to the appropriate FileTracker assembly to the PATH
        /// environment variable.  Used for inproc tracking, or when the .NET Framework may 
        /// not be on the PATH.
        /// </summary>
        /// <returns>The old value of PATH</returns>
        public static string EnsureFileTrackerOnPath()
        {
            return EnsureFileTrackerOnPath(null);
        }

        /// <summary>
        /// Prepends the path to the appropriate FileTracker assembly to the PATH
        /// environment variable.  Used for inproc tracking, or when the .NET Framework may 
        /// not be on the PATH.
        /// </summary>
        /// <param name="rootPath">The root path for FileTracker.dll.  Overrides the toolType if specified.</param>
        /// <returns>The old value of PATH</returns>
        public static string EnsureFileTrackerOnPath(string rootPath)
        {
            string oldPath = Environment.GetEnvironmentVariable("PATH");
            string fileTrackerPath = GetFileTrackerPath(ExecutableType.SameAsCurrentProcess, rootPath);

            if (!String.IsNullOrEmpty(fileTrackerPath))
            {
                Environment.SetEnvironmentVariable
                            (
                                "Path",
                                Path.GetDirectoryName(fileTrackerPath) + ";" + oldPath
                            );
            }

            return oldPath;
        }

        /// <summary>
        /// Searches %PATH% for the location of Tracker.exe, and returns the first 
        /// path that matches. 
        /// <returns>Matching full path to Tracker.exe or null if a matching path is not found.</returns>
        /// </summary>
        public static string FindTrackerOnPath()
        {
            string[] paths = Environment.GetEnvironmentVariable("PATH").Split(new char[] { ';' }, StringSplitOptions.RemoveEmptyEntries);

            foreach (string path in paths)
            {
                string trackerPath;

                try
                {
                    if (!Path.IsPathRooted(path))
                    {
                        trackerPath = Path.GetFullPath(path);
                    }
                    else
                    {
                        trackerPath = path;
                    }

                    trackerPath = Path.Combine(trackerPath, s_TrackerFilename);

                    if (File.Exists(trackerPath))
                    {
                        return trackerPath;
                    }
                }
                catch (Exception e)
                {
                    if (ExceptionHandling.NotExpectedException(e))
                    {
                        throw;
                    }

                    // Otherwise, just ignore this path and move on -- it's just bad for some reason. 
                }
            }

            // Still haven't found it.
            return null;
        }

        /// <summary>
        /// Determines whether we must track out-of-proc, or whether inproc tracking will work. 
        /// </summary>
        /// <param name="toolType">The executable type for the tool being tracked</param>
        /// <returns>True if we need to track out-of-proc, false if inproc tracking is OK</returns>
        public static bool ForceOutOfProcTracking(ExecutableType toolType)
        {
            return ForceOutOfProcTracking(toolType, null, null);
        }

        /// <summary>
        /// Determines whether we must track out-of-proc, or whether inproc tracking will work. 
        /// </summary>
        /// <param name="toolType">The executable type for the tool being tracked</param>
        /// <param name="cancelEventName">The name of the cancel event tracker should listen for, or null if there isn't one</param>
        /// <returns>True if we need to track out-of-proc, false if inproc tracking is OK</returns>
        public static bool ForceOutOfProcTracking(ExecutableType toolType, string dllName, string cancelEventName)
        {
            bool trackOutOfProc = false;
            string trackerPath = null;

            if (cancelEventName != null)
            {
                // If we have a cancel event, we must track out-of-proc. 
                trackOutOfProc = true;
            }
            else if (dllName != null)
            {
                // If we have a DLL name, we need to track out of proc -- inproc tracking just uses
                // the default FileTracker.dll from the path. 
                trackOutOfProc = true;
            }
            else if (IntPtr.Size == sizeof(Int32))
            {
                // Current process is 32-bit.  So we need to spawn Tracker.exe if our tool is 
                // explicitly marked as 64-bit OR is MSIL and we're installed on a 64-bit OS.  
                if (toolType == ExecutableType.Managed64Bit || toolType == ExecutableType.Native64Bit)
                {
                    trackOutOfProc = true;
                }
                else if (toolType == ExecutableType.ManagedIL)
                {
                    trackerPath = ToolLocationHelper.GetPathToDotNetFrameworkFile(s_TrackerFilename, TargetDotNetFrameworkVersion.VersionLatest, DotNetFrameworkArchitecture.Bitness64);

                    if (trackerPath != null)
                    {
                        // If we found a 64-bit path, we're on a 64-bit OS and need to use it.  Otherwise, we're fine. 
                        trackOutOfProc = true;
                    }
                }
            }
            else if (IntPtr.Size == sizeof(Int64))
            {
                // Current process is 64-bit.  We need to spawn Tracker.exe if our tool is
                // explicitly marked as 32-bit.  
                if (toolType == ExecutableType.Managed32Bit || toolType == ExecutableType.Native32Bit)
                {
                    trackOutOfProc = true;
                }
            }

            return trackOutOfProc;
        }

        /// <summary>
        /// Given the ExecutableType of the tool being wrapped and information that we 
        /// know about our current bitness, figures out and returns the path to the correct
        /// Tracker.exe. 
        /// </summary>
        /// <param name="toolExecutableTypeToUse">The executable type of the tool being wrapped</param>
        public static string GetTrackerPath(ExecutableType toolType)
        {
            return GetTrackerPath(toolType, null);
        }

        /// <summary>
        /// Given the ExecutableType of the tool being wrapped and information that we 
        /// know about our current bitness, figures out and returns the path to the correct
        /// Tracker.exe. 
        /// </summary>
        /// <param name="toolExecutableTypeToUse">The executable type of the tool being wrapped</param>
        /// <param name="rootPath">The root path for Tracker.exe.  Overrides the toolType if specified.</param>
        public static string GetTrackerPath(ExecutableType toolType, string rootPath)
        {
            return GetPath(s_TrackerFilename, toolType, rootPath);
        }

        /// <summary>
        /// Given the ExecutableType of the tool being wrapped and information that we 
        /// know about our current bitness, figures out and returns the path to the correct
        /// FileTracker.dll. 
        /// </summary>
        /// <param name="toolExecutableTypeToUse">The executable type of the tool being wrapped</param>
        public static string GetFileTrackerPath(ExecutableType toolType)
        {
            return GetFileTrackerPath(toolType, null);
        }

        /// <summary>
        /// Given the ExecutableType of the tool being wrapped and information that we 
        /// know about our current bitness, figures out and returns the path to the correct
        /// FileTracker.dll. 
        /// </summary>
        /// <param name="toolExecutableTypeToUse">The executable type of the tool being wrapped</param>
        /// <param name="rootPath">The root path for FileTracker.dll.  Overrides the toolType if specified.</param>
        public static string GetFileTrackerPath(ExecutableType toolType, string rootPath)
        {
            return GetPath(s_FileTrackerFilename, toolType, rootPath);
        }

        /// <summary>
        /// Given a filename (only really meant to support either Tracker.exe or FileTracker.dll), returns
        /// the appropriate path for the appropriate file type. 
        /// </summary>
        /// <param name="filename"></param>
        /// <param name="toolType"></param>
        /// <param name="rootPath">The root path for the file.  Overrides the toolType if specified.</param>
        private static string GetPath(string filename, ExecutableType toolType, string rootPath)
        {
            string trackerPath = null;

            if (!String.IsNullOrEmpty(rootPath))
            {
                trackerPath = Path.Combine(rootPath, filename);

                if (!File.Exists(trackerPath))
                {
                    // if an override path was specified, that's it -- we don't want to fall back if the file
                    // is not found there.  
                    trackerPath = null;
                }
            }
            else
            {
                switch (toolType)
                {
                    case ExecutableType.Native32Bit:
                        // A native executable that's 32-bit.  Just return the 32-bit path 
                        trackerPath = GetPath(filename, DotNetFrameworkArchitecture.Bitness32);
                        break;
                    case ExecutableType.Native64Bit:
                        // A native executable that's 64-bit.  Just return the 64-bit path 
                        trackerPath = GetPath(filename, DotNetFrameworkArchitecture.Bitness64);
                        break;
                    case ExecutableType.ManagedIL:
                        // Next most likely -- the tool is a managed executable that has not been explicitly marked
                        // either 32 or 64 bit.  
                        // This case is slightly tricky -- we have to return the path to the 64-bit tracker if we're running 
                        // on a 64-bit machine, and the 32-bit tracker if we're running on a 32-bit machine.  
                        trackerPath = GetPath(filename, DotNetFrameworkArchitecture.Bitness64);

                        // If the path is null, that means there is no 64-bit framework -- that's fine, return the 32-bit path instead. 
                        if (trackerPath == null)
                        {
                            trackerPath = GetPath(filename, DotNetFrameworkArchitecture.Bitness32);
                        }
                        break;
                    case ExecutableType.Managed32Bit:
                        // A managed executable that has been explicitly marked 32-bit.  Just return the 32-bit path.  
                        trackerPath = GetPath(filename, DotNetFrameworkArchitecture.Bitness32);
                        break;
                    case ExecutableType.Managed64Bit:
                        // A managed executable that has been explicitly marked 64-bit.  Just return the 64-bit path.  If this 
                        // is a 32-bit machine, then we will return null, and the error will be caught and handled appropriately 
                        // elsewhere. 
                        trackerPath = GetPath(filename, DotNetFrameworkArchitecture.Bitness64);
                        break;
                    case ExecutableType.SameAsCurrentProcess:
                        // Figure out what bitness the current process is and return that bitness of Tracker.exe.  
                        if (IntPtr.Size == sizeof(Int32))
                        {
                            trackerPath = GetPath(filename, DotNetFrameworkArchitecture.Bitness32);
                        }
                        else
                        {
                            trackerPath = GetPath(filename, DotNetFrameworkArchitecture.Bitness64);
                        }
                        break;
                    default:
                        ErrorUtilities.ThrowInternalErrorUnreachable();
                        return null;
                }
            }

            return trackerPath;
        }

        /// <summary>
        /// Given a filename (currently only Tracker.exe and FileTracker.dll are supported), return 
        /// the path to that file. 
        /// </summary>
        /// <param name="filename"></param>
        /// <param name="bitness"></param>
        /// <returns></returns>
        private static string GetPath(string filename, DotNetFrameworkArchitecture bitness)
        {
            string trackerPath;

            // Make sure that if someone starts passing the wrong thing to this method we don't silently 
            // eat it and do something possibly unexpected. 
            ErrorUtilities.VerifyThrow(
                                       s_TrackerFilename.Equals(filename, StringComparison.OrdinalIgnoreCase) ||
                                       s_FileTrackerFilename.Equals(filename, StringComparison.OrdinalIgnoreCase),
                                       "This method should only be passed s_TrackerFilename or s_FileTrackerFilename, but was passed {0} instead!",
                                       filename
                                       );

            // Look for FileTracker.dll/FileTracker.exe first in the MSBuild tools directory, then fall back to .NET framework directory
            // and finally to .NET SDK directories.
            trackerPath = ToolLocationHelper.GetPathToBuildToolsFile(filename, ToolLocationHelper.CurrentToolsVersion, bitness);

            if (String.IsNullOrEmpty(trackerPath))
            {
                trackerPath = ToolLocationHelper.GetPathToDotNetFrameworkFile(filename, TargetDotNetFrameworkVersion.VersionLatest, bitness);

                if ((String.IsNullOrEmpty(trackerPath) || !File.Exists(trackerPath)) && s_TrackerFilename.Equals(filename, StringComparison.OrdinalIgnoreCase))
                {
                    // fall back to looking in the SDK directory -- this is where Tracker.exe will be in the typical VS case.  First check 
                    // in the SDK for the latest target framework and latest Visual Studio version, since we want to make sure that we get 
                    // the most up-to-date version of FileTracker.  
                    trackerPath = ToolLocationHelper.GetPathToDotNetFrameworkSdkFile(filename, TargetDotNetFrameworkVersion.Version45, VisualStudioVersion.VersionLatest, bitness);

                    // If that didn't work, we may be in a scenario where, e.g., we have VS 10 on Windows 8, which comes with .NET 4.5
                    // pre-installed.  In which case, the Dev11 (or other "latest" SDK) may not exist, but the Dev10 SDK might.  Check 
                    // for that here. 
                    if (trackerPath == null)
                    {
                        trackerPath = ToolLocationHelper.GetPathToDotNetFrameworkSdkFile(filename, TargetDotNetFrameworkVersion.Version40, bitness);
                    }
                }
            }
            return trackerPath;
        }

        /// <summary>
        /// This method constructs the correct Tracker.exe response file arguments from its parameters
        /// </summary>
        /// <param name="dllName">The name of the dll that will do the tracking</param>
        /// <param name="intermediateDirectory">Intermediate directory where tracking logs will be written</param>
        /// <param name="rootFiles">Rooting marker</param>
        /// <returns>The arguments as a string</returns>
        public static string TrackerResponseFileArguments(string dllName, string intermediateDirectory, string rootFiles)
        {
            return TrackerResponseFileArguments(dllName, intermediateDirectory, rootFiles, null);
        }

        /// <summary>
        /// This method constructs the correct Tracker.exe response file arguments from its parameters
        /// </summary>
        /// <param name="dllName">The name of the dll that will do the tracking</param>
        /// <param name="intermediateDirectory">Intermediate directory where tracking logs will be written</param>
        /// <param name="rootFiles">Rooting marker</param>
        /// <param name="cancelEventName">If a cancel event has been created that Tracker should be listening for, its name is passed here</param>
        /// <returns>The arguments as a string</returns>
        public static string TrackerResponseFileArguments(string dllName, string intermediateDirectory, string rootFiles, string cancelEventName)
        {
            CommandLineBuilder builder = new CommandLineBuilder();

            builder.AppendSwitchIfNotNull("/d ", dllName);

            if (!String.IsNullOrEmpty(intermediateDirectory))
            {
                intermediateDirectory = FileUtilities.NormalizePath(intermediateDirectory);
                // If the intermediate directory ends up with a trailing slash, then be rid of it!
                if (FileUtilities.EndsWithSlash(intermediateDirectory))
                {
                    intermediateDirectory = Path.GetDirectoryName(intermediateDirectory);
                }
                builder.AppendSwitchIfNotNull("/i ", intermediateDirectory);
            }

            builder.AppendSwitchIfNotNull("/r ", rootFiles);

            builder.AppendSwitchIfNotNull("/b ", cancelEventName); // b for break

            return builder.ToString() + " ";
        }

        /// <summary>
        /// This method constructs the correct Tracker.exe command arguments from its parameters
        /// </summary>
        /// <param name="command">The command to track</param>
        /// <param name="arguments">The command to track's arguments</param>
        /// <returns>The arguments as a string</returns>
        public static string TrackerCommandArguments(string command, string arguments)
        {
            CommandLineBuilder builder = new CommandLineBuilder();

            builder.AppendSwitch(" /c");
            builder.AppendFileNameIfNotNull(command);

            string fullArguments = builder.ToString();

            fullArguments += " " + arguments;

            return fullArguments;
        }

        /// <summary>
        /// This method constructs the correct Tracker.exe arguments from its parameters
        /// </summary>
        /// <param name="command">The command to track</param>
        /// <param name="arguments">The command to track's arguments</param>
        /// <param name="dllName">The name of the dll that will do the tracking</param>
        /// <param name="intermediateDirectory">Intermediate directory where tracking logs will be written</param>
        /// <param name="rootFiles">Rooting marker</param>
        /// <returns>The arguments as a string</returns>
        public static string TrackerArguments(string command, string arguments, string dllName, string intermediateDirectory, string rootFiles)
        {
            return TrackerArguments(command, arguments, dllName, intermediateDirectory, rootFiles, null);
        }

        /// <summary>
        /// This method constructs the correct Tracker.exe arguments from its parameters
        /// </summary>
        /// <param name="command">The command to track</param>
        /// <param name="arguments">The command to track's arguments</param>
        /// <param name="dllName">The name of the dll that will do the tracking</param>
        /// <param name="intermediateDirectory">Intermediate directory where tracking logs will be written</param>
        /// <param name="rootFiles">Rooting marker</param>
        /// <param name="cancelEventName">If a cancel event has been created that Tracker should be listening for, its name is passed here</param>
        /// <returns>The arguments as a string</returns>
        public static string TrackerArguments(string command, string arguments, string dllName, string intermediateDirectory, string rootFiles, string cancelEventName)
        {
            string fullArguments = TrackerResponseFileArguments(dllName, intermediateDirectory, rootFiles, cancelEventName);

            fullArguments += TrackerCommandArguments(command, arguments);

            return fullArguments;
        }

        #region StartProcess methods

        /// <summary>
        /// Start the process; tracking the command.  
        /// </summary>
        /// <param name="command">The command to track</param>
        /// <param name="arguments">The command to track's arguments</param>
        /// <param name="toolType">The type of executable the wrapped tool is</param>
        /// <param name="dllName">The name of the dll that will do the tracking</param>
        /// <param name="intermediateDirectory">Intermediate directory where tracking logs will be written</param>
        /// <param name="rootFiles">Rooting marker</param>
        /// <param name="cancelEventName">If Tracker should be listening on a particular event for cancellation, pass its name here</param>
        /// <returns>Process instance</returns>
        public static Process StartProcess(string command, string arguments, ExecutableType toolType, string dllName, string intermediateDirectory, string rootFiles, string cancelEventName)
        {
            dllName = dllName ?? GetFileTrackerPath(toolType);

            string fullArguments = TrackerArguments(command, arguments, dllName, intermediateDirectory, rootFiles, cancelEventName);
            return Process.Start(GetTrackerPath(toolType), fullArguments);
        }

        /// <summary>
        /// Start the process; tracking the command.
        /// </summary>
        /// <param name="command">The command to track</param>
        /// <param name="arguments">The command to track's arguments</param>
        /// <param name="toolType">The type of executable the wrapped tool is</param>
        /// <param name="dllName">The name of the dll that will do the tracking</param>
        /// <param name="intermediateDirectory">Intermediate directory where tracking logs will be written</param>
        /// <param name="rootFiles">Rooting marker</param>
        /// <returns>Process instance</returns>
        public static Process StartProcess(string command, string arguments, ExecutableType toolType, string dllName, string intermediateDirectory, string rootFiles)
        {
            return StartProcess(command, arguments, toolType, dllName, intermediateDirectory, rootFiles, null);
        }

        /// <summary>
        /// Start the process; tracking the command.
        /// </summary>
        /// <param name="command">The command to track</param>
        /// <param name="arguments">The command to track's arguments</param>
        /// <param name="toolType">The type of executable the wrapped tool is</param>
        /// <param name="intermediateDirectory">Intermediate directory where tracking logs will be written</param>
        /// <param name="rootFiles">Rooting marker</param>
        /// <returns>Process instance</returns>
        public static Process StartProcess(string command, string arguments, ExecutableType toolType, string intermediateDirectory, string rootFiles)
        {
            return StartProcess(command, arguments, toolType, null, intermediateDirectory, rootFiles, null);
        }

        /// <summary>
        /// Start the process; tracking the command.
        /// </summary>
        /// <param name="command">The command to track</param>
        /// <param name="arguments">The command to track's arguments</param>
        /// <param name="toolType">The type of executable the wrapped tool is</param>
        /// <param name="rootFiles">Rooting marker</param>
        /// <returns>Process instance</returns>
        public static Process StartProcess(string command, string arguments, ExecutableType toolType, string rootFiles)
        {
            return StartProcess(command, arguments, toolType, null, null, rootFiles, null);
        }

        /// <summary>
        /// Start the process; tracking the command.
        /// </summary>
        /// <param name="command">The command to track</param>
        /// <param name="arguments">The command to track's arguments</param>
        /// <param name="toolType">The type of executable the wrapped tool is</param>
        /// <returns>Process instance</returns>
        public static Process StartProcess(string command, string arguments, ExecutableType toolType)
        {
            return StartProcess(command, arguments, toolType, null, null, null, null);
        }

        #endregion // StartProcess methods

        /// <summary>
        /// Logs a message of the given importance using the specified resource string. To the specified Log.
        /// </summary>
        /// <remarks>This method is not thread-safe.</remarks>
        /// <param name="Log">The Log to log to.</param>
        /// <param name="importance">The importance level of the message.</param>
        /// <param name="messageResourceName">The name of the string resource to load.</param>
        /// <param name="messageArgs">Optional arguments for formatting the loaded string.</param>
        /// <exception cref="ArgumentNullException">Thrown when <c>messageResourceName</c> is null.</exception>
        internal static void LogMessageFromResources(TaskLoggingHelper Log, MessageImportance importance, string messageResourceName, params object[] messageArgs)
        {
            // Only log when we have been passed a TaskLoggingHelper
            if (Log != null)
            {
                ErrorUtilities.VerifyThrowArgumentNull(messageResourceName, "messageResourceName");

                Log.LogMessage(importance, AssemblyResources.FormatResourceString(messageResourceName, messageArgs));
            }
        }

        /// <summary>
        /// Logs a message of the given importance using the specified string.
        /// </summary>
        /// <remarks>This method is not thread-safe.</remarks>
        /// <param name="importance">The importance level of the message.</param>
        /// <param name="message">The message string.</param>
        /// <param name="messageArgs">Optional arguments for formatting the message string.</param>
        /// <exception cref="ArgumentNullException">Thrown when <c>message</c> is null.</exception>
        internal static void LogMessage(TaskLoggingHelper Log, MessageImportance importance, string message, params object[] messageArgs)
        {
            // Only log when we have been passed a TaskLoggingHelper
            if (Log != null)
            {
                Log.LogMessage(importance, message, messageArgs);
            }
        }

        /// <summary>
        /// Logs a warning using the specified resource string.
        /// </summary>
        /// <param name="messageResourceName">The name of the string resource to load.</param>
        /// <param name="messageArgs">Optional arguments for formatting the loaded string.</param>
        /// <exception cref="ArgumentNullException">Thrown when <c>messageResourceName</c> is null.</exception>
        internal static void LogWarningWithCodeFromResources(TaskLoggingHelper Log, string messageResourceName, params object[] messageArgs)
        {
            // Only log when we have been passed a TaskLoggingHelper
            if (Log != null)
            {
                Log.LogWarningWithCodeFromResources(messageResourceName, messageArgs);
            }
        }

        #endregion
    }

    /// <summary>
    /// Dependency filter delegate. Used during TLog saves in order for tasks to selectively remove dependencies from the written
    /// graph.
    /// </summary>
    /// <param name="fullPath">The full path to the dependency file about to be written to the compacted TLog</param>
    /// <returns>If the file should actually be written to the TLog (true) or not (false)</returns>
    public delegate bool DependencyFilter(string fullPath);
}
