// SPDX-License-Identifier: GPL-3.0-or-later

package de.sesu8642.feudaltactics.frontend.ui.screens;

import javax.inject.Inject;

import com.google.common.eventbus.Subscribe;

import de.sesu8642.feudaltactics.events.GameResumedEvent;
import de.sesu8642.feudaltactics.events.GameStateChangeEvent;
import de.sesu8642.feudaltactics.events.moves.GameStartEvent;
import de.sesu8642.feudaltactics.frontend.events.CloseMenuEvent;
import de.sesu8642.feudaltactics.frontend.events.EndTurnUnconfirmedEvent;
import de.sesu8642.feudaltactics.frontend.events.ExitGameEvent;
import de.sesu8642.feudaltactics.frontend.events.OpenMenuEvent;
import de.sesu8642.feudaltactics.frontend.events.RetryGameUnconfirmedEvent;
import de.sesu8642.feudaltactics.frontend.events.input.EscInputEvent;
import de.sesu8642.feudaltactics.frontend.ui.screens.IngameScreen.IngameStages;

/** Handles events for the ingame screen. **/
public class IngameScreenEventHandler {

	private IngameScreen ingameScreen;

	/**
	 * Constructor.
	 * 
	 * @param ingameScreen ingame screen
	 */
	@Inject
	public IngameScreenEventHandler(IngameScreen ingameScreen) {
		this.ingameScreen = ingameScreen;
	}

	/**
	 * Event handler for ESC key events.
	 * 
	 * @param event event to handle
	 */
	@Subscribe
	public void handleEscInput(EscInputEvent event) {
		ingameScreen.togglePause();
	}

	/**
	 * Event handler for end turn attempt events.
	 * 
	 * @param event event to handle
	 */
	@Subscribe
	public void handleEndTurnAttempt(EndTurnUnconfirmedEvent event) {
		ingameScreen.handleEndTurnAttempt();
	}

	/**
	 * Event handler for open menu events.
	 * 
	 * @param event event to handle
	 */
	@Subscribe
	public void handleOpenMenuAttempt(OpenMenuEvent event) {
		ingameScreen.activateStage(IngameStages.MENU);
	}

	/**
	 * Event handler for close menu events.
	 * 
	 * @param event event to handle
	 */
	@Subscribe
	public void handleCloseMenuAttempt(CloseMenuEvent event) {
		ingameScreen.activateStage(IngameStages.HUD);
	}

	/**
	 * Event handler for game start events.
	 * 
	 * @param event event to handle
	 */
	@Subscribe
	public void handleGameStart(GameStartEvent event) {
		ingameScreen.activateStage(IngameStages.HUD);
	}

	/**
	 * Event handler for game resumed events.
	 * 
	 * @param event event to handle
	 */
	@Subscribe
	public void handleGameResumed(GameResumedEvent event) {
		ingameScreen.activateStage(IngameStages.HUD);
	}

	/**
	 * Event handler for unconfirmed retry game events.
	 * 
	 * @param event event to handle
	 */
	@Subscribe
	public void handleUnconfirmedRetryGame(RetryGameUnconfirmedEvent event) {
		ingameScreen.handleUnconfirmedRetryGame();
	}

	/**
	 * Event handler for exit game events.
	 * 
	 * @param event event to handle
	 */
	@Subscribe
	public void handleExitGameAttempt(ExitGameEvent event) {
		ingameScreen.handleExitGameAttempt();
	}

	/**
	 * Event handler for gameState changes.
	 * 
	 * @param event event to handle
	 */
	@Subscribe
	public void handleGameStateChange(GameStateChangeEvent event) {
		ingameScreen.handleGameStateChange(event.getGameState(), event.isMapDimensionsChanged());
	}

}
