namespace Orleans.Runtime
{
    using System;
    using System.Collections.Generic;
    using System.Reflection;
    using System.Linq;

    using Orleans.CodeGeneration;
    using Orleans.Serialization;

    /// <summary>
    /// The assembly processor.
    /// </summary>
    internal class AssemblyProcessor
    {
        /// <summary>
        /// The collection of assemblies which have already been processed.
        /// </summary>
        private static readonly HashSet<Assembly> ProcessedAssemblies = new HashSet<Assembly>();

        /// <summary>
        /// The logger.
        /// </summary>
        private static readonly TraceLogger Logger;
        
        /// <summary>
        /// The initialization lock.
        /// </summary>
        private static readonly object InitializationLock = new object();

        /// <summary>
        /// Whether or not this class has been initialized.
        /// </summary>
        private static bool initialized;

        /// <summary>
        /// Initializes static members of the <see cref="AssemblyProcessor"/> class.
        /// </summary>
        static AssemblyProcessor()
        {
            Logger = TraceLogger.GetLogger("AssemblyProcessor");
        }

        /// <summary>
        /// Initializes this instance.
        /// </summary>
        public static void Initialize()
        {
            if (initialized)
            {
                return;
            }

            lock (InitializationLock)
            {
                if (initialized)
                {
                    return;
                }

                // load the code generator before intercepting assembly loading
                CodeGeneratorManager.Initialize(); 

                // initialize serialization for all assemblies to be loaded.
                AppDomain.CurrentDomain.AssemblyLoad += OnAssemblyLoad;

                Assembly[] assemblies = AppDomain.CurrentDomain.GetAssemblies();

                // initialize serialization for already loaded assemblies.
                CodeGeneratorManager.GenerateAndCacheCodeForAllAssemblies();
                foreach (var assembly in assemblies)
                {
                    ProcessAssembly(assembly);
                }

                initialized = true;
            }
        }

        /// <summary>
        /// Handles <see cref="AppDomain.AssemblyLoad"/> events.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="args">The event arguments.</param>
        private static void OnAssemblyLoad(object sender, AssemblyLoadEventArgs args)
        {
            ProcessAssembly(args.LoadedAssembly);
        }

        /// <summary>
        /// Processes the provided assembly.
        /// </summary>
        /// <param name="assembly">The assembly to process.</param>
        private static void ProcessAssembly(Assembly assembly)
        {
            string assemblyName = assembly.GetName().Name;
            if (Logger.IsVerbose3)
            {
                Logger.Verbose3("Processing assembly {0}", assemblyName);
            }
            // If the assembly is loaded for reflection only avoid processing it.
            if (assembly.ReflectionOnly)
            {
                return;
            }

            // Don't bother re-processing an assembly we've already scanned
            lock (ProcessedAssemblies)
            {
                if (!ProcessedAssemblies.Add(assembly))
                {
                    return;
                }
            }

            // If the assembly does not reference Orleans, avoid generating code for it.
            if (TypeUtils.IsOrleansOrReferencesOrleans(assembly))
            {
                // Code generation occurs in a self-contained assembly, so invoke it separately.
                CodeGeneratorManager.GenerateAndCacheCodeForAssembly(assembly);
            }

            // Process each type in the assembly.
            var shouldProcessSerialization = SerializationManager.ShouldFindSerializationInfo(assembly);
            var assemblyTypes = TypeUtils.GetDefinedTypes(assembly, Logger).ToArray();

            // Process each type in the assembly.
            foreach (TypeInfo type in assemblyTypes)
            {
                try
                {
                    string typeName = type.FullName;
                    if (Logger.IsVerbose3)
                    {
                        Logger.Verbose3("Processing type {0}", typeName);
                    }
                    if (shouldProcessSerialization)
                    {
                        SerializationManager.FindSerializationInfo(type);
                    }
    
                    GrainFactory.FindSupportClasses(type);
                }
                catch (Exception exception)
                {
                    Logger.Error(ErrorCode.SerMgr_TypeRegistrationFailure, "Failed to load type " + type.FullName + " in assembly " + assembly.FullName + ".", exception);
                }
            }
        }
    }
}
