package io.quarkus.smallrye.jwt.runtime;

import org.wildfly.security.auth.realm.token.TokenSecurityRealm;
import org.wildfly.security.auth.server.SecurityDomain;
import org.wildfly.security.auth.server.SecurityRealm;

import io.quarkus.arc.runtime.BeanContainer;
import io.quarkus.runtime.RuntimeValue;
import io.quarkus.runtime.annotations.Template;
import io.quarkus.smallrye.jwt.runtime.auth.ElytronJwtCallerPrincipal;
import io.quarkus.smallrye.jwt.runtime.auth.JWTAuthMethodExtension;
import io.quarkus.smallrye.jwt.runtime.auth.JwtIdentityManager;
import io.quarkus.smallrye.jwt.runtime.auth.MpJwtValidator;
import io.undertow.security.idm.IdentityManager;
import io.undertow.servlet.ServletExtension;

/**
 * The runtime value service used to create values related to the MP-JWT services
 */
@Template
public class SmallRyeJwtTemplate {

    /**
     * Create the JwtIdentityManager
     * 
     * @param securityDomain - the SecurityDomain to use for auth decisions
     * @return - the IdentityManager instance to register
     */
    public IdentityManager createIdentityManager(RuntimeValue<SecurityDomain> securityDomain) {
        return new JwtIdentityManager(securityDomain.getValue());
    }

    /**
     * Create the JWTAuthMethodExtension servlet extension
     * 
     * @param authMechanism - name to use for MP-JWT auth mechanism
     * @param container - bean container to create JWTAuthMethodExtension bean
     * @return JWTAuthMethodExtension
     */
    public ServletExtension createAuthExtension(String authMechanism, BeanContainer container) {
        JWTAuthMethodExtension authExt = container.instance(JWTAuthMethodExtension.class);
        authExt.setAuthMechanism(authMechanism);
        return authExt;
    }

    /**
     * Create the TokenSecurityRealm
     * 
     * @return runtime wrapped TokenSecurityRealm
     */
    public RuntimeValue<SecurityRealm> createTokenRealm(BeanContainer container) {
        MpJwtValidator jwtValidator = container.instance(MpJwtValidator.class);
        TokenSecurityRealm tokenRealm = TokenSecurityRealm.builder()
                .claimToPrincipal(claims -> new ElytronJwtCallerPrincipal(claims))
                .validator(jwtValidator)
                .build();
        return new RuntimeValue<>(tokenRealm);
    }
}
