/* Copyright 2024 Marimo. All rights reserved. */
import { describe, expect, it } from "vitest";
import { fileSplit, getProtocolAndParentDirectories } from "./pathUtils";

describe("getProtocolAndParentDirectories", () => {
  it("should extract protocol and list parent directories correctly", () => {
    const path = "http://example.com/folder/subfolder/";
    const delimiter = "/";
    const initialPath = "http://example.com/folder";
    const restrictNavigation = true;

    const { protocol, parentDirectories } = getProtocolAndParentDirectories(
      path,
      delimiter,
      initialPath,
      restrictNavigation,
    );

    expect(protocol).toBe("http://");
    expect(parentDirectories).toEqual([
      "http://example.com/folder/subfolder",
      "http://example.com/folder",
    ]);
  });

  it("should handle Google Cloud Storage paths", () => {
    const path = "gs://bucket/folder/subfolder/";
    const delimiter = "/";
    const initialPath = "gs://bucket/folder";
    const restrictNavigation = true;

    const { protocol, parentDirectories } = getProtocolAndParentDirectories(
      path,
      delimiter,
      initialPath,
      restrictNavigation,
    );

    expect(protocol).toBe("gs://");
    expect(parentDirectories).toEqual([
      "gs://bucket/folder/subfolder",
      "gs://bucket/folder",
    ]);
  });

  it("should handle S3 paths", () => {
    const path = "s3://bucket/folder/subfolder/";
    const delimiter = "/";
    const initialPath = "s3://bucket/folder";
    const restrictNavigation = true;

    const { protocol, parentDirectories } = getProtocolAndParentDirectories(
      path,
      delimiter,
      initialPath,
      restrictNavigation,
    );

    expect(protocol).toBe("s3://");
    expect(parentDirectories).toEqual([
      "s3://bucket/folder/subfolder",
      "s3://bucket/folder",
    ]);
  });

  it("should handle paths without protocol", () => {
    const path = "/folder/subfolder/";
    const delimiter = "/";
    const initialPath = "/folder";
    const restrictNavigation = false;

    const { protocol, parentDirectories } = getProtocolAndParentDirectories(
      path,
      delimiter,
      initialPath,
      restrictNavigation,
    );

    expect(protocol).toBe("/");
    expect(parentDirectories).toEqual(["/folder/subfolder", "/folder", "/"]);
  });
  it("should handle Windows paths", () => {
    const path = "C:\\folder\\subfolder\\";
    const delimiter = "\\";
    const initialPath = "C:\\folder";
    const restrictNavigation = false;

    const { protocol, parentDirectories } = getProtocolAndParentDirectories(
      path,
      delimiter,
      initialPath,
      restrictNavigation,
    );

    expect(protocol).toBe("C:\\");
    expect(parentDirectories).toEqual([
      "C:\\folder\\subfolder",
      "C:\\folder",
      "C:\\",
    ]);
  });
});

describe("fileSplit", () => {
  it("should split a path into a name and extension", () => {
    expect(fileSplit("path/to/file.txt")).toEqual(["path/to/file", ".txt"]);
    expect(fileSplit("path/to/file")).toEqual(["path/to/file", ""]);
    expect(fileSplit("path/to/file.txt.md")).toEqual([
      "path/to/file.txt",
      ".md",
    ]);
  });
});
