/*
Copyright 2016 The Rook Authors. All rights reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

	http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Some of the code below came from https://github.com/digitalocean/ceph_exporter
which has the same license.
*/
package client

import (
	"encoding/json"
	"fmt"

	"github.com/rook/rook/pkg/clusterd"
	"github.com/rook/rook/pkg/model"
)

const (
	// CephHealthOK denotes the status of ceph cluster when healthy.
	CephHealthOK = "HEALTH_OK"

	// CephHealthWarn denotes the status of ceph cluster when unhealthy but recovering.
	CephHealthWarn = "HEALTH_WARN"

	// CephHealthErr denotes the status of ceph cluster when unhealthy but usually needs
	// manual intervention.
	CephHealthErr = "HEALTH_ERR"
)

type CephStatus struct {
	Health        HealthStatus `json:"health"`
	FSID          string       `json:"fsid"`
	ElectionEpoch int          `json:"election_epoch"`
	Quorum        []int        `json:"quorum"`
	QuorumNames   []string     `json:"quorum_names"`
	MonMap        MonMap       `json:"monmap"`
	OsdMap        struct {
		OsdMap OsdMap `json:"osdmap"`
	} `json:"osdmap"`
	PgMap  PgMap  `json:"pgmap"`
	MgrMap MgrMap `json:"mgrmap"`
}

type HealthStatus struct {
	Status string                  `json:"status"`
	Checks map[string]CheckMessage `json:"checks"`
}

type CheckMessage struct {
	Severity string `json:"severity"`
	Message  string `json:"message"`
}

type MonMap struct {
	Epoch        int           `json:"epoch"`
	FSID         string        `json:"fsid"`
	CreatedTime  string        `json:"created"`
	ModifiedTime string        `json:"modified"`
	Mons         []MonMapEntry `json:"mons"`
}

type MgrMap struct {
	Epoch      int          `json:"epoch"`
	ActiveGID  int          `json:"active_gid"`
	ActiveName string       `json:"active_name"`
	ActiveAddr string       `json:"active_addr"`
	Available  bool         `json:"available"`
	Standbys   []MgrStandby `json:"standbys"`
}

type MgrStandby struct {
	GID  int    `json:"gid"`
	Name string `json:"name"`
}

type OsdMap struct {
	Epoch          int  `json:"epoch"`
	NumOsd         int  `json:"num_osds"`
	NumUpOsd       int  `json:"num_up_osds"`
	NumInOsd       int  `json:"num_in_osds"`
	Full           bool `json:"full"`
	NearFull       bool `json:"nearfull"`
	NumRemappedPgs int  `json:"num_remapped_pgs"`
}

type PgMap struct {
	PgsByState     []PgStateEntry `json:"pgs_by_state"`
	Version        int            `json:"version"`
	NumPgs         int            `json:"num_pgs"`
	DataBytes      uint64         `json:"data_bytes"`
	UsedBytes      uint64         `json:"bytes_used"`
	AvailableBytes uint64         `json:"bytes_avail"`
	TotalBytes     uint64         `json:"bytes_total"`
}

type PgStateEntry struct {
	StateName string `json:"state_name"`
	Count     int    `json:"count"`
}

func Status(context *clusterd.Context, clusterName string) (CephStatus, error) {
	args := []string{"status"}
	buf, err := ExecuteCephCommand(context, clusterName, args)
	if err != nil {
		return CephStatus{}, fmt.Errorf("failed to get status: %+v", err)
	}

	var status CephStatus
	if err := json.Unmarshal(buf, &status); err != nil {
		return CephStatus{}, fmt.Errorf("failed to unmarshal status response: %+v", err)
	}

	return status, nil
}

func StatusPlain(context *clusterd.Context, clusterName string) ([]byte, error) {
	args := []string{"status"}
	buf, err := ExecuteCephCommandPlain(context, clusterName, args)
	if err != nil {
		return nil, fmt.Errorf("failed to get status: %+v", err)
	}

	return buf, nil
}

func HealthToModelHealthStatus(cephHealth string) model.HealthStatus {
	switch cephHealth {
	case CephHealthOK:
		return model.HealthOK
	case CephHealthWarn:
		return model.HealthWarning
	case CephHealthErr:
		return model.HealthError
	default:
		return model.HealthUnknown
	}
}
