import includes from "lodash/collection/includes";
import { explode } from "../visitors";
import traverse from "../index";
import defaults from "lodash/object/defaults";
import * as messages from "../../messages";
import Binding from "./binding";
import globals from "globals";
import flatten from "lodash/array/flatten";
import extend from "lodash/object/extend";
import object from "../../helpers/object";
import each from "lodash/collection/each";
import * as t from "../../types";

var functionVariableVisitor = {
  enter(node, parent, scope, state) {
    if (t.isFor(node)) {
      for (var key of (t.FOR_INIT_KEYS: Array)) {
        var declar = this.get(key);
        if (declar.isVar()) state.scope.registerBinding("var", declar);
      }
    }

    // this block is a function so we'll stop since none of the variables
    // declared within are accessible
    if (this.isFunction()) return this.skip();

    // function identifier doesn't belong to this scope
    if (state.blockId && node === state.blockId) return;

    // delegate block scope handling to the `blockVariableVisitor`
    if (this.isBlockScoped()) return;

    // this will be hit again once we traverse into it after this iteration
    if (this.isExportDeclaration() && t.isDeclaration(node.declaration)) return;

    // we've ran into a declaration!
    if (this.isDeclaration()) state.scope.registerDeclaration(this);
  }
};

var programReferenceVisitor = explode({
  ReferencedIdentifier(node, parent, scope, state) {
    var bindingInfo = scope.getBinding(node.name);
    if (bindingInfo) {
      bindingInfo.reference();
    } else {
      state.addGlobal(node);
    }
  },

  Scopable(node, parent, scope, state) {
    for (var name in scope.bindings) {
      state.references[name] = true;
    }
  },

  ExportDeclaration: {
    exit(node, parent, scope, state) {
      var declar = node.declaration;
      if (t.isClassDeclaration(declar) || t.isFunctionDeclaration(declar)) {
        scope.getBinding(declar.id.name).reference();
      } else if (t.isVariableDeclaration(declar)) {
        for (var decl of (declar.declarations: Array)) {
          var ids = t.getBindingIdentifiers(decl);
          for (var name in ids) {
            scope.getBinding(name).reference();
          }
        }
      }
    }
  },

  LabeledStatement(node, parent, scope, state) {
    state.addGlobal(node);
  },

  AssignmentExpression(node, parent, scope, state) {
    scope.registerConstantViolation(this.get("left"), this.get("right"));
  },

  UpdateExpression(node, parent, scope, state) {
    scope.registerConstantViolation(this.get("argument"), null);
  },

  UnaryExpression(node, parent, scope, state) {
    if (node.operator === "delete") scope.registerConstantViolation(this.get("left"), null);
  }
});

var blockVariableVisitor = explode({
  Scope() {
    this.skip();
  },

  enter(node, parent, scope, state) {
    if (this.isFunctionDeclaration() || this.isBlockScoped()) {
      state.registerDeclaration(this);
    }
  }
});

var renameVisitor = explode({
  ReferencedIdentifier(node, parent, scope, state) {
    if (node.name === state.oldName) {
      node.name = state.newName;
    }
  },

  Declaration(node, parent, scope, state) {
    var ids = this.getBindingIdentifiers();;

    for (var name in ids) {
      if (name === state.oldName) ids[name].name = state.newName;
    }
  },

  Scopable(node, parent, scope, state) {
    if (this.isScope()) {
      if (!scope.bindingIdentifierEquals(state.oldName, state.binding)) {
        this.skip();
      }
    }
  }
});

export default class Scope {

  /**
   * This searches the current "scope" and collects all references/bindings
   * within.
   */

  constructor(path: TraversalPath, parent?: Scope, file?: File) {
    if (parent && parent.block === path.node) {
      return parent;
    }

    var cached = path.getData("scope");
    if (cached && cached.parent === parent) {
      return cached;
    } else {
      //path.setData("scope", this);
    }

    this.parent = parent;
    this.file   = parent ? parent.file : file;

    this.parentBlock = path.parent;
    this.block       = path.node;
    this.path        = path;

    this.crawl();
  }

  static globals = flatten([globals.builtin, globals.browser, globals.node].map(Object.keys));
  static contextVariables = ["this", "arguments", "super"];

  /**
   * Description
   */

  traverse(node: Object, opts: Object, state?) {
    traverse(node, opts, this, state, this.path);
  }


  /**
   * Since `Scope` instances are unique to their traversal we need some other
   * way to compare if scopes are the same. Here we just compare `this.bindings`
   * as it will be the same across all instances.
   */

  is(scope) {
    return this.bindings === scope.bindings;
  }

  /**
   * Description
   */

  generateDeclaredUidIdentifier(name: string = "temp") {
    var id = this.generateUidIdentifier(name);
    this.push({ id });
    return id;
  }

  /**
   * Description
   */

  generateUidIdentifier(name: string) {
    return t.identifier(this.generateUid(name));
  }

  /**
   * Description
   */

  generateUid(name: string) {
    name = t.toIdentifier(name).replace(/^_+/, "");

    var uid;
    var i = 0;
    do {
      uid = this._generateUid(name, i);
      i++;
    } while (this.hasBinding(uid) || this.hasGlobal(uid) || this.hasReference(uid));


    var program = this.getProgramParent();
    program.references[uid] = true;
    program.uids[uid] = true;

    return uid;
  }

  _generateUid(name, i) {
    var id = name;
    if (i > 1) id += i;
    return `_${id}`;
  }

  /*
   * Description
   */

  generateUidIdentifierBasedOnNode(parent: Object, defaultName?: String):  Object {
    var node = parent;

    if (t.isAssignmentExpression(parent)) {
      node = parent.left;
    } else if (t.isVariableDeclarator(parent)) {
      node = parent.id;
    } else if (t.isProperty(node)) {
      node = node.key;
    }

    var parts = [];

    var add = function (node) {
      if (t.isModuleDeclaration(node)) {
        if (node.source) {
          add(node.source);
        } else if (node.specifiers && node.specifiers.length) {
          for (var specifier of (node.specifiers: Array)) {
            add(specifier);
          }
        } else if (node.declaration) {
          add(node.declaration);
        }
      } else if (t.isModuleSpecifier(node)) {
        add(node.local);
      } else if (t.isMemberExpression(node)) {
        add(node.object);
        add(node.property);
      } else if (t.isIdentifier(node)) {
        parts.push(node.name);
      } else if (t.isLiteral(node)) {
        parts.push(node.value);
      } else if (t.isCallExpression(node)) {
        add(node.callee);
      } else if (t.isObjectExpression(node) || t.isObjectPattern(node)) {
        for (var prop of (node.properties: Array)) {
          add(prop.key || prop.argument);
        }
      }
    };

    add(node);

    var id = parts.join("$");
    id = id.replace(/^_/, "") || defaultName || "ref";

    return this.generateUidIdentifier(id);
  }

  /**
   * Determine whether evaluating the specific input `node` is a consequenceless reference. ie.
   * evaluating it wont result in potentially arbitrary code from being ran. The following are
   * whitelisted and determined not cause side effects:
   *
   *  - `this` expressions
   *  - `super` expressions
   *  - Bound identifiers
   */

  isStatic(node: Object): boolean {
    if (t.isThisExpression(node) || t.isSuper(node)) {
      return true;
    }

    if (t.isIdentifier(node) && this.hasBinding(node.name)) {
      return true;
    }

    return false;
  }

  /**
   * Description
   */

  maybeGenerateMemoised(node: Object, dontPush?: boolean): ?Object {
    if (this.isStatic(node)) {
      return null;
    } else {
      var id = this.generateUidIdentifierBasedOnNode(node);
      if (!dontPush) this.push({ id });
      return id;
    }
  }

  /**
   * Description
   */

  checkBlockScopedCollisions(kind: string, name: string, id: Object) {
    var local = this.getOwnBindingInfo(name);
    if (!local) return;


    if (kind === "param") return;
    if (kind === "hoisted" && local.kind === "let") return;

    var duplicate = false;
    if (!duplicate) duplicate = kind === "let" || kind === "const" || local.kind === "let" || local.kind === "const" || local.kind === "module";
    if (!duplicate) duplicate = local.kind === "param" && (kind === "let" || kind === "const");

    if (duplicate) {
      throw this.file.errorWithNode(id, messages.get("scopeDuplicateDeclaration", name), TypeError);
    }
  }

  /**
   * Description
   */

  rename(oldName: string, newName: string, block?) {
    newName = newName || this.generateUidIdentifier(oldName).name;

    var info = this.getBinding(oldName);
    if (!info) return;

    var state = {
      newName: newName,
      oldName: oldName,
      binding: info.identifier,
      info:    info
    };

    var scope = info.scope;
    scope.traverse(block || scope.block, renameVisitor, state);

    if (!block) {
      scope.removeOwnBinding(oldName);
      scope.bindings[newName] = info;
      state.binding.name = newName;
    }

    var file = this.file;
    if (file) {
      this._renameFromMap(file.moduleFormatter.localImports, oldName, newName, state.binding);
      //this._renameFromMap(file.moduleFormatter.localExports, oldName, newName);
    }
  }

  _renameFromMap(map, oldName, newName, value) {
    if (map[oldName]) {
      map[newName] = value;
      map[oldName] = null;
    }
  }

  /**
   * Description
   */

  dump() {
    var scope = this;
    do {
      console.log(scope.block.type, "Bindings:", Object.keys(scope.bindings));
    } while(scope = scope.parent);
    console.log("-------------");
  }

  /**
   * Description
   */

  toArray(node: Object, i?: number) {
    var file = this.file;

    if (t.isIdentifier(node)) {
      var binding = this.getBinding(node.name);
      if (binding && binding.constant && binding.isTypeGeneric("Array")) return node;
    }

    if (t.isArrayExpression(node)) {
      return node;
    }

    if (t.isIdentifier(node, { name: "arguments" })) {
      return t.callExpression(t.memberExpression(file.addHelper("slice"), t.identifier("call")), [node]);
    }

    var helperName = "to-array";
    var args = [node];
    if (i === true) {
      helperName = "to-consumable-array";
    } else if (i) {
      args.push(t.literal(i));
      helperName = "sliced-to-array";
      if (this.file.isLoose("es6.forOf")) helperName += "-loose";
    }
    return t.callExpression(file.addHelper(helperName), args);
  }

  /**
   * Description
   */

  registerDeclaration(path: TraversalPath) {
    var node = path.node;
    if (t.isFunctionDeclaration(node)) {
      this.registerBinding("hoisted", path);
    } else if (t.isVariableDeclaration(node)) {
      var declarations = path.get("declarations");
      for (var declar of (declarations: Array)) {
        this.registerBinding(node.kind, declar);
      }
    } else if (t.isClassDeclaration(node)) {
      this.registerBinding("let", path);
    } else if (t.isImportDeclaration(node) || t.isExportDeclaration(node)) {
      this.registerBinding("module", path);
    } else {
      this.registerBinding("unknown", path);
    }
  }

  /**
   * Description
   */

  registerConstantViolation(left: TraversalPath, right: TraversalPath) {
    var ids = left.getBindingIdentifiers();
    for (var name in ids) {
      var binding = this.getBinding(name);
      if (!binding) continue;

      if (right) {
        var rightType = right.typeAnnotation;
        if (rightType && binding.isCompatibleWithType(rightType)) continue;
      }

      binding.reassign(left, right);
    }
  }

  /**
   * Description
   */

  registerBinding(kind: string, path: TraversalPath) {
    if (!kind) throw new ReferenceError("no `kind`");

    if (path.isVariableDeclaration()) {
      var declarators = path.get("declarations");
      for (var declar of (declarators: Array)) {
        this.registerBinding(kind, declar);
      }
      return;
    }

    var ids = path.getBindingIdentifiers();

    for (var name in ids) {
      var id = ids[name];

      this.checkBlockScopedCollisions(kind, name, id);

      this.bindings[name] = new Binding({
        identifier: id,
        scope:      this,
        path:       path,
        kind:       kind
      });
    }
  }

  /**
   * Description
   */

  addGlobal(node: Object) {
    this.globals[node.name] = node;
  }

  /**
   * Description
   */

  hasUid(name): boolean {
    var scope = this;

    do {
      if (scope.uids[name]) return true;
    } while (scope = scope.parent);

    return false;
  }

  /**
   * Description
   */

  hasGlobal(name: string): boolean {
    var scope = this;

    do {
      if (scope.globals[name]) return true;
    } while (scope = scope.parent);

    return false;
  }

  /**
   * Description
   */

  hasReference(name: string): boolean {
    var scope = this;

    do {
      if (scope.references[name]) return true;
    } while (scope = scope.parent);

    return false;
  }

  /**
   * Description
   */

  recrawl() {
    this.path.setData("scopeInfo", null);
    this.crawl();
  }

  /**
   * Description
   */

  isPure(node) {
    if (t.isIdentifier(node)) {
      var bindingInfo = this.getBinding(node.name);
      return bindingInfo.constant;
    } else {
      return t.isPure(node);
    }
  }

  /**
   * Description
   */

  crawl() {
    var path = this.path;

    //

    var info = this.block._scopeInfo;
    if (info) return extend(this, info);

    info = this.block._scopeInfo = {
      references: object(),
      bindings:   object(),
      globals:    object(),
      uids:       object(),
    };

    extend(this, info);

    // ForStatement - left, init

    if (path.isLoop()) {
      for (let key of (t.FOR_INIT_KEYS: Array)) {
        var node = path.get(key);
        if (node.isBlockScoped()) this.registerBinding(node.node.kind, node);
      }
    }

    // FunctionExpression - id

    if (path.isFunctionExpression() && path.has("id")) {
      if (!t.isProperty(path.parent, { method: true })) {
        this.registerBinding("var", path.get("id"));
      }
    }

    // Class

    if (path.isClass() && path.has("id")) {
      this.registerBinding("var", path.get("id"));
    }

    // Function - params, rest

    if (path.isFunction()) {
      var params = path.get("params");
      for (let param of (params: Array)) {
        this.registerBinding("param", param);
      }
      this.traverse(path.get("body").node, blockVariableVisitor, this);
    }

    // Program, Function - var variables

    if (path.isProgram() || path.isFunction()) {
      this.traverse(path.node, functionVariableVisitor, {
        blockId: path.get("id").node,
        scope:   this
      });
    }

    // Program, BlockStatement, Function - let variables

    if (path.isBlockStatement() || path.isProgram()) {
      this.traverse(path.node, blockVariableVisitor, this);
    }

    // CatchClause - param

    if (path.isCatchClause()) {
      this.registerBinding("let", path.get("param"));
    }

    // ComprehensionExpression - blocks

    if (path.isComprehensionExpression()) {
      this.registerBinding("let", path);
    }

    // Program

    if (path.isProgram()) {
      this.traverse(path.node, programReferenceVisitor, this);
    }
  }

  /**
   * Description
   */

  push(opts: Object) {
    var path = this.path;

    if (path.isLoop() || path.isCatchClause() || path.isFunction()) {
      t.ensureBlock(path.node);
      path = path.get("body");
    }

    if (!path.isBlockStatement() && !path.isProgram()) {
      path = this.getBlockParent().path;
    }

    var unique = opts.unique;
    var kind   = opts.kind || "var";

    var dataKey = `declaration:${kind}`;
    var declar  = !unique && path.getData(dataKey);

    if (!declar) {
      declar = t.variableDeclaration(kind, []);
      declar._generated = true;
      declar._blockHoist = 2;

      this.file.attachAuxiliaryComment(declar);

      var [declarPath] = path.get("body")[0]._containerInsertBefore([declar]);
      this.registerBinding(kind, declarPath);
      if (!unique) path.setData(dataKey, declar);
    }

    declar.declarations.push(t.variableDeclarator(opts.id, opts.init));
  }

  /**
   * Walk up to the top of the scope tree and get the `Program`.
   */

  getProgramParent() {
    var scope = this;
    while (scope.parent) {
      scope = scope.parent;
    }
    return scope;
  }

  /**
   * Walk up the scope tree until we hit either a Function or reach the
   * very top and hit Program.
   */

  getFunctionParent() {
    var scope = this;
    while (scope.parent && !t.isFunction(scope.block)) {
      scope = scope.parent;
    }
    return scope;
  }

  /**
   * Walk up the scope tree until we hit either a BlockStatement/Loop or reach the
   * very top and hit Program.
   */

  getBlockParent() {
    var scope = this;
    while (scope.parent && !t.isFunction(scope.block) && !t.isLoop(scope.block) && !t.isFunction(scope.block)) {
      scope = scope.parent;
    }
    return scope;
  }

  /**
   * Walks the scope tree and gathers **all** bindings.
   */

  getAllBindings(): Object {
    var ids = object();

    var scope = this;
    do {
      defaults(ids, scope.bindings);
      scope = scope.parent;
    } while (scope);

    return ids;
  }

  /**
   * Walks the scope tree and gathers all declarations of `kind`.
   */

  getAllBindingsOfKind(): Object {
    var ids = object();

    for (let kind of (arguments: Array)) {
      var scope = this;
      do {
        for (var name in scope.bindings) {
          var binding = scope.bindings[name];
          if (binding.kind === kind) ids[name] = binding;
        }
        scope = scope.parent;
      } while (scope);
    }

    return ids;
  }

  /**
   * Description
   */

  bindingIdentifierEquals(name: string, node: Object): boolean {
    return this.getBindingIdentifier(name) === node;
  }

  /**
   * Description
   */

  getBinding(name: string) {
    var scope = this;

    do {
      var binding = scope.getOwnBindingInfo(name);
      if (binding) return binding;
    } while (scope = scope.parent);
  }

  /**
   * Description
   */

  getOwnBindingInfo(name: string) {
    return this.bindings[name];
  }

  /**
   * Description
   */

  getBindingIdentifier(name: string) {
    var info = this.getBinding(name);
    return info && info.identifier;
  }

  /**
   * Description
   */

  getOwnBindingIdentifier(name: string) {
    var binding = this.bindings[name];
    return binding && binding.identifier;
  }

  /**
   * Description
   */

  hasOwnBinding(name: string) {
    return !!this.getOwnBindingInfo(name);
  }

  /**
   * Description
   */

  hasBinding(name: string) {
    if (!name) return false;
    if (this.hasOwnBinding(name)) return true;
    if (this.parentHasBinding(name)) return true;
    if (this.hasUid(name)) return true;
    if (includes(Scope.globals, name)) return true;
    if (includes(Scope.contextVariables, name)) return true;
    return false;
  }

  /**
   * Description
   */

  parentHasBinding(name: string) {
    return this.parent && this.parent.hasBinding(name);
  }

  /**
   * Move a binding of `name` to another `scope`.
   */

  moveBindingTo(name, scope) {
    var info = this.getBinding(name);
    if (info) {
      info.scope.removeOwnBinding(name);
      info.scope = scope;
      scope.bindings[name] = info;
    }
  }

  /**
   * Description
   */

  removeOwnBinding(name: string) {
    delete this.bindings[name];
  }

  /**
   * Description
   */

  removeBinding(name: string) {
    var info = this.getBinding(name);
    if (info) info.scope.removeOwnBinding(name);
  }
}
