// Copyright (c) 2017 Computer Vision Center (CVC) at the Universitat Autonoma
// de Barcelona (UAB).
//
// This work is licensed under the terms of the MIT license.
// For a copy, see <https://opensource.org/licenses/MIT>.

#pragma once

#include "CoreMinimal.h"
#include "GameFramework/Actor.h"

#include "Components/SceneComponent.h"
#include "Components/BillboardComponent.h"

#include "Traffic/RoutePlanner.h"

#include "Vehicle/VehicleSpawnPoint.h"

#include "Runtime/Core/Public/Misc/OutputDeviceNull.h"

#include <compiler/disable-ue4-macros.h>
#include <carla/opendrive/OpenDrive.h>
#include <compiler/enable-ue4-macros.h>

#include "OpenDriveActor.generated.h"

UCLASS()
class CARLA_API AOpenDriveActor : public AActor
{
  GENERATED_BODY()

protected:

  /// A UBillboardComponent to hold Icon sprite
  UBillboardComponent *SpriteComponent;

  /// Sprite for the Billboard Component
  UTexture2D *SpriteTexture;

private:

  UPROPERTY()
  TArray<ARoutePlanner *> RoutePlanners;

  UPROPERTY()
  TArray<AVehicleSpawnPoint *> VehicleSpawners;

  UPROPERTY()
  TArray<AActor *> PersistentTrafficLights;

  UPROPERTY()
  TArray<AActor *> PersistentTrafficSigns;

  UPROPERTY()
  TSubclassOf<class AActor> TrafficLightBlueprintClass;

  UPROPERTY()
  TSubclassOf<class AActor> TrafficGroupBlueprintClass;

  UPROPERTY()
  TSubclassOf<class AActor> TrafficSign30BlueprintClass;

  UPROPERTY()
  TSubclassOf<class AActor> TrafficSign40BlueprintClass;

  UPROPERTY()
  TSubclassOf<class AActor> TrafficSign60BlueprintClass;

  UPROPERTY()
  TSubclassOf<class AActor> TrafficSign90BlueprintClass;

  UPROPERTY()
  TSubclassOf<class AActor> TrafficSign100BlueprintClass;

#if WITH_EDITORONLY_DATA
  /// Generate the road network using an OpenDrive file (named as the current
  /// .umap)
  UPROPERTY(Category = "Generate", EditAnywhere)
  bool bGenerateRoutes = false;
#endif // WITH_EDITORONLY_DATA

  /// Distance between waypoints where the cars will drive
  UPROPERTY(Category = "Generate", EditAnywhere, meta = (ClampMin = "0.01", UIMin = "0.01"))
  float RoadAccuracy = 2.f;

  /// Trigger elevantion
  UPROPERTY(Category = "Generate", EditAnywhere, meta = (ClampMin = "0.01", UIMin = "0.01"))
  float TriggersHeight = 100.f;

#if WITH_EDITORONLY_DATA
  /// Remove the previously generated road network. Also, it will remove
  /// spawners if necessary
  UPROPERTY(Category = "Generate", EditAnywhere)
  bool bRemoveRoutes = false;
#endif // WITH_EDITORONLY_DATA

  /// If true, spawners will be placed when generating the routes
  UPROPERTY(Category = "Spawners", EditAnywhere)
  bool bAddSpawners = false;

  /// If true, spawners will be placed on junctions too
  UPROPERTY(Category = "Spawners", EditAnywhere)
  bool bOnIntersections = false;

  /// Determine the height where the spawners will be placed, relative to each
  /// RoutePlanner
  UPROPERTY(Category = "Spawners", EditAnywhere)
  float SpawnersHeight = 300.f;

#if WITH_EDITORONLY_DATA
  /// Remove already placed spawners if necessary
  UPROPERTY(Category = "Spawners", EditAnywhere)
  bool bRemoveCurrentSpawners = false;
#endif // WITH_EDITORONLY_DATA

#if WITH_EDITORONLY_DATA
  /// Show / Hide additional debug information
  UPROPERTY(Category = "Debug", EditAnywhere)
  bool bShowDebug = true;
#endif // WITH_EDITORONLY_DATA

public:

  AOpenDriveActor(const FObjectInitializer &ObjectInitializer);

  void BuildRoutes();

  void BuildRoutes(FString MapName);

  /// Remove all the existing ARoutePlanner and VehicleSpawners previously
  /// generated by this class to avoid overlapping
  void RemoveRoutes();

  void DebugRoutes() const;

  void RemoveDebugRoutes() const;

#if WITH_EDITOR
  void PostEditChangeProperty(struct FPropertyChangedEvent &PropertyChangedEvent);

#endif // WITH_EDITOR

  void AddSpawners();

  void RemoveSpawners();

};
