/*
 * Copyright 2012 GitHub Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.mobile.tests;

import static org.eclipse.egit.github.core.event.Event.TYPE_COMMIT_COMMENT;
import static org.eclipse.egit.github.core.event.Event.TYPE_CREATE;
import static org.eclipse.egit.github.core.event.Event.TYPE_DELETE;
import static org.eclipse.egit.github.core.event.Event.TYPE_FOLLOW;
import static org.eclipse.egit.github.core.event.Event.TYPE_GIST;
import static org.eclipse.egit.github.core.event.Event.TYPE_GOLLUM;
import static org.eclipse.egit.github.core.event.Event.TYPE_ISSUES;
import static org.eclipse.egit.github.core.event.Event.TYPE_ISSUE_COMMENT;
import static org.eclipse.egit.github.core.event.Event.TYPE_MEMBER;
import static org.eclipse.egit.github.core.event.Event.TYPE_PUBLIC;
import static org.eclipse.egit.github.core.event.Event.TYPE_PULL_REQUEST;
import static org.eclipse.egit.github.core.event.Event.TYPE_PUSH;
import static org.eclipse.egit.github.core.event.Event.TYPE_TEAM_ADD;
import static org.eclipse.egit.github.core.event.Event.TYPE_WATCH;
import android.content.Context;
import android.test.InstrumentationTestCase;
import android.test.UiThreadTest;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.TextView;

import com.github.mobile.R;
import com.github.mobile.ui.user.NewsListAdapter;
import com.github.mobile.util.AvatarLoader;

import java.util.Date;

import org.eclipse.egit.github.core.Gist;
import org.eclipse.egit.github.core.Issue;
import org.eclipse.egit.github.core.Team;
import org.eclipse.egit.github.core.User;
import org.eclipse.egit.github.core.event.CommitCommentPayload;
import org.eclipse.egit.github.core.event.CreatePayload;
import org.eclipse.egit.github.core.event.DeletePayload;
import org.eclipse.egit.github.core.event.Event;
import org.eclipse.egit.github.core.event.EventRepository;
import org.eclipse.egit.github.core.event.FollowPayload;
import org.eclipse.egit.github.core.event.GistPayload;
import org.eclipse.egit.github.core.event.IssueCommentPayload;
import org.eclipse.egit.github.core.event.IssuesPayload;
import org.eclipse.egit.github.core.event.MemberPayload;
import org.eclipse.egit.github.core.event.PullRequestPayload;
import org.eclipse.egit.github.core.event.PushPayload;
import org.eclipse.egit.github.core.event.TeamAddPayload;

/**
 * Tests of the news text rendering
 */
public class NewsEventTextTest extends InstrumentationTestCase {

    private NewsListAdapter adapterWithRepoName;
    private NewsListAdapter adapterWithoutRepoName;

    private TextView text;

    private User actor;

    private EventRepository repo;

    private Date date;

    @Override
    protected void setUp() throws Exception {
        super.setUp();

        date = new Date();
        actor = new User().setLogin("user");
        repo = new EventRepository().setName("user/repo");

        Context context = getInstrumentation().getTargetContext();
        adapterWithRepoName = new NewsListAdapter(LayoutInflater.from(context),
                new AvatarLoader(context));
        adapterWithoutRepoName = new NewsListAdapter(LayoutInflater.from(context),
                new Event[0], new AvatarLoader(context), false);
    }

    private Event createEvent(String type) {
        Event event = new Event();
        event.setCreatedAt(date);
        event.setType(type);
        event.setActor(actor);
        event.setRepo(repo);
        return event;
    }

    private void verify(String expected) {
        CharSequence actual = text.getText();
        assertNotNull(actual);
        assertEquals(expected, actual.toString());
    }

    private void updateViewWithRepoName(Event event) {
        adapterWithRepoName.setItems(new Object[] { event });
        View view = adapterWithRepoName.getView(0, null, null);
        assertNotNull(view);
        text = (TextView) view.findViewById(R.id.tv_event);
        assertNotNull(text);
    }

    private void updateViewWithoutRepoName(Event event) {
        adapterWithoutRepoName.setItems(new Object[] { event });
        View view = adapterWithoutRepoName.getView(0, null, null);
        assertNotNull(view);
        text = (TextView) view.findViewById(R.id.tv_event);
        assertNotNull(text);
    }

    /**
     * Verify text of commit comment event
     */
    @UiThreadTest
    public void testCommitCommentEvent() {
        Event event = createEvent(TYPE_COMMIT_COMMENT);
        event.setPayload(new CommitCommentPayload());

        updateViewWithRepoName(event);
        verify("user commented on user/repo");
        updateViewWithoutRepoName(event);
        verify("user commented");
    }

    /**
     * Verify text of create event
     */
    @UiThreadTest
    public void testCreateRepositoryEvent() {
        Event event = createEvent(TYPE_CREATE);
        CreatePayload payload = new CreatePayload();
        payload.setRefType("repository");
        event.setPayload(payload);

        updateViewWithRepoName(event);
        verify("user created repository user/repo");
        updateViewWithoutRepoName(event);
        verify("user created repository");
    }

    /**
     * Verify text of create event
     */
    @UiThreadTest
    public void testCreateBranchEvent() {
        Event event = createEvent(TYPE_CREATE);
        CreatePayload payload = new CreatePayload();
        payload.setRefType("branch");
        payload.setRef("b1");
        event.setPayload(payload);

        updateViewWithRepoName(event);
        verify("user created branch b1 at user/repo");
        updateViewWithoutRepoName(event);
        verify("user created branch b1");
    }

    /**
     * Verify text of deleve event
     */
    @UiThreadTest
    public void testDelete() {
        Event event = createEvent(TYPE_DELETE);
        DeletePayload payload = new DeletePayload();
        payload.setRefType("branch");
        payload.setRef("b1");
        event.setPayload(payload);

        updateViewWithRepoName(event);
        verify("user deleted branch b1 at user/repo");
        updateViewWithoutRepoName(event);
        verify("user deleted branch b1");
    }

    /**
     * Verify text of follow event
     */
    @UiThreadTest
    public void testFollow() {
        Event event = createEvent(TYPE_FOLLOW);
        FollowPayload payload = new FollowPayload();
        payload.setTarget(new User().setLogin("user2"));
        event.setPayload(payload);

        updateViewWithRepoName(event);
        verify("user started following user2");
        updateViewWithoutRepoName(event);
        verify("user started following user2");
    }

    /**
     * Verify text of Gist event
     */
    @UiThreadTest
    public void testGist() {
        Event event = createEvent(TYPE_GIST);
        GistPayload payload = new GistPayload();
        payload.setAction("create");
        payload.setGist(new Gist().setId("1"));
        event.setPayload(payload);

        updateViewWithRepoName(event);
        verify("user created Gist 1");
        updateViewWithoutRepoName(event);
        verify("user created Gist 1");
    }

    /**
     * Verify text of wiki event
     */
    @UiThreadTest
    public void testWiki() {
        Event event = createEvent(TYPE_GOLLUM);

        updateViewWithRepoName(event);
        verify("user updated the wiki in user/repo");
        updateViewWithoutRepoName(event);
        verify("user updated the wiki");
    }

    /**
     * Verify text of issue comment event
     */
    @UiThreadTest
    public void testIssueComment() {
        Event event = createEvent(TYPE_ISSUE_COMMENT);
        IssueCommentPayload payload = new IssueCommentPayload();
        payload.setIssue(new Issue().setNumber(5));
        event.setPayload(payload);

        updateViewWithRepoName(event);
        verify("user commented on issue 5 on user/repo");
        updateViewWithoutRepoName(event);
        verify("user commented on issue 5");
    }

    /**
     * Verify text of issue event
     */
    @UiThreadTest
    public void testIssue() {
        Event event = createEvent(TYPE_ISSUES);
        IssuesPayload payload = new IssuesPayload();
        payload.setAction("closed");
        payload.setIssue(new Issue().setNumber(8));
        event.setPayload(payload);

        updateViewWithRepoName(event);
        verify("user closed issue 8 on user/repo");
        updateViewWithoutRepoName(event);
        verify("user closed issue 8");
    }

    /**
     * Verify text of member event
     */
    @UiThreadTest
    public void testAddMember() {
        Event event = createEvent(TYPE_MEMBER);
        event.setPayload(new MemberPayload().setMember(new User()
                .setLogin("person")));

        updateViewWithRepoName(event);
        verify("user added person as a collaborator to user/repo");
        updateViewWithoutRepoName(event);
        verify("user added person as a collaborator");
    }

    /**
     * Verify text of open sourced event
     */
    @UiThreadTest
    public void testOpenSourced() {
        Event event = createEvent(TYPE_PUBLIC);

        updateViewWithRepoName(event);
        verify("user made user/repo public");
        updateViewWithoutRepoName(event);
        verify("user made repository public");
    }

    /**
     * Verify text of watch event
     */
    @UiThreadTest
    public void testWatch() {
        Event event = createEvent(TYPE_WATCH);

        updateViewWithRepoName(event);
        verify("user starred user/repo");
        updateViewWithoutRepoName(event);
        verify("user starred repository");
    }

    /**
     * Verify text of pull request event
     */
    @UiThreadTest
    public void testPullRequest() {
        Event event = createEvent(TYPE_PULL_REQUEST);
        PullRequestPayload payload = new PullRequestPayload();
        payload.setNumber(30);
        payload.setAction("merged");
        event.setPayload(payload);

        updateViewWithRepoName(event);
        verify("user merged pull request 30 on user/repo");
        updateViewWithoutRepoName(event);
        verify("user merged pull request 30");
    }

    /**
     * Verify text of push event
     */
    @UiThreadTest
    public void testPush() {
        Event event = createEvent(TYPE_PUSH);
        PushPayload payload = new PushPayload();
        payload.setRef("refs/heads/master");
        event.setPayload(payload);

        updateViewWithRepoName(event);
        verify("user pushed to master at user/repo");
        updateViewWithoutRepoName(event);
        verify("user pushed to master");
    }

    /**
     * Verify text of push event
     */
    @UiThreadTest
    public void testTeamAdd() {
        Event event = createEvent(TYPE_TEAM_ADD);
        TeamAddPayload payload = new TeamAddPayload();
        payload.setTeam(new Team().setName("t1"));
        payload.setUser(new User().setLogin("u2"));
        event.setPayload(payload);

        updateViewWithRepoName(event);
        verify("user added u2 to team t1");
        updateViewWithoutRepoName(event);
        verify("user added u2 to team t1");
    }
}
