/* Copyright 2020 Obosob <obosob@riseup.net>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include QMK_KEYBOARD_H

enum layer_names {
  _QWERTY,
  _COLMAK,
  _NUM,
  _NAV,
  _SYM,
  _MISC,
  _ADJUST
};

#define MY_SPC  LT(_NAV, KC_SPC)
#define MY_ENT  LT(_NUM, KC_ENT)
#define MY_BSPC SFT_T(KC_BSPC)
#define MY_DEL  SFT_T(KC_DEL)
#define MY_ESC  CTL_T(KC_ESC)
#define MY_TAB  RALT_T(KC_TAB)

#define KC_NUPI S(KC_NUBS)

#define OS_SFT  OSM(MOD_LSFT)
#define OS_CTL  OSM(MOD_LCTL)
#define OS_GUI  OSM(MOD_LGUI)
#define OS_ALT  OSM(MOD_LALT)

enum keycodes {
  QWERTY = SAFE_RANGE,
  COLEMAK,
// layers
  SYM,
  MISC,
// special keys
  ENC_PRS,
  DOTSPC,
  COMMSPC,
  EXLMSPC,
  QUESSPC,
  DQUOT,
  QUOT,
  PRN,
  BRC,
  CBR,
  LTGT,
  LTGTC,
  GRV,
  A_TAB,
  AS_TAB,
  WOKE,
};

const uint16_t PROGMEM keymaps[][MATRIX_ROWS][MATRIX_COLS] = {

//   Qwerty            ________                                                         ________
//            ________|      E |________                                       ________|      I |________
//           |      W |        |      R |________                     ________|      U |        |      O |
//   ________|        |________|        |      T |                   |      Y |        |________|        |________
//  |      Q |________|      D |________|        |                   |        |________|      K |________|      P |
//  |        |      S |        |      F |________|                   |________|      J |        |      L |        |
//  |________|        |________|        |      G |                   |      H |        |________|        |________|
//  |      A |________|      C |________|        |                   |        |________|    SYM |________|      : |
//  |        |      X |        |      V |________|                   |________|      M |        |   MISC |      ; |
//  |________|        |________|        |      B |                   |      N |        |________|        |________|
//  |      Z |________|        |________|        |                   |        |________|        |________|      ? |
//  |        |                 |    Esc |________|                   |________|    Tab |                 |      / |
//  |________|                 | Ctrl   |  Enter |________   ________|  Space | Alt    |                 |________|
//                             |________| NUM    |  Del   | |   BkSp | NAV    |________|
//                                       ________| Shift  | | Shift  |________
//                                               |________| |________|

    [_QWERTY] = LAYOUT_split_3x5_3(
      KC_Q,    KC_W,    KC_E,    KC_R,    KC_T,                      KC_Y,    KC_U,    KC_I,    KC_O,    KC_P,
      KC_A,    KC_S,    KC_D,    KC_F,    KC_G,                      KC_H,    KC_J,    KC_K,    KC_L,    KC_SCLN,
      KC_Z,    KC_X,    KC_C,    KC_V,    KC_B,                      KC_N,    KC_M,    SYM,     MISC,    KC_SLSH,
                                 MY_ESC,  MY_ENT,  MY_DEL,  MY_BSPC, MY_SPC,  MY_TAB
    ),
//   Colemak (Mod-DH)  ________                                                         ________
//            ________|      F |________                                       ________|      U |________
//           |      W |        |      P |________                     ________|      L |        |      Y |
//   ________|        |________|        |      B |                   |      J |        |________|        |________
//  |      Q |________|      S |________|        |                   |        |________|      E |________|      : |
//  |        |      R |        |      T |________|                   |________|      N |        |      I |      ; |
//  |________|        |________|        |      G |                   |      K |        |________|        |________|
//  |      A |________|      C |________|        |                   |        |________|    SYM |________|      O |
//  |        |      X |        |      D |________|                   |________|      H |        |   MISC |        |
//  |________|        |________|        |      V |                   |      M |        |________|        |________|
//  |      Z |________|        |________|        |                   |        |________|        |________|      ? |
//  |        |                 |    Esc |________|                   |________|    Tab |                 |      / |
//  |________|                 | Ctrl   |  Enter |________   ________|  Space | Alt    |                 |________|
//                             |________| NUM    |  Del   | |   BkSp | NAV    |________|
//                                       ________| Shift  | | Shift  |________
//                                               |________| |________|

    [_COLMAK] = LAYOUT_split_3x5_3(
      KC_Q,    KC_W,    KC_F,    KC_P,    KC_B,                      KC_J,    KC_L,    KC_U,    KC_Y,    KC_SCLN,
      KC_A,    KC_R,    KC_S,    KC_T,    KC_G,                      KC_K,    KC_N,    KC_E,    KC_I,    KC_O,
      KC_Z,    KC_X,    KC_C,    KC_D,    KC_V,                      KC_M,    KC_H,    SYM,     MISC,    KC_SLSH,
                                 MY_ESC,  MY_ENT,  MY_DEL,  MY_BSPC, MY_SPC,  MY_TAB
    ),
//   Number            ________                                                         ________
//            ________|     F3 |________                                       ________|      8 |________
//           |     F2 |        |     F4 |________                     ________|      7 |        |      9 |
//   ________|        |________|        |        |                   |      + |        |________|        |________
//  |     F1 |________|     F7 |________|        |                   |        |________|      5 |________|      / |
//  |        |     F6 |        |     F8 |________|                   |________|      4 |        |      6 |        |
//  |________|        |________|        |        |                   |      0 |        |________|        |________|
//  |     F5 |________|    F11 |________|        |                   |        |________|      2 |________|      . |
//  |        |    F10 |        |    F12 |________|                   |________|      1 |        |      3 |        |
//  |________|        |________|        |        |                   |      - |        |________|        |________|
//  |     F9 |________|        |________|        |                   |        |________|        |________|      * |
//  |        |                 |        |________|                   |________|        |                 |        |
//  |________|                 |        |        |________   ________| ADJUST |        |                 |________|
//                             |________|  ****  |        | |        |        |________|
//                                      |________|        | |        |________|
//                                               |________| |________|
    [_NUM] = LAYOUT_split_3x5_3(
     KC_F1,   KC_F2,   KC_F3,   KC_F4,   XXXXXXX,                    KC_PLUS, KC_7,    KC_8,    KC_9,    KC_SLSH,
     KC_F5,   KC_F6,   KC_F7,   KC_F8,   XXXXXXX,                    KC_COMM, KC_4,    KC_5,    KC_6,    KC_DOT,
     KC_F9,   KC_F10,  KC_F11,  KC_F12,  XXXXXXX,                    KC_MINS, KC_1,    KC_2,    KC_3,    KC_ASTR,
                                XXXXXXX, _______, XXXXXXX,  _______, _______, KC_0
    ),
//   Navigation        ________                                                          ________
//            ________|     OS |________                                        ________|   PgUp |________
//           |     OS |   Ctrl |     OS |________                      ________|   PgDn |        |    End |
//   ________|    Alt |________|  Shift | AltTab |                    |   Home |        |________|        |________
//  |     OS |________|   Ctrl |________|        |                    |        |________|     Up |________| PrtScr |
//  |  Super |    Alt |        |  Shift |________|                    |________|   Down |        |  Right |        |
//  |________|        |________|        |        |                    |   Left |        |________|        |________|
//  |  Super |________|        |________|        |                    |        |________|        |________| Insert |
//  |        |        |        |        |________|                    |________|        |        |        |        |
//  |________|        |________|        |        |                    |        |        |________|        |________|
//  |        |________|        |________|        |                    |        |________|        |________|        |
//  |        |                 |        |________|                    |________|        |                 |        |
//  |________|                 |        | ADJUST |________    ________|        |        |                 |________|
//                             |________|        |        |  |        |  ****  |________|
//                                      |________|        |  |        |________|
//                                               |________|  |________|
    [_NAV] = LAYOUT_split_3x5_3(
      OS_GUI,  OS_ALT,  OS_CTL,  OS_SFT,  XXXXXXX,                    KC_HOME, KC_PGDN, KC_PGUP, KC_END,  KC_PSCR,
      KC_LGUI, KC_LALT, KC_LCTL, KC_LSFT, XXXXXXX,                    KC_LEFT, KC_DOWN, KC_UP,   KC_RGHT, KC_INS,
      XXXXXXX, XXXXXXX, XXXXXXX, XXXXXXX, XXXXXXX,                    XXXXXXX, AS_TAB,  A_TAB,   XXXXXXX, XXXXXXX,
                                 _______, _______, _______,  XXXXXXX, _______, XXXXXXX
      ),
//   Symbol            ________                                                          ________
//            ________|      { |________                                        ________|        |________
//           |      > |        |      } |________                      ________|        |        |        |
//   ________|        |________|        |      ` |                    |        |        |________|        |________
//  |      < |________|      ( |________|        |                    |        |________|        |________|      | |
//  |        |      £ |        |      ) |________|                    |________|      - |        |      & |        |
//  |________|        |________|        |      ^ |                    |      _ |        |________|        |________|
//  |      * |________|     [  |________|        |                    |        |________|        |________|      ~ |
//  |        |      $ |        |      ] |________|                    |________|      + |  ****  |      % |        |
//  |________|        |________|        |      @ |                    |      = |        |________|        |________|
//  |      \ |________|        |________|        |                    |        |________|        |________|      # |
//  |        |                 |      ? |________|                    |________|      ! |                 |        |
//  |________|                 |        |      ' |________    ________|      . |        |                 |________|
//                             |________|        |      " |  |      , |        |________|
//                                      |________|        |  |        |________|
//                                               |________|  |________|
//
    [_SYM] = LAYOUT_split_3x5_3(
         KC_LT,   KC_GT,   KC_LCBR, KC_RCBR, KC_GRV,                   XXXXXXX, XXXXXXX, XXXXXXX, XXXXXXX, KC_NUPI,
         KC_ASTR, KC_HASH, KC_LPRN, KC_RPRN, KC_CIRC,                  KC_UNDS, KC_MINS, XXXXXXX, KC_AMPR, KC_PIPE,
         KC_NUBS, KC_DLR,  KC_LBRC, KC_RBRC, KC_DQUO,                  KC_EQL,  KC_PLUS, _______, KC_PERC, KC_NUHS,
                                    KC_QUES, KC_QUOT, KC_AT,  KC_COMM, KC_DOT,  KC_EXLM
        ),
//    Miscellaneous    ________                                                          ________
//            ________|    {|} |________                                        ________|        |________
//           |   </|> |        |    {|} |________                      ________|        |        |        |
//   ________|        |________|        |        |                    |        |        |________|        |________
//  |    <|> |________|    (|) |________|        |                    |        |________|        |________|        |
//  |        |        |        |    (|) |________|                    |________|        |        |        |        |
//  |________|        |________|        |        |                    |        |        |________|        |________|
//  |        |________|    [|] |________|        |                    |        |________|        |________|        |
//  |        |        |        |    [|] |________|                    |________|        |        |        |        |
//  |________|        |________|        |        |                    |        |        |________|  ****  |________|
//  |        |________|        |________|        |                    |        |________|        |________|        |
//  |        |                 |    ?_^ |________|                    |________|    !_^ |                 |        |
//  |________|                 |        |    '|' |________    ________|    ._^ |        |                 |________|
//                             |________|        |    "|" |  |     ,_ |        |________|
//                                      |________|        |  |        |________|
//                                               |________|  |________|
//
    [_MISC] = LAYOUT_split_3x5_3(
        LTGT,    LTGTC,   CBR,     CBR,     GRV,                        XXXXXXX, XXXXXXX, XXXXXXX, XXXXXXX, XXXXXXX,
        XXXXXXX, XXXXXXX, PRN,     PRN,     XXXXXXX,                    XXXXXXX, XXXXXXX, XXXXXXX, XXXXXXX, XXXXXXX,
        XXXXXXX, XXXXXXX, BRC,     BRC,     XXXXXXX,                    XXXXXXX, XXXXXXX, XXXXXXX, _______, XXXXXXX,
                                   QUESSPC, QUOT,    DQUOT,    COMMSPC, DOTSPC,  EXLMSPC
        ),
//   Adjust            ________                                                          ________
//            ________|  QK_BOOT |________                                        ________|        |________
//           |        | EEPROM |  QK_BOOT |________                      ________|        |        |        |
//   ________|        |________|        |        |                    |        |        |________|        |________
//  |        |________| RGB    |________|        |                    |        |________|        |________|        |
//  |        | RGB    | Hue+   | RGB    |________|                    |________|        |        |        |        |
//  |________| Toggle |________| Sat+   | RGB    |                    |        |        |________|        |________|
//  |        |________| RGB    |________| Bri+   |                    |        |________|        |________|        |
//  |        | RGB    | Hue-   | RGB    |________|                    |________|        |        |        |        |
//  |________| Mode   |________| Sat-   | RGB    |                    |        |        |________|        |________|
//  |        |________|        |________| Bri-   |                    |        |________|        |________|        |
//  |        |                 |        |________|                    |________|        |                 |        |
//  |________|                 |        |        |________    ________|        |        |                 |________|
//                             |________|  ****  |        |  |        |  ****  |________|
//                                      |________|        |  |        |________|
//                                               |________|  |________|
    [_ADJUST] = LAYOUT_split_3x5_3(
        XXXXXXX, WOKE,    EE_CLR,  QK_BOOT, XXXXXXX,                     XXXXXXX, XXXXXXX, XXXXXXX, XXXXXXX, XXXXXXX,
        XXXXXXX, RGB_TOG, RGB_HUI, RGB_SAI, RGB_VAI,                     XXXXXXX, QWERTY,  COLEMAK, XXXXXXX, XXXXXXX,
        XXXXXXX, RGB_MOD, RGB_HUD, RGB_SAD, RGB_VAD,                     XXXXXXX, XXXXXXX, XXXXXXX, XXXXXXX, XXXXXXX,
                                   XXXXXXX, XXXXXXX, XXXXXXX,   XXXXXXX, XXXXXXX, XXXXXXX
        ),
};

bool alt_tabbing = false;
bool woke_mode = false;
uint16_t rgb_edit_timer = 0;
uint16_t last_rgb_char = 0;

bool process_record_user(uint16_t keycode, keyrecord_t *record) {
    switch (keycode) {
        case QWERTY:
            if (record->event.pressed) {
                set_single_persistent_default_layer(_QWERTY);
            }
            return false;
        case COLEMAK:
            if (record->event.pressed) {
                set_single_persistent_default_layer(_COLMAK);
            }
            return false;
        case SYM:
            if (record->event.pressed) {
                layer_on(_SYM);
            } else {
                layer_off(_SYM);
            }
            return false;
        case MY_SPC:
            if (!record->event.pressed) {
                if(alt_tabbing) {
                    unregister_code(KC_LALT);
                    alt_tabbing = false;
                }
            }
            return true;
        case MISC:
            if (record->event.pressed) {
                layer_on(_MISC);
            } else {
                layer_off(_MISC);
            }
            return false;
        case DOTSPC:
            if (record->event.pressed) {
                tap_code(KC_DOT);
                tap_code(KC_SPC);
                set_oneshot_mods(MOD_LSFT);
            }
            return false;
        case COMMSPC:
            if (record->event.pressed) {
                tap_code(KC_COMM);
                tap_code(KC_SPC);
            }
            return false;
        case EXLMSPC:
            if (record->event.pressed) {
                tap_code16(KC_EXLM);
                tap_code(KC_SPC);
                set_oneshot_mods(MOD_LSFT);
            }
            return false;
        case QUESSPC:
            if (record->event.pressed) {
                tap_code16(KC_QUES);
                tap_code(KC_SPC);
                set_oneshot_mods(MOD_LSFT);
            }
            return false;
        case QUOT:
            if (record->event.pressed) {
                tap_code(KC_QUOT);
                tap_code(KC_QUOT);
                tap_code(KC_LEFT);
            }
            return false;
        case DQUOT:
            if (record->event.pressed) {
                tap_code16(KC_DQUO);
                tap_code16(KC_DQUO);
                tap_code(KC_LEFT);
            }
            return false;
        case PRN:
            if (record->event.pressed) {
                tap_code16(KC_LPRN);
                tap_code16(KC_RPRN);
                tap_code(KC_LEFT);
            }
            return false;
        case BRC:
            if (record->event.pressed) {
                tap_code(KC_LBRC);
                tap_code(KC_RBRC);
                tap_code(KC_LEFT);
            }
            return false;
        case CBR:
            if (record->event.pressed) {
                tap_code16(KC_LCBR);
                tap_code16(KC_RCBR);
                tap_code(KC_LEFT);
            }
            return false;
        case LTGT:
            if (record->event.pressed) {
                tap_code16(KC_LT);
                tap_code16(KC_GT);
                tap_code(KC_LEFT);
            }
            return false;
        case LTGTC:
            if (record->event.pressed) {
                tap_code16(KC_LT);
                tap_code(KC_SLSH);
                tap_code16(KC_GT);
                tap_code(KC_LEFT);
            }
            return false;
        case GRV:
            if (record->event.pressed) {
                tap_code(KC_GRV);
                tap_code(KC_GRV);
                tap_code(KC_LEFT);
            }
            return false;
        case AS_TAB:
            if (record->event.pressed) {
                register_code(KC_LSFT);
            } else {
                unregister_code(KC_LSFT);
            }
        case A_TAB:
            if (record->event.pressed) {
                register_code(KC_LALT);
                register_code(KC_TAB);
                alt_tabbing = true;
            } else {
                unregister_code(KC_TAB);
            }
            return false;
        case WOKE:
            if (!record->event.pressed) {
                woke_mode = !woke_mode;
            }
            return false;
        case RGB_TOG:
        case RGB_MOD:
        case RGB_HUD:
        case RGB_HUI:
        case RGB_SAD:
        case RGB_SAI:
        case RGB_VAI:
        case RGB_VAD:
            rgb_edit_timer = timer_read();
            last_rgb_char = keycode;
            return true;
    }

    if (woke_mode && ((keycode >= KC_A && keycode <= KC_Z) || keycode == MY_BSPC)) {
        if (record->event.pressed) {
            tap_code(KC_CAPS);
        }
    }

    return true;
}

layer_state_t layer_state_set_user(layer_state_t state) {
    state = update_tri_layer_state(state, _NAV, _NUM, _ADJUST);
    return state;
}

#ifdef OLED_ENABLE

#define ANIM_NUM_FRAMES 4
#define ANIM_FRAME_DURATION 100

#ifdef ANIM_REVERSE
#define ANIM_TOTAL_FRAMES (2*((ANIM_NUM_FRAMES)-1))
#else
#define ANIM_TOTAL_FRAMES ANIM_NUM_FRAMES
#endif

uint16_t anim_timer = 0;
uint8_t current_anim_frame = 0;

oled_rotation_t oled_init_user(oled_rotation_t rotation) {
    return OLED_ROTATION_180;
}

void render_default_layer_state(void) {
    oled_write_P(PSTR("Layout: "), false);
    switch (get_highest_layer(default_layer_state)) {
        case _QWERTY:
            if(woke_mode) {
                oled_write_ln_P(PSTR("QwErTy"), false);
            } else {
                oled_write_ln_P(PSTR("Qwerty"), false);
            }
            break;
        case _COLMAK:
            oled_write_ln_P(PSTR("Colemak"), false);
            break;
    }
}

#ifdef RGBLIGHT_ENABLE
bool cleared = false;
void render_rgb_state(void) {
    if(!rgblight_get_mode()) {
        if(!cleared) {
            oled_clear();
            cleared = true;
        }
        return;
    }
    cleared = false;
    uint8_t width = (OLED_DISPLAY_WIDTH/OLED_FONT_WIDTH)-10;
    uint8_t hue = (rgblight_get_hue()*width/255),
        sat = (rgblight_get_sat()*width/255),
        val = (rgblight_get_val()*width/255);
    bool changing_hue = timer_elapsed(rgb_edit_timer) < 1000 && (last_rgb_char == RGB_HUI || last_rgb_char == RGB_HUD);
    bool changing_sat =  timer_elapsed(rgb_edit_timer) < 1000 && (last_rgb_char == RGB_SAI || last_rgb_char == RGB_SAD);
    bool changing_val =  timer_elapsed(rgb_edit_timer) < 1000 && (last_rgb_char == RGB_VAI || last_rgb_char == RGB_VAD);

    uint8_t i;

    oled_write_ln_P(PSTR(""), false);

    oled_write_P(PSTR("Hue:    ["), changing_hue);
    for(i = 0; i < width; ++i) {
        oled_write_P(i<hue?PSTR("-"):i==hue?PSTR("|"):PSTR(" "), changing_hue);
    }
    oled_write_P(PSTR("]"), changing_hue);
    oled_write_P(PSTR("Sat:    ["), changing_sat);
    for(i = 0; i < width; ++i) {
        oled_write_P(i<sat?PSTR("-"):i==sat?PSTR("|"):PSTR(" "), changing_sat);
    }
    oled_write_P(PSTR("]"), changing_sat);
    oled_write_P(PSTR("Bri:    ["), changing_val);
    for(i = 0; i < width; ++i) {
        oled_write_P(i<val?PSTR("-"):i==val?PSTR("|"):PSTR(" "), changing_val);
    }
    oled_write_P(PSTR("]"), changing_val);
}
#endif // RGBLIGHT_ENABLE

void render_layer_state(void) {
    oled_write_P(PSTR("Layer:  "), false);
    switch (get_highest_layer(layer_state)) {
        case _QWERTY:
            oled_write_ln_P(PSTR("Base"), false);
            break;
        case _NAV:
            oled_write_ln_P(PSTR("Navigation"), false);
            break;
        case _NUM:
            oled_write_ln_P(PSTR("Number"), false);
            break;
        case _SYM:
            oled_write_ln_P(PSTR("Symbol"), false);
            break;
        case _MISC:
            oled_write_ln_P(PSTR("Misc"), false);
            break;
        case _ADJUST:
            oled_write_ln_P(PSTR("Adjust"), false);
            break;
        default:
            oled_write_ln_P(PSTR("Unknown"), false);
    }
}

void render_keylock_status(led_t led_state) {
    oled_write_P(PSTR("Lock:   "), false);
    oled_write_P(PSTR("123"), led_state.num_lock);
    oled_write_P(PSTR(" "), false);
    oled_write_ln_P(led_state.caps_lock?PSTR("ABC"):PSTR("abc"), led_state.caps_lock);
}

void render_mod_status(uint8_t modifiers) {
    oled_write_P(PSTR("Mods:   "), false);
    oled_write_P(PSTR("S"), (modifiers & MOD_MASK_SHIFT));
    oled_write_P(PSTR(" "), false);
    oled_write_P(PSTR("C"), (modifiers & MOD_MASK_CTRL));
    oled_write_P(PSTR(" "), false);
    oled_write_P(PSTR("A"), (modifiers & MOD_MASK_ALT));
    oled_write_P(PSTR(" "), false);
    oled_write_ln_P(PSTR("W"), (modifiers & MOD_MASK_GUI));
}

static void render_status(void) {
    // Host Keyboard Layer Status
    render_default_layer_state();
    render_layer_state();
    render_keylock_status(host_keyboard_led_state());
    render_mod_status(get_mods() | get_oneshot_mods());
#ifdef RGBLIGHT_ENABLE
    render_rgb_state();
#endif // RGBLIGHT_ENABLE
}

static void render_logo(void) {
    static const char PROGMEM frame[ANIM_NUM_FRAMES][1024] = {
        {
            // leekspin: frame 0
            0xff, 0xbf, 0xfb, 0x6f, 0xff, 0xdb, 0x7f, 0xf7, 0xde, 0x7c, 0xf1, 0xc6, 0x98, 0x21, 0x66, 0x9a,
            0xc9, 0x24, 0x33, 0xcf, 0x2f, 0xfd, 0xff, 0x6b, 0x7f, 0x9d, 0x81, 0x76, 0x14, 0x81, 0x67, 0x0f,
            0xfb, 0xff, 0xef, 0xbb, 0xff, 0xef, 0xbb, 0xff, 0xef, 0xbb, 0xff, 0xef, 0xbb, 0xff, 0xef, 0xbb,
            0xff, 0x6f, 0xfb, 0xdf, 0x77, 0xff, 0x0d, 0xe0, 0x3f, 0xc0, 0x0d, 0x37, 0xd0, 0x26, 0xa9, 0xd8,
            0xc7, 0x1d, 0xf0, 0xff, 0xff, 0xf0, 0xc7, 0xb1, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xf1, 0xc6,
            0xff, 0xf7, 0xf8, 0xf3, 0xcf, 0x90, 0xf7, 0xfc, 0xe3, 0x9b, 0x2e, 0xe2, 0xfc, 0x03, 0xfc, 0x7c,
            0x83, 0x1b, 0x6c, 0xa5, 0xb3, 0x4e, 0x59, 0x64, 0x13, 0xdb, 0xae, 0x20, 0xdf, 0xff, 0xbb, 0xef,
            0xff, 0xbb, 0xef, 0xdf, 0x7b, 0xef, 0xbf, 0xfb, 0x6f, 0xff, 0xdb, 0x7f, 0xf7, 0xdf, 0x7b, 0xef,
            0xff, 0xff, 0xdd, 0x77, 0xff, 0xdd, 0x77, 0xff, 0xdd, 0x77, 0xff, 0xdd, 0x77, 0xfe, 0xdc, 0xf1,
            0xcc, 0x23, 0x99, 0x64, 0x26, 0x98, 0x63, 0x0f, 0x30, 0xcc, 0x33, 0x24, 0xcd, 0x19, 0xe0, 0xff,
            0xff, 0xde, 0x7b, 0xef, 0xbe, 0xfb, 0xef, 0xbe, 0xfb, 0x6f, 0xfe, 0xdb, 0x7f, 0xf6, 0xdf, 0xfd,
            0xb7, 0xff, 0x6d, 0xff, 0xdb, 0xff, 0x00, 0xbd, 0x67, 0x48, 0x13, 0xfc, 0x85, 0xff, 0x5f, 0x3f,
            0xb7, 0x97, 0xcf, 0xcb, 0xeb, 0xcf, 0x97, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xef, 0x9f, 0xd7, 0xcf, 0xcb, 0xdf, 0xd7, 0xcf, 0x8f, 0x3f, 0x1f, 0x7f, 0xff, 0xfe, 0xff, 0xf2,
            0x07, 0xe9, 0x3e, 0x00, 0x35, 0xef, 0xc9, 0x1b, 0xc2, 0xbc, 0x37, 0xc0, 0xff, 0xbf, 0xfb, 0x6e,
            0xef, 0xfb, 0xde, 0xf7, 0xb7, 0xfd, 0xef, 0xbd, 0xf7, 0x77, 0xdd, 0xff, 0xb7, 0xfd, 0xef, 0xbb,
            0xff, 0xff, 0xdd, 0xf7, 0x7f, 0xdd, 0xf7, 0x7f, 0xdd, 0xf7, 0x7f, 0xdd, 0xf7, 0xbf, 0xed, 0xff,
            0xdf, 0xf7, 0xf6, 0x7c, 0xf9, 0xb1, 0xe6, 0xcb, 0x18, 0xa4, 0xb3, 0x09, 0xfc, 0xff, 0xfb, 0x6f,
            0xfe, 0xdf, 0xfb, 0xbf, 0xf7, 0x7e, 0xef, 0xfd, 0xb7, 0xff, 0x6d, 0xff, 0xdb, 0xff, 0xb6, 0xff,
            0x6d, 0xff, 0xdb, 0x7f, 0xf6, 0xdf, 0xf9, 0x21, 0x8e, 0xfb, 0x00, 0x7e, 0xc0, 0xff, 0xfe, 0xff,
            0xf7, 0xff, 0xff, 0xff, 0x3f, 0x3f, 0x7f, 0x7f, 0x3f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f,
            0x3f, 0x7f, 0x7f, 0x3f, 0x3f, 0x7f, 0x37, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xef, 0x3f, 0xc0,
            0x6e, 0x3b, 0x09, 0xe0, 0x6f, 0x1a, 0x92, 0x64, 0x6d, 0x99, 0xf2, 0x7f, 0xef, 0xfd, 0xb7, 0xff,
            0x7b, 0xde, 0xf7, 0xbf, 0xfb, 0x7e, 0xef, 0xbf, 0xff, 0xfb, 0xef, 0xbe, 0xfb, 0x6f, 0xfe, 0xdb,
            0xff, 0xff, 0xfe, 0xf7, 0xbf, 0xfb, 0xff, 0xbf, 0xef, 0xfd, 0xdf, 0xf7, 0x7d, 0xff, 0xee, 0x7f,
            0xf7, 0xbd, 0xff, 0xef, 0xfb, 0xbf, 0xfd, 0x77, 0xfe, 0xa0, 0xf9, 0xfe, 0xf0, 0xef, 0xbf, 0xf7,
            0x7f, 0xfb, 0xde, 0xff, 0xbb, 0xff, 0xf7, 0xbe, 0xfb, 0xdf, 0xff, 0xfd, 0xb7, 0xfe, 0xef, 0x7f,
            0xfb, 0xbf, 0xed, 0xff, 0xf7, 0xbe, 0xfb, 0xdf, 0xf0, 0x06, 0x98, 0xe7, 0x3c, 0x03, 0xcf, 0x2f,
            0x1f, 0x9f, 0x7f, 0x7f, 0x7e, 0xf8, 0xf1, 0xe6, 0xf8, 0xc9, 0xca, 0xfa, 0xc8, 0xda, 0xf2, 0xcc,
            0xd9, 0xf2, 0xe0, 0xed, 0xf1, 0xfa, 0xfc, 0x7f, 0x7f, 0x3f, 0xbf, 0x9f, 0x0f, 0xe3, 0x34, 0x8d,
            0xe3, 0x18, 0x17, 0x85, 0xa0, 0x38, 0xcf, 0x13, 0xf8, 0xef, 0xdd, 0x7f, 0xff, 0xf7, 0xfd, 0xef,
            0xbb, 0xff, 0x7f, 0xf7, 0xfd, 0xef, 0xbf, 0xfb, 0xee, 0xff, 0xff, 0xbb, 0xef, 0x7f, 0xfb, 0xee,
            0xff, 0xff, 0xde, 0xf7, 0xff, 0xef, 0xbd, 0xff, 0xdb, 0xff, 0xfe, 0x77, 0x7f, 0xf7, 0xfd, 0x7f,
            0xf7, 0xff, 0xdb, 0x7e, 0xff, 0xef, 0xfb, 0xff, 0xb7, 0xff, 0xc8, 0xff, 0xff, 0xff, 0xfc, 0x27,
            0xff, 0xff, 0xff, 0xed, 0xbf, 0xfe, 0xf7, 0x7f, 0xfb, 0xdf, 0xfe, 0xf7, 0xdf, 0xff, 0xfd, 0x6f,
            0xff, 0xbf, 0xfb, 0xff, 0xde, 0xff, 0xf7, 0xfd, 0xbf, 0x00, 0x77, 0x6c, 0x89, 0x11, 0x04, 0xa2,
            0x08, 0x69, 0x46, 0x90, 0x02, 0x64, 0x94, 0x24, 0x09, 0x91, 0x85, 0x21, 0x09, 0xff, 0xf7, 0xdd,
            0x3d, 0x37, 0x1d, 0x1d, 0x06, 0xc6, 0x07, 0x10, 0x20, 0x83, 0x84, 0x30, 0x66, 0x19, 0x92, 0x06,
            0x40, 0x31, 0x0c, 0xc0, 0x14, 0xb7, 0x09, 0xfa, 0xff, 0xdf, 0xfb, 0xbf, 0xfd, 0xf7, 0xef, 0xef,
            0xff, 0xdf, 0xf7, 0xff, 0x6e, 0xff, 0xff, 0xde, 0xf7, 0xff, 0x7e, 0xdb, 0xff, 0xb7, 0xff, 0xee,
            0xff, 0xff, 0x7f, 0xf6, 0xdf, 0xfd, 0xff, 0xff, 0xef, 0xbb, 0xfe, 0x7f, 0xf7, 0xdf, 0xff, 0xf7,
            0xdd, 0xff, 0xff, 0xbf, 0xed, 0xff, 0xfe, 0xef, 0x3f, 0xde, 0xee, 0xc9, 0xff, 0xff, 0xdf, 0xff,
            0xc9, 0xff, 0x9e, 0xdf, 0xdf, 0xb7, 0xfd, 0xff, 0xef, 0xff, 0xfb, 0xef, 0xfd, 0xbf, 0xf7, 0xff,
            0xbb, 0xef, 0xff, 0xfd, 0xff, 0xef, 0x7e, 0xff, 0xb1, 0x06, 0x7b, 0xc9, 0x00, 0x42, 0x98, 0x31,
            0x46, 0x44, 0x38, 0x89, 0xc4, 0x30, 0x00, 0x42, 0x10, 0x84, 0x20, 0x08, 0x81, 0x03, 0x10, 0x80,
            0x24, 0x00, 0x88, 0x21, 0x04, 0xc0, 0x08, 0xa1, 0x34, 0x4c, 0x03, 0x90, 0x46, 0x00, 0x30, 0xd1,
            0x04, 0x69, 0x88, 0x30, 0x22, 0xcb, 0x04, 0x3f, 0xfb, 0xff, 0xfd, 0x6f, 0xfe, 0xf7, 0xbf, 0xfe,
            0xdb, 0xff, 0xee, 0x7f, 0xf7, 0xbf, 0xfd, 0xef, 0x7e, 0xf7, 0xff, 0xdb, 0xff, 0xb7, 0xfe, 0x7b,
            0xff, 0xff, 0xff, 0xbb, 0xef, 0xff, 0xf6, 0xff, 0xbf, 0xef, 0xff, 0xdf, 0xfd, 0x7f, 0xfe, 0xbf,
            0xef, 0xfe, 0xff, 0xdf, 0xff, 0xfe, 0xff, 0xe4, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xfe, 0xfb, 0x37, 0xfd, 0xfe, 0xff, 0xff, 0x6f, 0xff, 0xf6, 0xff, 0xbf, 0xfb, 0xdf,
            0xff, 0xff, 0x77, 0x7f, 0x9d, 0x9f, 0x27, 0x21, 0xcc, 0x18, 0x63, 0x84, 0x80, 0x2a, 0x29, 0x44,
            0x59, 0x43, 0x94, 0x34, 0x42, 0x49, 0x2c, 0x92, 0x92, 0x64, 0x0c, 0x80, 0xb8, 0x22, 0x86, 0x34,
            0x48, 0x23, 0x34, 0x44, 0xc9, 0x18, 0x46, 0x09, 0x80, 0x72, 0x08, 0x61, 0x80, 0x9b, 0x12, 0x64,
            0x8d, 0x90, 0x23, 0xa8, 0x26, 0x44, 0x59, 0x89, 0x24, 0xd3, 0x1f, 0xff, 0xfd, 0xef, 0xff, 0x7d,
            0xef, 0xfe, 0xf7, 0xbf, 0xfb, 0xdf, 0x7d, 0xff, 0xf7, 0xbf, 0xfe, 0xdb, 0x7f, 0xfd, 0xff, 0xcf,
            0xff, 0xff, 0xfb, 0xff, 0x7f, 0xff, 0xb6, 0xff, 0xff, 0xff, 0xfd, 0xb7, 0xff, 0xfb, 0xbf, 0xff,
            0xef, 0xfb, 0x7e, 0xdf, 0xff, 0xf6, 0xff, 0xdf, 0xff, 0xed, 0xfb, 0xff, 0xef, 0x2f, 0xef, 0xef,
            0xf7, 0xf7, 0x9f, 0xfb, 0xfc, 0xff, 0xf7, 0xdf, 0xbb, 0x7f, 0xff, 0xef, 0xfe, 0xf7, 0x1f, 0x07,
            0xd1, 0x19, 0x22, 0x66, 0x44, 0x98, 0x8b, 0x21, 0x64, 0x1a, 0x42, 0xd4, 0x14, 0x23, 0xa8, 0x4d,
            0x91, 0x92, 0x64, 0x25, 0x99, 0x44, 0x23, 0x88, 0x50, 0x96, 0xa3, 0x18, 0x44, 0x65, 0x99, 0x22,
            0x8a, 0x69, 0x25, 0x86, 0xd8, 0x1d, 0xb7, 0x9a, 0x9e, 0x30, 0x8b, 0xe2, 0xcc, 0xd9, 0xe1, 0xfc,
            0xee, 0xf8, 0xfe, 0xe1, 0xc9, 0x06, 0x98, 0x69, 0x22, 0x92, 0x4c, 0x23, 0xa7, 0x1f, 0x7d, 0xff,
            0xff, 0xbf, 0xef, 0xfb, 0xbe, 0xef, 0x7f, 0xfb, 0xfe, 0x77, 0xff, 0xbf, 0xef, 0xfd, 0xf7, 0xff,

        },
        {
            // leekspin: frame 1
            0xff, 0xbf, 0xfb, 0xcf, 0xff, 0xbb, 0xef, 0xff, 0xbb, 0xef, 0xff, 0xbb, 0xef, 0xff, 0xff, 0xdf,
            0x77, 0xff, 0xdb, 0xff, 0xb7, 0xff, 0xfb, 0x6f, 0xff, 0xff, 0xdd, 0xf7, 0xff, 0xff, 0xff, 0xbb,
            0xef, 0xff, 0xbb, 0xef, 0xff, 0xbb, 0xef, 0xff, 0xbb, 0xef, 0xff, 0xbb, 0x6f, 0xff, 0xbb, 0xef,
            0xff, 0xbb, 0xef, 0x7f, 0xdb, 0xff, 0x27, 0xd0, 0x5f, 0x60, 0x8d, 0x37, 0xe0, 0x8d, 0x31, 0xd8,
            0xc7, 0x1d, 0xf0, 0xff, 0xff, 0xf0, 0xcf, 0xe0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xf9, 0xc1,
            0xff, 0xfe, 0xf9, 0xe7, 0xcc, 0x91, 0xf7, 0xfc, 0xe3, 0x99, 0x2e, 0xe2, 0xfc, 0x03, 0xfc, 0x7c,
            0x03, 0x3b, 0xec, 0x85, 0x33, 0x76, 0xcd, 0xa8, 0x23, 0x5b, 0xee, 0x20, 0x9f, 0xff, 0x7b, 0xef,
            0xbf, 0xfb, 0xef, 0xbf, 0xfb, 0xef, 0xbf, 0xfb, 0xef, 0xbf, 0xfb, 0x6f, 0xff, 0xdb, 0xff, 0xb7,
            0xff, 0xff, 0x76, 0xdf, 0xfd, 0x77, 0xde, 0xfb, 0xbf, 0xf6, 0xdf, 0xfd, 0xb7, 0xff, 0x6d, 0xff,
            0xfe, 0xfb, 0xdf, 0xf6, 0xbf, 0xed, 0xff, 0xef, 0xbf, 0xf6, 0xff, 0xef, 0xfb, 0xbf, 0xee, 0xfb,
            0xbf, 0xee, 0xfb, 0xbf, 0xee, 0xfb, 0xbf, 0xee, 0xfb, 0xbf, 0xee, 0x7b, 0xdf, 0xfd, 0xb7, 0xfe,
            0xef, 0xbd, 0xf7, 0x7f, 0xed, 0xff, 0x00, 0x6e, 0xdb, 0x91, 0x24, 0xee, 0x09, 0xff, 0x5f, 0x3f,
            0xa7, 0x9f, 0xcb, 0xcb, 0xef, 0xcb, 0x9f, 0xb7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0x8f, 0xd7, 0xdf, 0xcb, 0xcf, 0xd7, 0xdf, 0x8f, 0x3f, 0x1f, 0x7f, 0xff, 0xff, 0xff, 0xf0,
            0x07, 0xb3, 0xec, 0x08, 0x33, 0xf7, 0x8c, 0x66, 0x11, 0xdb, 0x2e, 0xa0, 0xff, 0xdf, 0xf7, 0xbd,
            0xef, 0xfe, 0xbb, 0xef, 0x7e, 0xdb, 0xff, 0xb6, 0xff, 0x6d, 0xff, 0xdb, 0xff, 0xb6, 0xff, 0x6d,
            0xff, 0xff, 0xb7, 0xfd, 0xef, 0xbb, 0xff, 0x6e, 0xfb, 0xdf, 0x76, 0xff, 0xdd, 0x77, 0xff, 0xef,
            0xbf, 0xfb, 0xf6, 0x7f, 0xef, 0xff, 0x7f, 0xed, 0xfd, 0xf7, 0xbf, 0xff, 0xde, 0xf7, 0xfd, 0xbf,
            0xf7, 0xde, 0xfb, 0xbf, 0xee, 0xfb, 0xdf, 0x76, 0xff, 0xed, 0xbf, 0xfb, 0x6f, 0xfe, 0xdb, 0xff,
            0xb6, 0xff, 0xed, 0xbf, 0xfb, 0x6f, 0xfc, 0x83, 0x32, 0xde, 0x00, 0xbf, 0xc0, 0xff, 0xfe, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xef, 0x3f, 0xc0,
            0xde, 0x33, 0x0c, 0xe0, 0xb7, 0x1c, 0x85, 0x70, 0x6f, 0x8d, 0xf0, 0x7f, 0xdf, 0xf7, 0xfe, 0xbb,
            0xef, 0x7f, 0xd9, 0xff, 0xbf, 0xf7, 0xff, 0x6f, 0xff, 0xfb, 0xff, 0xb6, 0xff, 0x6d, 0xff, 0xdb,
            0xff, 0xff, 0xff, 0xed, 0xbf, 0xfb, 0xff, 0xdf, 0xf7, 0xfd, 0x7f, 0xef, 0xbb, 0xff, 0xdd, 0xff,
            0x77, 0xfe, 0xef, 0xbb, 0xff, 0x7d, 0xef, 0xff, 0xb6, 0xff, 0xdd, 0xff, 0x7b, 0xde, 0xff, 0xed,
            0x7f, 0xfb, 0xde, 0xff, 0xb7, 0xfd, 0xff, 0xdb, 0x7f, 0xfe, 0xbb, 0xef, 0x7f, 0xdb, 0xfe, 0xef,
            0x7b, 0xfe, 0xef, 0x7d, 0xf7, 0xbf, 0xed, 0x7f, 0xf8, 0x02, 0xc6, 0xbd, 0x34, 0x47, 0x47, 0x9f,
            0x1f, 0x3f, 0x3f, 0x7f, 0x7e, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xfd, 0xff, 0xfe, 0xff, 0xfd, 0xff,
            0xfe, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x7f, 0xff, 0x3f, 0x7f, 0x9f, 0x9f, 0x0d, 0xe3, 0x3a, 0x8c,
            0x63, 0x58, 0x17, 0x84, 0x21, 0xb8, 0xcf, 0x63, 0xd8, 0xff, 0xef, 0xbb, 0xff, 0xff, 0xfd, 0xf7,
            0xdd, 0xff, 0xbf, 0x7b, 0xff, 0xf6, 0xdf, 0xff, 0xb7, 0xfd, 0xff, 0xef, 0x7d, 0xdf, 0xfb, 0xee,
            0xff, 0xff, 0xbe, 0xef, 0xfb, 0x7f, 0xef, 0xbe, 0xfb, 0xff, 0x77, 0xff, 0xef, 0x7d, 0xff, 0xfb,
            0xff, 0xed, 0xff, 0xdf, 0x7d, 0xef, 0xff, 0xf6, 0x7f, 0xdf, 0xfd, 0xb7, 0xff, 0xfb, 0xdf, 0xfd,
            0xff, 0xb7, 0xff, 0xed, 0x7f, 0xfe, 0xb7, 0xff, 0xfb, 0xdf, 0xff, 0xf7, 0xff, 0xef, 0xff, 0x7d,
            0xef, 0xff, 0xb7, 0xff, 0xfd, 0xff, 0xdf, 0x77, 0xff, 0x80, 0x36, 0xed, 0x09, 0x12, 0x44, 0x21,
            0x0c, 0x51, 0x51, 0x86, 0x00, 0x58, 0x46, 0x30, 0x24, 0x85, 0x09, 0x21, 0x85, 0xff, 0x7b, 0xed,
            0x3d, 0x1d, 0x1d, 0x47, 0x0e, 0x8e, 0x23, 0x00, 0x22, 0x02, 0xc4, 0xb0, 0x26, 0x19, 0x4c, 0x03,
            0xc0, 0x18, 0x06, 0x60, 0x89, 0xbd, 0x02, 0xfa, 0xff, 0xff, 0xb6, 0xff, 0xfb, 0xbe, 0xf7, 0xf7,
            0xdf, 0x7f, 0xf7, 0xff, 0x6f, 0xfe, 0xff, 0xdf, 0xfb, 0x6f, 0xfe, 0xdf, 0x7b, 0xff, 0xf7, 0xde,
            0x23, 0xcf, 0x9f, 0x37, 0x7d, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0x6f, 0xff, 0xdb, 0xff, 0xbf, 0xfe,
            0xdb, 0xef, 0xff, 0xff, 0xb7, 0xfd, 0x7f, 0xff, 0xef, 0xfb, 0x7f, 0xf7, 0xbd, 0xff, 0xde, 0xdf,
            0xfb, 0xdf, 0xde, 0xd7, 0xbf, 0xef, 0xfb, 0xf7, 0xfe, 0xfb, 0xfb, 0xfd, 0x7c, 0x7f, 0xfd, 0xfe,
            0x7d, 0xfd, 0x9e, 0x62, 0x6f, 0xff, 0xff, 0x7f, 0xd2, 0x08, 0xef, 0x31, 0x00, 0x42, 0x48, 0x39,
            0x86, 0x60, 0x0c, 0xd9, 0x90, 0x20, 0x02, 0x10, 0x44, 0x08, 0x02, 0x10, 0x40, 0x03, 0x11, 0x80,
            0x20, 0x24, 0x81, 0x08, 0x42, 0x90, 0x04, 0xc1, 0x78, 0x06, 0x24, 0x09, 0x42, 0x40, 0x98, 0x22,
            0x30, 0xc9, 0x88, 0x32, 0x44, 0x4d, 0x80, 0x3f, 0xff, 0xfe, 0xf7, 0x7f, 0xed, 0xbf, 0xfe, 0xf7,
            0xbf, 0xfb, 0xdf, 0xfd, 0xef, 0x7f, 0xfb, 0xde, 0xff, 0xef, 0xfb, 0xbf, 0xff, 0x6d, 0xff, 0xf6,
            0x92, 0x24, 0x65, 0x99, 0xc6, 0x30, 0x8b, 0x67, 0x37, 0x87, 0x6f, 0xff, 0xf5, 0x77, 0x7f, 0xfe,
            0xff, 0xef, 0xfe, 0x7f, 0xf7, 0xed, 0x0f, 0xfe, 0xfe, 0xfb, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0x7f, 0xb6, 0xec, 0xfe, 0xff, 0x7e, 0xef, 0xff, 0x7f, 0xf7, 0xff, 0xbf,
            0xef, 0xff, 0xf7, 0x7f, 0x1f, 0x9f, 0x63, 0x03, 0xdc, 0x90, 0x23, 0x8a, 0x20, 0x05, 0xc9, 0x5a,
            0x11, 0xa4, 0x8b, 0x30, 0x24, 0x4b, 0x48, 0x26, 0x34, 0xc1, 0x8c, 0x22, 0x98, 0x91, 0x66, 0x08,
            0x32, 0xc4, 0x99, 0x12, 0x62, 0xcc, 0x00, 0x33, 0x04, 0xc0, 0x38, 0x83, 0x40, 0x96, 0x19, 0x61,
            0x86, 0x18, 0x60, 0x93, 0x4c, 0x24, 0x93, 0x48, 0x6c, 0x83, 0x3f, 0xff, 0xfb, 0xff, 0xdd, 0xff,
            0xf7, 0xfd, 0xff, 0xde, 0xf7, 0x7f, 0xfb, 0xbf, 0xee, 0xff, 0xbd, 0xf7, 0x7f, 0xfb, 0xff, 0xde,
            0xfe, 0xff, 0xde, 0xfe, 0xfe, 0xfe, 0x6f, 0xfe, 0xff, 0xff, 0xff, 0xb7, 0xff, 0xdf, 0x7f, 0xff,
            0xd7, 0xfd, 0xef, 0xff, 0x7d, 0xff, 0xdf, 0xff, 0xf5, 0x7f, 0xf7, 0xff, 0xef, 0xef, 0xe7, 0x7f,
            0xfb, 0xfb, 0xfe, 0xfd, 0xff, 0xfb, 0xef, 0xfd, 0xdf, 0x7f, 0xff, 0xfb, 0xdf, 0xff, 0x0b, 0x07,
            0x35, 0xc9, 0x48, 0x92, 0x93, 0x6c, 0x08, 0x93, 0xc4, 0x30, 0x46, 0x99, 0xa1, 0x26, 0x48, 0x8c,
            0x33, 0xc4, 0x28, 0x0b, 0xd1, 0x34, 0x01, 0xc3, 0x18, 0x64, 0x86, 0x30, 0x4d, 0x90, 0xa6, 0x29,
            0x49, 0x44, 0xb4, 0x83, 0x5a, 0x9c, 0x37, 0x56, 0x9e, 0x21, 0xcc, 0xd1, 0xd6, 0xe4, 0xc9, 0xfc,
            0xec, 0xef, 0xf8, 0xe4, 0xc3, 0x19, 0xa4, 0x26, 0xd0, 0x0d, 0x24, 0xd3, 0x47, 0x1f, 0x7e, 0xff,
            0xff, 0x6f, 0xff, 0xd9, 0xff, 0xbf, 0xe7, 0xff, 0xf9, 0x7f, 0xff, 0xdf, 0xdf, 0xf5, 0xff, 0xfb,

        },
        {
            // leekspin: frame 2
            0xff, 0xbf, 0xfb, 0xef, 0xbf, 0xfb, 0xef, 0xbf, 0xfb, 0xef, 0xbf, 0xfb, 0x6f, 0xff, 0xff, 0xef,
            0xbf, 0xfb, 0xdf, 0xf7, 0xbf, 0xfb, 0x7f, 0x7f, 0xdb, 0xff, 0xbf, 0xfb, 0xef, 0xff, 0x7f, 0xdf,
            0xff, 0xf3, 0xbf, 0xef, 0xfb, 0xbf, 0xef, 0xfb, 0xbf, 0xef, 0xfb, 0xbf, 0xef, 0xfb, 0xbf, 0xef,
            0xfb, 0xbf, 0xef, 0x7b, 0xdf, 0xf7, 0x2f, 0x60, 0xdf, 0x80, 0x2d, 0x57, 0xd0, 0x93, 0x24, 0xfc,
            0xc3, 0x1b, 0xf4, 0xf7, 0xff, 0xf0, 0xcf, 0xb0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xf9, 0xc1,
            0xff, 0xfe, 0xf9, 0xe7, 0xcc, 0x91, 0xf7, 0xfc, 0xe3, 0x99, 0x2e, 0xe2, 0xfc, 0x03, 0xec, 0x7c,
            0x93, 0x2b, 0x6c, 0x95, 0x53, 0xce, 0x39, 0xc4, 0x33, 0x6b, 0xce, 0x90, 0xbf, 0xef, 0xfb, 0xbf,
            0xef, 0xfb, 0xbf, 0xef, 0xfb, 0xbf, 0xef, 0xfb, 0xbf, 0xef, 0xfb, 0xbf, 0xef, 0xfb, 0xbf, 0xed,
            0xff, 0xff, 0xee, 0xbb, 0xff, 0x6e, 0xfb, 0xdf, 0x76, 0xff, 0xdd, 0xf7, 0xbf, 0xed, 0xff, 0xff,
            0xee, 0xff, 0x75, 0xdf, 0xff, 0xf6, 0xdf, 0xff, 0xb7, 0xff, 0xfb, 0xef, 0xfd, 0xb7, 0xff, 0xdb,
            0xfe, 0x6f, 0xfb, 0xbe, 0xef, 0xfb, 0xbe, 0xef, 0xfb, 0xbe, 0xef, 0xfb, 0xbe, 0xef, 0x7b, 0xde,
            0xf7, 0x7d, 0xdf, 0xf7, 0xbd, 0xff, 0x00, 0xf7, 0x9c, 0x21, 0x27, 0xd8, 0x46, 0xff, 0x5f, 0x3f,
            0xa7, 0x9f, 0xcb, 0xcb, 0xef, 0xcb, 0xdf, 0x9f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xef, 0x9f, 0xd7, 0xcf, 0xcb, 0xdf, 0xd7, 0xcf, 0x8f, 0x3f, 0x1f, 0x7f, 0xff, 0xff, 0xff, 0xf0,
            0x07, 0x73, 0xdc, 0x01, 0x6e, 0x99, 0x73, 0x4e, 0x88, 0xf3, 0x1e, 0xc0, 0xff, 0x7f, 0xde, 0xf3,
            0xbf, 0xee, 0xfb, 0xbf, 0xee, 0xfb, 0xbf, 0xee, 0xfb, 0xbf, 0xee, 0x7b, 0xdf, 0xf6, 0xbf, 0xed,
            0xff, 0xff, 0x76, 0xdf, 0xfd, 0xb7, 0xff, 0xed, 0xbf, 0xfb, 0x6e, 0xff, 0xbd, 0xef, 0xfe, 0xf7,
            0xbf, 0xfd, 0xf7, 0xdf, 0x7f, 0xff, 0xb7, 0xfc, 0xef, 0xff, 0xf7, 0xbf, 0xfa, 0xdf, 0xfd, 0xef,
            0x7b, 0xff, 0xed, 0x7f, 0xde, 0xfb, 0xef, 0xbe, 0xfb, 0xdf, 0xf6, 0x7f, 0xdd, 0xf7, 0xbf, 0xed,
            0xff, 0xbb, 0xef, 0x7e, 0xdb, 0xff, 0x76, 0xc0, 0x1d, 0x77, 0x00, 0x7f, 0xc0, 0xff, 0xfe, 0xff,
            0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xef, 0x3f, 0xc0,
            0xbb, 0x37, 0x04, 0xf0, 0xce, 0x39, 0x07, 0x70, 0xdd, 0x86, 0xf2, 0x7f, 0xdf, 0xf7, 0xfd, 0xbf,
            0xef, 0x79, 0xff, 0xef, 0xfd, 0xdf, 0x7b, 0xef, 0xbf, 0xfd, 0xff, 0xfb, 0xdf, 0xf6, 0xbf, 0xed,
            0xff, 0xff, 0xef, 0x7d, 0xff, 0xfb, 0xdf, 0xfd, 0xff, 0x6f, 0xfb, 0xbf, 0xfd, 0xef, 0x7e, 0xf7,
            0xbf, 0xee, 0xfb, 0x7f, 0xdf, 0xfb, 0xbf, 0xfd, 0xef, 0xfe, 0xdf, 0xfb, 0xef, 0xfe, 0xbf, 0xed,
            0xff, 0xf7, 0xbd, 0xff, 0x6f, 0xfb, 0xfe, 0xdf, 0xf7, 0xfd, 0xbf, 0xfb, 0x6f, 0xfe, 0xf7, 0xff,
            0xdd, 0x7f, 0xfb, 0xef, 0xfd, 0xdf, 0x7b, 0xff, 0xe4, 0x09, 0xc9, 0xb6, 0x36, 0x4b, 0x47, 0x9f,
            0x1f, 0x3f, 0x3f, 0x7f, 0x7f, 0xff, 0xfe, 0xfe, 0xff, 0xfd, 0xff, 0xfe, 0xfd, 0xff, 0xfd, 0xff,
            0xfe, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x7f, 0xff, 0x3f, 0x3f, 0x9f, 0xdf, 0x0d, 0xa3, 0x3a, 0x4c,
            0xe3, 0x18, 0x17, 0x06, 0x60, 0xd8, 0xcf, 0x33, 0xd8, 0xfe, 0xef, 0x7b, 0xdf, 0xff, 0x7f, 0xfd,
            0xfd, 0xf7, 0xdf, 0xff, 0xbe, 0xfb, 0xff, 0xb7, 0xdf, 0xfd, 0xff, 0x6e, 0xff, 0xdd, 0xf7, 0xbe,
            0xff, 0xff, 0xdd, 0xff, 0xef, 0x7b, 0xfe, 0xbf, 0xef, 0x7b, 0xff, 0xbd, 0x37, 0xff, 0xfb, 0x7f,
            0xf7, 0xfd, 0xef, 0xff, 0xbb, 0xff, 0xf7, 0xfe, 0xdb, 0xff, 0xee, 0xff, 0xfb, 0xbe, 0xff, 0xef,
            0xfb, 0x7e, 0xdf, 0xf7, 0xff, 0xbd, 0xff, 0xee, 0xff, 0xdb, 0xff, 0xf7, 0x7f, 0xff, 0xf7, 0xbe,
            0xff, 0x77, 0xff, 0xbb, 0xfe, 0xff, 0xff, 0xb7, 0xff, 0x00, 0xb6, 0xed, 0x09, 0x02, 0x64, 0x81,
            0x0c, 0x31, 0xc5, 0x48, 0x02, 0x24, 0x98, 0x62, 0x24, 0x05, 0x89, 0x21, 0x05, 0xff, 0xfb, 0x6d,
            0x3d, 0x9d, 0x17, 0x1c, 0x4d, 0x07, 0x26, 0x80, 0x11, 0x41, 0x84, 0xb0, 0x66, 0x19, 0x0b, 0x82,
            0x10, 0x64, 0x09, 0x90, 0xc6, 0x3b, 0x08, 0xf6, 0xff, 0xbf, 0xfb, 0xef, 0xff, 0xdd, 0xff, 0xbb,
            0xff, 0x77, 0xf7, 0xbf, 0xfb, 0xfe, 0xb7, 0xff, 0xdf, 0xfd, 0xef, 0x7f, 0xdd, 0xf7, 0xff, 0x6d,
            0xff, 0xff, 0x7b, 0xfe, 0xdf, 0xf7, 0xf7, 0x7d, 0xe9, 0xbe, 0xfe, 0xff, 0xff, 0xfd, 0xff, 0xf7,
            0xfd, 0xcf, 0xff, 0x7f, 0xef, 0xbe, 0xfb, 0xdf, 0xbd, 0xbf, 0xbb, 0xbe, 0xff, 0xbf, 0xef, 0x7d,
            0x77, 0xff, 0xff, 0xdb, 0xfe, 0xff, 0xef, 0x7b, 0xfe, 0xff, 0xbf, 0xfd, 0xef, 0xff, 0xfb, 0xef,
            0xff, 0x7b, 0xff, 0xff, 0xfe, 0xbf, 0xef, 0xff, 0xd8, 0x03, 0xbd, 0x64, 0x00, 0x89, 0xa8, 0x22,
            0x1c, 0xe1, 0x08, 0x56, 0x90, 0x20, 0x04, 0x90, 0x02, 0x20, 0x04, 0x10, 0x41, 0x03, 0x88, 0x20,
            0x04, 0x40, 0x08, 0x81, 0x30, 0x02, 0x10, 0xc4, 0x30, 0x8e, 0x21, 0x08, 0x22, 0xc1, 0x18, 0xc0,
            0x31, 0x0c, 0x60, 0x98, 0xc2, 0x23, 0x0c, 0x5f, 0xff, 0xfb, 0xef, 0xfe, 0x77, 0xfd, 0xff, 0xdb,
            0xff, 0x6f, 0xfb, 0xbf, 0xfd, 0xef, 0xff, 0xdb, 0xfe, 0x7f, 0xf7, 0xdf, 0xfd, 0xef, 0xfb, 0xbf,
            0xff, 0xff, 0xff, 0xf7, 0xff, 0xdd, 0xff, 0xef, 0xff, 0x7b, 0xff, 0xde, 0xff, 0xff, 0x7b, 0xff,
            0xf7, 0xff, 0xef, 0xef, 0xde, 0x73, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xfe, 0xfe, 0xff, 0x89, 0xbf, 0xff, 0xbf, 0xf7, 0x7f, 0x7d, 0xff, 0xef, 0x7e, 0xff, 0xfb,
            0xdf, 0xff, 0xef, 0x7d, 0x1f, 0x97, 0xa7, 0x21, 0x1c, 0xd0, 0x23, 0x0a, 0x40, 0x14, 0xa4, 0x93,
            0x99, 0x24, 0x62, 0x8a, 0x19, 0x51, 0x46, 0xb0, 0x8c, 0x43, 0x18, 0xc4, 0x30, 0x26, 0xc4, 0x18,
            0xc9, 0x26, 0x98, 0x41, 0x4c, 0xb3, 0x80, 0x0d, 0x02, 0x70, 0x88, 0x23, 0xc4, 0x10, 0x96, 0x64,
            0x09, 0x93, 0x24, 0x68, 0x44, 0x95, 0x89, 0x6a, 0x10, 0xc7, 0x3f, 0xfb, 0xff, 0xff, 0x6e, 0xfb,
            0xbf, 0xff, 0xfb, 0xef, 0xff, 0x76, 0xff, 0x9f, 0xfd, 0xdf, 0xff, 0xbb, 0xee, 0xff, 0x3d, 0xff,
            0xff, 0xff, 0xde, 0xfb, 0xff, 0xff, 0xef, 0xbe, 0xff, 0xff, 0xff, 0x77, 0xfd, 0xff, 0xdf, 0xff,
            0xfb, 0x7e, 0xdf, 0xf7, 0xff, 0xff, 0xfe, 0xdf, 0xfd, 0xfb, 0xfb, 0xf7, 0xdf, 0xd9, 0xbf, 0xbf,
            0xd7, 0xf7, 0xfb, 0xfc, 0xff, 0xfb, 0xfb, 0xef, 0xb7, 0x7f, 0xff, 0xf7, 0xff, 0xee, 0x0f, 0x0e,
            0xae, 0x6f, 0x8c, 0x90, 0x53, 0x8c, 0x28, 0x23, 0x99, 0x48, 0x66, 0x91, 0x19, 0x66, 0x48, 0x92,
            0x94, 0x65, 0x24, 0x89, 0x69, 0x46, 0x12, 0xc4, 0x88, 0x32, 0x93, 0x4c, 0x48, 0x33, 0x84, 0x32,
            0xca, 0x0c, 0xb1, 0x22, 0xce, 0x9c, 0x33, 0x96, 0xdf, 0x18, 0xa1, 0xe6, 0xd8, 0xc6, 0xf1, 0xf8,
            0xee, 0xfd, 0xf9, 0xf2, 0xc2, 0x0c, 0x66, 0x90, 0x93, 0x4c, 0x48, 0x33, 0x87, 0x9f, 0x7f, 0xff,
            0xff, 0xff, 0xd7, 0x7d, 0xfd, 0xf7, 0xdf, 0xfb, 0xbf, 0xfe, 0x7b, 0xdf, 0xff, 0xf7, 0xf7, 0xfd,
        },
        {
            // leekspin: frame 3
            0xff, 0xef, 0xbf, 0xfb, 0xef, 0xbf, 0xfb, 0xef, 0xbf, 0xfb, 0x6f, 0xff, 0xdb, 0x7f, 0xff, 0xff,
            0xbb, 0xef, 0x7f, 0xdb, 0xff, 0xb7, 0xfd, 0x5d, 0x02, 0xb8, 0xc6, 0x12, 0x30, 0xc4, 0x32, 0x4a,
            0x0c, 0xf1, 0xfd, 0xbf, 0xfb, 0x6d, 0xfb, 0xbb, 0x77, 0x77, 0xbb, 0xef, 0x6f, 0xfb, 0xcf, 0xdf,
            0xbb, 0x6f, 0x7f, 0xdb, 0xff, 0xf7, 0x0f, 0xe0, 0x3f, 0xc0, 0x0d, 0xb7, 0xe0, 0x0d, 0xa9, 0xf8,
            0xc7, 0x1d, 0xf0, 0xff, 0xff, 0xf0, 0xc7, 0xf0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xf1, 0xc6,
            0xff, 0xf7, 0xf8, 0xf3, 0xcf, 0x90, 0xf7, 0xfc, 0xe3, 0x99, 0x2e, 0xe2, 0xfc, 0x03, 0xfc, 0x78,
            0x13, 0x27, 0xec, 0x19, 0x53, 0xe6, 0x2d, 0xd8, 0x83, 0x3b, 0xee, 0x00, 0x7f, 0xff, 0xbb, 0xef,
            0xff, 0xbb, 0xef, 0xff, 0xbb, 0xef, 0xff, 0xbb, 0xef, 0xff, 0xbb, 0xef, 0x7f, 0xdb, 0xff, 0xb7,
            0xff, 0xfb, 0x6f, 0xfe, 0xdb, 0x7f, 0xf6, 0xdf, 0xfd, 0xb7, 0xff, 0x6d, 0xff, 0xdb, 0xff, 0xf7,
            0xff, 0xfd, 0x9f, 0xfb, 0xf7, 0xff, 0xbb, 0x00, 0x66, 0x99, 0x4c, 0x63, 0x99, 0x8c, 0x63, 0x7c,
            0x1f, 0xce, 0x26, 0x35, 0xc9, 0x0b, 0x30, 0x4c, 0x83, 0xb0, 0xfc, 0x6f, 0xef, 0xdb, 0xbf, 0x6e,
            0x7b, 0xdf, 0xf7, 0xbe, 0xee, 0xfd, 0x00, 0xd9, 0x7b, 0x02, 0x86, 0x7c, 0x09, 0xff, 0x5f, 0x3f,
            0xa7, 0x9f, 0xcb, 0xcb, 0xef, 0xcb, 0xdb, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xef, 0x9f, 0xd7, 0xcf, 0xcb, 0xdf, 0xd7, 0xcf, 0x8f, 0x3f, 0x1f, 0x7f, 0xff, 0xff, 0xfb, 0x72,
            0x87, 0xb9, 0x6c, 0x03, 0x3b, 0xcc, 0xd3, 0x36, 0x88, 0xdb, 0x75, 0x84, 0xff, 0xfb, 0xef, 0xbe,
            0xfb, 0xef, 0xfe, 0x7b, 0xef, 0xfe, 0xdb, 0xff, 0xb6, 0xff, 0xed, 0xbf, 0xfb, 0x6f, 0xfe, 0xdb,
            0xff, 0xff, 0x77, 0xfd, 0xef, 0x7b, 0xff, 0xb6, 0xff, 0xed, 0x7f, 0xfb, 0xdf, 0xfe, 0xb7, 0xff,
            0xee, 0xfe, 0xb7, 0xff, 0x7f, 0xef, 0xf9, 0x02, 0x66, 0x99, 0x92, 0x66, 0x49, 0x98, 0xa6, 0xb1,
            0xcd, 0xe4, 0xf3, 0xf9, 0xbc, 0xef, 0xfd, 0xde, 0x77, 0xff, 0xed, 0xbb, 0xfb, 0xf6, 0x6f, 0x1f,
            0x37, 0x4e, 0xce, 0x31, 0x93, 0x63, 0x4c, 0x84, 0x32, 0x59, 0x45, 0x22, 0xaa, 0xc8, 0xfe, 0xff,
            0xf7, 0xf7, 0xdf, 0x7f, 0x3f, 0x3f, 0x7f, 0x7f, 0x3f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f,
            0x7f, 0x3f, 0x7f, 0x7f, 0x3f, 0x3f, 0x77, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xef, 0x3f, 0xc4,
            0xdd, 0x33, 0x82, 0xc8, 0x3f, 0x32, 0x82, 0xec, 0x33, 0x86, 0xf8, 0x7f, 0xf7, 0xdd, 0xff, 0xbb,
            0xee, 0x7f, 0xdd, 0xf7, 0xff, 0xee, 0xbb, 0xff, 0xf7, 0xbf, 0xfe, 0xff, 0xed, 0xbf, 0xfb, 0x6e,
            0xff, 0xff, 0xbf, 0xed, 0xff, 0xff, 0xb7, 0xff, 0xfd, 0xdf, 0xf7, 0xff, 0x6e, 0xfb, 0xff, 0xb7,
            0xfe, 0xef, 0x7d, 0xf7, 0xbf, 0xfd, 0xdf, 0xf0, 0x06, 0x99, 0xe4, 0x26, 0xff, 0xfd, 0xdf, 0xff,
            0xf6, 0xbf, 0xff, 0xee, 0xfb, 0x7f, 0xdd, 0xff, 0x37, 0x9f, 0xce, 0x27, 0x31, 0xcc, 0x62, 0x9b,
            0x89, 0x26, 0xb2, 0x98, 0x71, 0x46, 0x9e, 0x1d, 0x70, 0x86, 0x89, 0x33, 0x94, 0x87, 0x2f, 0x4f,
            0x1f, 0xbf, 0x3f, 0x7f, 0x7e, 0xf9, 0xe4, 0x60, 0xdb, 0xd8, 0xc6, 0xf8, 0xc9, 0xda, 0xf2, 0xcc,
            0xc8, 0xf3, 0xf4, 0xe0, 0xfb, 0xf8, 0xfe, 0x7f, 0x7f, 0x3f, 0x9f, 0x5f, 0x0f, 0xc9, 0x72, 0x0e,
            0xe1, 0x3c, 0x06, 0x09, 0x61, 0xb4, 0xd7, 0x4c, 0xf1, 0xbf, 0xff, 0xfd, 0xe7, 0xbf, 0xff, 0x7d,
            0xf7, 0xff, 0xed, 0xff, 0xfe, 0xcf, 0xff, 0xbe, 0xf7, 0xdd, 0xff, 0xef, 0xfe, 0x7b, 0xdf, 0xf7,
            0xff, 0xff, 0x7b, 0xdf, 0xfe, 0xfb, 0xdf, 0xfd, 0xef, 0xfe, 0xf7, 0xdf, 0xfb, 0xff, 0x76, 0xff,
            0xef, 0xfd, 0xff, 0xf7, 0xdf, 0x7d, 0xff, 0xff, 0xb0, 0xbf, 0xff, 0xff, 0xf6, 0x9f, 0xff, 0xfb,
            0xfe, 0xdf, 0xf7, 0xfd, 0x9f, 0xef, 0xf1, 0xf4, 0x66, 0xc9, 0xfc, 0xf3, 0xf9, 0xfc, 0xb6, 0xf8,
            0xf1, 0xf6, 0xdc, 0xf1, 0xf2, 0xee, 0xf1, 0xf4, 0x64, 0x01, 0xb4, 0x26, 0x49, 0x10, 0x24, 0x81,
            0x12, 0x94, 0x65, 0x08, 0x02, 0x64, 0x18, 0x42, 0x24, 0x19, 0x01, 0x85, 0x21, 0xbf, 0xff, 0x6d,
            0x5d, 0x37, 0x1d, 0x1d, 0x86, 0x26, 0x07, 0x10, 0x40, 0x09, 0x21, 0xd2, 0x4c, 0x19, 0x42, 0x13,
            0x40, 0x4c, 0x11, 0xc0, 0x0e, 0xbb, 0x80, 0xfe, 0xff, 0x6f, 0xfb, 0xbe, 0xf7, 0xdf, 0xfb, 0xff,
            0xbd, 0xef, 0xfb, 0xff, 0xbd, 0xef, 0xfb, 0xff, 0xdf, 0xfb, 0xef, 0x7e, 0xf7, 0xbf, 0xff, 0x6d,
            0xff, 0xff, 0xef, 0xfb, 0xbf, 0xfe, 0xef, 0x7b, 0xfe, 0xdf, 0xff, 0xf6, 0xdf, 0x7f, 0xf7, 0xdf,
            0xfe, 0xfb, 0xdf, 0x77, 0xff, 0xef, 0xfd, 0xff, 0x6f, 0xfc, 0xa7, 0xbf, 0xff, 0xff, 0xd9, 0xdf,
            0xbe, 0xbf, 0x77, 0x7e, 0xff, 0xdf, 0xfb, 0xfc, 0x7f, 0xdf, 0xff, 0xff, 0xbe, 0xfb, 0xff, 0xfd,
            0xdf, 0xfe, 0xff, 0x7b, 0xfe, 0xbf, 0xff, 0xef, 0xd9, 0x11, 0xef, 0x28, 0x00, 0x41, 0x4c, 0x31,
            0xc4, 0x0c, 0x68, 0x93, 0x10, 0x64, 0x01, 0x10, 0x04, 0x41, 0x10, 0x04, 0x00, 0x43, 0x11, 0x00,
            0x88, 0x22, 0x00, 0x48, 0x82, 0x90, 0x11, 0xc4, 0x30, 0x0d, 0x43, 0x10, 0x24, 0xc1, 0x10, 0xb4,
            0x81, 0x68, 0x24, 0x90, 0x4a, 0x2b, 0x80, 0x3f, 0xff, 0xdb, 0xff, 0xf7, 0xfd, 0xbf, 0xee, 0xff,
            0x7b, 0xef, 0xbe, 0xff, 0xed, 0x7f, 0xf6, 0xbf, 0xfb, 0xfe, 0x6f, 0xff, 0xdb, 0xff, 0xfd, 0x6f,
            0xff, 0xff, 0xfb, 0xfe, 0xef, 0xff, 0xbd, 0xff, 0xf7, 0xff, 0xbd, 0xff, 0xef, 0xff, 0xdf, 0xfe,
            0xff, 0xbd, 0xff, 0xdf, 0xff, 0xff, 0xf7, 0x9c, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xf9, 0xdf, 0x5f, 0xe6, 0xfd, 0xff, 0xff, 0xf7, 0xbf, 0xfb, 0xff, 0xfd, 0xdf, 0xfe, 0xff,
            0xf7, 0xbd, 0xff, 0x5f, 0x37, 0x9f, 0x47, 0x61, 0x9c, 0x10, 0x63, 0x06, 0xc0, 0x09, 0x64, 0x93,
            0x18, 0x66, 0x82, 0x59, 0x11, 0xa6, 0xa4, 0x19, 0x42, 0x32, 0x8c, 0x44, 0x20, 0x98, 0x47, 0x68,
            0x08, 0xd2, 0x93, 0x24, 0x8c, 0x70, 0x02, 0x0d, 0xc0, 0x32, 0x08, 0xe1, 0x82, 0x0a, 0x3a, 0xc0,
            0x16, 0x44, 0x61, 0x19, 0x46, 0xc8, 0x31, 0x06, 0xd8, 0x43, 0x3f, 0xfe, 0xff, 0xf7, 0xbd, 0xff,
            0xdb, 0xff, 0xf7, 0xff, 0xfd, 0xbf, 0xef, 0xef, 0xff, 0x6d, 0xff, 0xf7, 0xfe, 0xfb, 0xaf, 0xff,
            0xff, 0xff, 0xbe, 0xef, 0xfd, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0x7f, 0xfb, 0xdf, 0xfe, 0xef, 0xfd,
            0x7f, 0xff, 0xdb, 0xff, 0xfd, 0xdf, 0xfe, 0x7e, 0xfb, 0xef, 0xef, 0xdf, 0x9f, 0xff, 0xdf, 0xdf,
            0x6f, 0xef, 0xf3, 0xfc, 0xfd, 0xff, 0xf7, 0xef, 0xbf, 0x7d, 0xff, 0xfe, 0xef, 0xdf, 0x1b, 0x26,
            0x27, 0xc9, 0x98, 0x12, 0x63, 0x2c, 0x82, 0x5a, 0x41, 0x94, 0x93, 0x2c, 0x20, 0xd3, 0x8c, 0x24,
            0x53, 0x48, 0x25, 0xb4, 0x83, 0x28, 0x24, 0xc5, 0x11, 0x52, 0x8c, 0x20, 0x9b, 0xa0, 0x26, 0x59,
            0x82, 0x2a, 0x68, 0x86, 0x99, 0x5c, 0x37, 0x96, 0x5e, 0xb1, 0x84, 0xcc, 0xf1, 0xc6, 0xc9, 0xf8,
            0xfe, 0xed, 0xf8, 0xeb, 0xc2, 0x14, 0x51, 0x4d, 0xa4, 0x92, 0x18, 0x63, 0x47, 0x9f, 0x3f, 0xff,
            0xff, 0xff, 0xf7, 0xdc, 0xff, 0xbb, 0xff, 0xf6, 0xdf, 0x7f, 0xff, 0xde, 0xf7, 0xff, 0xfb, 0xfe
        }
    };

    if(timer_elapsed(anim_timer) > ANIM_FRAME_DURATION) {
        anim_timer = timer_read();
        current_anim_frame = (current_anim_frame + 1) % ANIM_TOTAL_FRAMES;
        oled_write_raw_P(frame[abs((ANIM_NUM_FRAMES-1)-current_anim_frame)], 1024);
    }
}

bool oled_task_user(void) {
    if (is_keyboard_master()) {
        render_status();     // Renders the current keyboard state (layer, lock, caps, scroll, etc)
    }
    else {
        render_logo();
    }
    return false;
}
#endif
