/* Copyright 2019 Thomas Baart <thomas@splitkb.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include QMK_KEYBOARD_H

#ifdef CONSOLE_ENABLE
	#include <print.h>
#endif
enum custom_keycodes {
  QWERTY = SAFE_RANGE,
  WORKMAN,
  ADJUST,
  SBKSP,
  ENDW
};

enum layers {
	_QWERTY = 0,
	_WORKMAN,
	_LOWER,
	_RAISE,
	_ADJUST
};
#define RAISE LT(_RAISE, KC_ENT)
#define LOWER MO(_LOWER)
#define HOME_A KC_A
#define HOME_S KC_S
#define HOME_H CTL_T(KC_H)
#define HOME_T SFT_T(KC_T)
#define HOME_N SFT_T(KC_N)
#define HOME_E CTL_T(KC_E)
#define HOME_O KC_O
#define HOME_I KC_I
const uint16_t PROGMEM keymaps[][MATRIX_ROWS][MATRIX_COLS] = {
/*
 * Base Layer: QWERTY
 *
 * ,-------------------------------------------.                              ,-------------------------------------------.
 * |   ESC  |   Q  |   W  |   E  |   R  |   T  |                              |   Y  |   U  |   I  |   O  |   P  |        |
 * |--------+------+------+------+------+------|                              |------+------+------+------+------+--------|
 * |   TAB  |   A  |   S  |   D  |   F  |   G  |                              |   H  |   J  |   K  |   L  |   ;  |    '   |
 * |--------+------+------+------+------+------+-------------.  ,-------------+------+------+------+------+------+--------|
 * |  CTRL  |   Z  |   X  |   C  |   V  |   B  |      |      |  |      |      |   N  |   M  |   ,  |   .  |   /  |  ENTER |
 * `----------------------+------+------+------+------+------|  |------+------+------+------+------+----------------------'
 *                        | GUI  |  ALT | LOWER| Space|      |  | SHIFT| Bksp | RAISE|      |      |
 *                        `----------------------------------'  `----------------------------------'
 */
	[_QWERTY] = LAYOUT(
	KC_ESC,  KC_Q,    KC_W,    KC_E,    KC_R,    KC_T,                                          KC_Y,    KC_U,    KC_I,    KC_O,    KC_P,    _______,
	KC_TAB,  KC_A,    KC_S,    KC_D,    KC_F,    KC_G,                                          KC_H,    KC_J,    KC_K,    KC_L,    KC_SCLN, KC_QUOT,
	KC_LCTL, KC_Z,    KC_X,    KC_C,    KC_V,    KC_B, _______, _______,     _______, _______,  KC_N,    KC_M,    KC_COMM, KC_DOT,  KC_SLSH, KC_ENT,
							   KC_LGUI, KC_LALT, LOWER, KC_SPC, _______,     KC_LSFT, KC_BSPC, RAISE, _______, _______
	),
	/*
 * Workman Layer
 *
 * ,-------------------------------------------.                              ,-------------------------------------------.
 * |   ESC  |   Q  |   D  |   R  |   W  |   B  |                              |   J  |   F  |   U  |   P  |   ;  |  Bksp  |
 * |--------+------+------+------+------+------|                              |------+------+------+------+------+--------|
 * |   TAB  |   A  |   S  |   H  |   T  |   G  |                              |   Y  |   N  |   E  |   O  |   I  |    '   |
 * |--------+------+------+------+------+------+-------------.  ,-------------+------+------+------+------+------+--------|
 * |        |   Z  |   X  |   M  |   C  |   V  | SHIFT| Space|  |      | ENTER|   K  |   L  |   ,  |   .  |   /  |        |
 * `----------------------+------+------+------+------+------|  |------+------+------+------+------+----------------------'
 *                        |  GUI |  ESC | LOWER| Space| SWAP |  | SWAP | Bksp | RAISE|      |      |
 *                        `----------------------------------'  `----------------------------------'
 */
	[_WORKMAN] = LAYOUT(
	KC_ESC, KC_Q,    KC_D,    KC_R,    KC_W,    KC_B,                                          KC_J,    KC_F,    KC_U,    KC_P,    KC_SCLN, KC_BSPC,
	KC_TAB, HOME_A,  HOME_S,  HOME_H,  HOME_T,  KC_G,                                          KC_Y,    HOME_N,  HOME_E,  HOME_O,  HOME_I,  KC_QUOT,
    XXXXXXX, KC_Z,    KC_X,    KC_M,    KC_C,    KC_V,  KC_LSFT, KC_SPC,     SH_MON, KC_ENT,    KC_K,    KC_L,    KC_COMM, KC_DOT,  KC_SLSH, XXXXXXX,
							   KC_LGUI, KC_LALT,  LOWER, KC_SPC,  SH_MON,     SH_MON, KC_BSPC,   RAISE,   XXXXXXX, _______
	),
/*
 * Lower Layer: NUM/symb
 *
 * ,-------------------------------------------.                              ,-------------------------------------------.
 * |    `   |   1  |   2  |   3  |   4  |   5  |                              |   6  |   7  |   8  |   9  |   0  |        |
 * |--------+------+------+------+------+------|                              |------+------+------+------+------+--------|
 * |   DEL  |   $  |   +  |   (  |   )  |   @  |                              |   |  |   -  |   =  |   _  |   *  |    \   |
 * |--------+------+------+------+------+------+-------------.  ,-------------+------+------+------+------+------+--------|
 * |        |   !  |   #  |   {  |   }  |   ~  |      |      |  |      |      |   &  |   [  |   ]  |   %  |   ^  |        |
 * `----------------------+------+------+------+------+------|  |------+------+------+------+------+----------------------'
 *                        |prints|      |      |      |      |  |      |      |      |      |      |
 *                        `----------------------------------'  `----------------------------------'
 */
	[_LOWER] = LAYOUT(
	KC_GRV,  KC_1,    KC_2,    KC_3,    KC_4,    KC_5,                                            KC_6,    KC_7,    KC_8,    KC_9,    KC_0,    _______,
	KC_DEL,  KC_DLR,  KC_PLUS, KC_LPRN, KC_RPRN, KC_AT,                                           KC_PIPE, KC_MINS, KC_EQL,  KC_UNDS, KC_ASTR, KC_BSLS,
	_______, KC_EXLM, KC_HASH, KC_LCBR, KC_RCBR, KC_TILD, _______, _______,     _______, _______, KC_AMPR, KC_LBRC, KC_RBRC, KC_PERC, KC_CIRC, _______,
							   KC_PSCR, _______, _______, _______, _______,     _______, KC_BSPC, _______, _______, _______
	),
 /*
  * Raise Layer: VIM Movement
  *
  * ,-------------------------------------------.                              ,-------------------------------------------.
  * |        |      | PgDn |      | Word | Back |                              | Down |      | PgUp |      |      |        |
  * |--------+------+------+------+------+------|                              |------+------+------+------+------+--------|
  * |        |      |      | Left |      |      |                              |      |      |  End |      |      |        |
  * |--------+------+------+------+------+------+-------------.  ,-------------+------+------+------+------+------+--------|
  * |        |      |      |      |      |      |      |      |  |      |      |  Up  | Right|      |      |      |        |
  * `----------------------+------+------+------+------+------|  |------+------+------+------+------+----------------------'
  *                        |      |      |      |      |      |  |      |      |      |      |      |
  *                        |      |      |      |      |      |  |      |      |      |      |      |
  *                        `----------------------------------'  `----------------------------------'
  */
	 [_RAISE] = LAYOUT(
       _______, _______, KC_PGDN, _______, C(KC_RGHT), C(KC_LEFT),                               KC_DOWN, _______, KC_PGUP, _______, _______, _______,
       _______, _______, _______, KC_LEFT, _______, _______,                                     _______, _______, ENDW,     _______, _______, _______,
       _______, _______, _______, _______, _______, _______, _______, _______, _______, _______, KC_UP,   KC_RGHT, _______, _______, _______, _______,
                                  _______, _______, _______, _______, _______, _______, _______, _______, _______, _______
	 ),
 /*
  * Layer template
  *
  * ,-------------------------------------------.                              ,-------------------------------------------.
  * |    F1  |  F2  |  F3  |  F4  |  F5  |  F6  |                              |  F7  |  F8  |  F9  |  F10 |  F11 |   F12  |
  * |--------+------+------+------+------+------|                              |------+------+------+------+------+--------|
  * |        |      |      |      |      |      |                              |      |      |      |      |      |        |
  * |--------+------+------+------+------+------+-------------.  ,-------------+------+------+------+------+------+--------|
  * |        |      |      |      |QWERTY|WRKMAN|      |      |  |      |      |      |      |      |      |      |        |
  * `----------------------+------+------+------+------+------|  |------+------+------+------+------+----------------------'
  *                        |      |      |      |      |      |  |      |      |      |      |      |
  *                        |      |      |      |      |      |  |      |      |      |      |      |
  *                        `----------------------------------'  `----------------------------------'
  */
	 [_ADJUST] = LAYOUT(
	   KC_F1,   KC_F2,   KC_F3,   KC_F4,   KC_F5,   KC_F6,                                       KC_F7,   KC_F8,   KC_F9,   KC_F10,  KC_F11,  KC_F12,
	   _______, _______, _______, _______, _______, _______,                                     _______, _______, _______, _______, _______, _______,
	   _______, _______, _______, _______, QWERTY,  WORKMAN, _______, _______, _______, _______, _______, _______, _______, _______, _______, _______,
								  _______, _______, _______, _______, _______, _______, _______, _______, _______, _______
	 ),
// /*
//  * Layer template
//  *
//  * ,-------------------------------------------.                              ,-------------------------------------------.
//  * |        |      |      |      |      |      |                              |      |      |      |      |      |        |
//  * |--------+------+------+------+------+------|                              |------+------+------+------+------+--------|
//  * |        |      |      |      |      |      |                              |      |      |      |      |      |        |
//  * |--------+------+------+------+------+------+-------------.  ,-------------+------+------+------+------+------+--------|
//  * |        |      |      |      |      |      |      |      |  |      |      |      |      |      |      |      |        |
//  * `----------------------+------+------+------+------+------|  |------+------+------+------+------+----------------------'
//  *                        |      |      |      |      |      |  |      |      |      |      |      |
//  *                        |      |      |      |      |      |  |      |      |      |      |      |
//  *                        `----------------------------------'  `----------------------------------'
//  */
//     [_LAYERINDEX] = LAYOUT(
//       _______, _______, _______, _______, _______, _______,                                     _______, _______, _______, _______, _______, _______,
//       _______, _______, _______, _______, _______, _______,                                     _______, _______, _______, _______, _______, _______,
//       _______, _______, _______, _______, _______, _______, _______, _______, _______, _______, _______, _______, _______, _______, _______, _______,
//                                  _______, _______, _______, _______, _______, _______, _______, _______, _______, _______
//     ),
};


layer_state_t layer_state_set_user(layer_state_t state) {
	return update_tri_layer_state(state, _LOWER, _RAISE, _ADJUST);
}

#ifdef ENCODER_ENABLE
bool encoder_update_user(uint8_t index, bool clockwise) {
	if (index == 0) {
		// Volume control
		if (clockwise) {
			tap_code(KC_RGHT);
		} else {
			tap_code(KC_LEFT);
		}
	}
	else if (index == 1) {
		// Page up/Page down
		if (clockwise) {
			tap_code(KC_DOWN);
		} else {
			tap_code(KC_UP);
		}
	}
    return true;
}
#endif
#ifdef OLED_ENABLE
bool left = false;
bool right = false;
bool lastl = false;
bool lastr = false;
#endif
bool bksp = false;
bool process_record_user(uint16_t keycode, keyrecord_t *record) {
	#ifdef CONSOLE_ENABLE
	uprintf("KL: kc: %u, col: %u, row: %u, pressed: %u, total: %u\n", keycode, record->event.key.col, record->event.key.row, record->event.pressed, record->event.key.col + 10 * record->event.key.row);
	#endif
	#ifdef OLED_ENABLE
	if(record->event.pressed){
		uint8_t n = record->event.key.col + 10 * record->event.key.row;
		if (n<40) {
			left = true;
		}
		else {
			right = true;
		}
	}
	#endif
	switch (keycode) {
		case QWERTY:
			if (record->event.pressed) {
				set_single_persistent_default_layer(_QWERTY);
			}
			return false;
			break;
		case WORKMAN:
			if (record->event.pressed) {
				set_single_persistent_default_layer(_WORKMAN);
			}
			return false;
			break;
		case SBKSP:
			if (record->event.pressed) {
				register_code(KC_LSFT);
				bksp = true;
			}
			else {
				unregister_code(KC_LSFT);
				if (bksp) {
					tap_code(KC_BSPC);
				}
			}
			return false;
			break;
        case ENDW:
            if (record->event.pressed) {
                register_code(KC_LCTL);
                tap_code(KC_RGHT);
                unregister_code(KC_LCTL);
                tap_code(KC_LEFT);
            }
	}
	bksp = false;
	return true;

}

#ifdef OLED_ENABLE
oled_rotation_t oled_init_user(oled_rotation_t rotation) {
	return OLED_ROTATION_180;
}



char wpm_str[10];
// WPM-responsive animation stuff here
#define IDLE_FRAMES 2
#define IDLE_SPEED 10 // below this wpm value your animation will idle


#define TAP_FRAMES 3

#define ANIM_FRAME_DURATION 400 // how long each frame lasts in ms
// #define SLEEP_TIMER 60000 // should sleep after this period of 0 wpm, needs fixing
#define ANIM_SIZE 896 // number of bytes in array, minimize for adequate firmware size, max is 1024

uint32_t anim_timer = 0;
uint32_t anim_sleep = 0;
uint8_t current_idle_frame = 0;
// uint8_t current_prep_frame = 0; // uncomment if PREP_FRAMES >1
uint8_t current_tap_frame = 0;
// Credit to obosob for initial animation approach.
static void render_anim(void) {
	static const char PROGMEM idle[IDLE_FRAMES][ANIM_SIZE] = {
		{
// 'bongo0', 128x56px
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x40, 0x60, 0x20, 0x30, 0x30, 0x10,
0x18, 0x08, 0x0c, 0x04, 0x04, 0x04, 0x86, 0x82, 0xc2, 0x42, 0x22, 0x12, 0x12, 0x12, 0x12, 0x14,
0x14, 0x14, 0x14, 0x14, 0x18, 0x19, 0x19, 0x11, 0x11, 0x10, 0x10, 0x00, 0x20, 0xf0, 0x2f, 0x24,
0x66, 0xda, 0xd1, 0x11, 0x91, 0x91, 0x11, 0x11, 0x11, 0x12, 0x1e, 0x04, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0xc0, 0x20, 0x18, 0x04, 0x02, 0x01, 0x80, 0x80, 0x00, 0x40, 0x40, 0xc0, 0x40, 0x30,
0x90, 0x88, 0x44, 0x42, 0x42, 0x41, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x20, 0x20, 0x20,
0x20, 0x20, 0x20, 0x20, 0x20, 0x10, 0x10, 0xf0, 0x88, 0x08, 0x04, 0x02, 0x01, 0x00, 0x00, 0x00,
0x00, 0x00, 0x01, 0x03, 0x06, 0x08, 0x71, 0x81, 0x02, 0x06, 0x04, 0x08, 0x10, 0x20, 0xc0, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x03, 0x82, 0xe2, 0x39, 0x09, 0x05, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x39,
0x1c, 0x00, 0x00, 0x00, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
0xe0, 0xf0, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x06, 0x18, 0x20, 0x40, 0x80, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x20, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x01,
0x06, 0x38, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x40, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xc0, 0x40, 0x20, 0x30,
0x18, 0x0e, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02,
0x02, 0x04, 0x08, 0x08, 0x10, 0x08, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0d, 0xf3, 0x06, 0x0c, 0x10,
0x20, 0x20, 0xc1, 0xff, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x02, 0xfa, 0x06, 0x06, 0x03, 0x03, 0x02, 0x02, 0x04, 0x18,
0xf0, 0x80, 0xc0, 0x60, 0x20, 0x10, 0x18, 0x0c, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x10, 0x10,
0x10, 0x10, 0x10, 0x10, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x40, 0x40, 0x40, 0x40, 0x40,
0x40, 0x40, 0x80, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x01, 0x01,
0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8,
0x8e, 0x03, 0x01, 0x01, 0x01, 0x03, 0x06, 0x9c, 0xf0, 0x80, 0xc0, 0x40, 0x60, 0x20, 0x30, 0x10,
0x08, 0x0d, 0x07, 0x03, 0x03, 0x02, 0x02, 0x02, 0x02, 0x04, 0x04, 0x04, 0x04, 0x07, 0x04, 0x00,
0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x10, 0x10, 0x10, 0x10, 0x10,
0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
		},
		{
// 'bongo1', 128x56px
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0xc0, 0x40,
0x60, 0x20, 0x20, 0x20, 0x20, 0x00, 0x10, 0x10, 0x10, 0x90, 0x90, 0x90, 0x90, 0x90, 0xa0, 0xa0,
0xa0, 0xa0, 0xa0, 0xc0, 0xc0, 0xc1, 0x80, 0x80, 0x81, 0x81, 0x02, 0x06, 0x84, 0x7c, 0x5f, 0x58,
0x4c, 0x46, 0x42, 0x42, 0x42, 0x43, 0x61, 0x23, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xa0, 0x20, 0x20,
0x30, 0x10, 0x10, 0x18, 0x08, 0x08, 0x0c, 0x04, 0x06, 0x02, 0x03, 0x01, 0x08, 0x18, 0x18, 0x28,
0x68, 0x44, 0x84, 0x84, 0x86, 0x02, 0x02, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x80, 0x40, 0x40, 0x20, 0x10, 0x09, 0x07, 0x01, 0x01, 0x03,
0x06, 0x06, 0x10, 0x3c, 0x4c, 0xc8, 0x88, 0x08, 0x08, 0x18, 0x10, 0x30, 0x60, 0xc0, 0x80, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,
0x01, 0x01, 0x81, 0xc1, 0x21, 0x31, 0x11, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x06, 0x0e, 0xc6, 0xe4,
0x04, 0x04, 0x04, 0x06, 0x02, 0x03, 0x01, 0x03, 0x02, 0x02, 0x02, 0x02, 0x01, 0x01, 0x01, 0x01,
0x81, 0x01, 0x01, 0x01, 0x00, 0x00, 0x07, 0x0c, 0x30, 0x40, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x07, 0x1e, 0x30, 0x40, 0x80, 0x00, 0x00, 0x00, 0x01, 0x07,
0x1c, 0x70, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x40, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xc0, 0x40, 0x20, 0x30,
0x18, 0x0e, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
0x00, 0x00, 0x00, 0x02, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07,
0x0f, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x03, 0x02, 0x02,
0x06, 0x04, 0x04, 0x04, 0x08, 0x08, 0x08, 0x0c, 0x04, 0x0c, 0x3c, 0x15, 0xd3, 0x72, 0x10, 0x20,
0x20, 0x20, 0x21, 0x23, 0x26, 0x24, 0x2c, 0x18, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x02, 0xfa, 0x06, 0x06, 0x03, 0x03, 0x02, 0x02, 0x04, 0x18,
0xf0, 0x80, 0xc0, 0x60, 0x20, 0x10, 0x18, 0x0c, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x10, 0x10,
0x10, 0x10, 0x10, 0x10, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x40, 0x40, 0x40, 0x40, 0x40,
0x40, 0x40, 0x80, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x01, 0x01,
0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8,
0x8e, 0x03, 0x01, 0x01, 0x01, 0x03, 0x06, 0x9c, 0xf0, 0x80, 0xc0, 0x40, 0x60, 0x20, 0x30, 0x10,
0x08, 0x0d, 0x07, 0x03, 0x03, 0x02, 0x02, 0x02, 0x02, 0x04, 0x04, 0x04, 0x04, 0x07, 0x04, 0x00,
0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x10, 0x10, 0x10, 0x10, 0x10,
0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
		}
	};
	static const char PROGMEM prep[][ANIM_SIZE] = {
		{
// 'bongo2', 128x56px
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x40, 0x60, 0x20, 0x30, 0x30, 0x10,
0x18, 0x08, 0x0c, 0x04, 0x04, 0x04, 0x86, 0x82, 0xc2, 0x42, 0x22, 0x12, 0x12, 0x12, 0x12, 0x14,
0x14, 0x14, 0x14, 0x14, 0x18, 0x19, 0x19, 0x11, 0x11, 0x10, 0x10, 0x00, 0x20, 0xf0, 0x2f, 0x24,
0x66, 0xda, 0xd1, 0x11, 0x91, 0x91, 0x11, 0x11, 0x11, 0x12, 0x1e, 0x04, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0xc0, 0x20, 0x18, 0x04, 0x02, 0x01, 0x80, 0x80, 0x00, 0x40, 0x40, 0xc0, 0x40, 0x30,
0x90, 0x88, 0x44, 0x42, 0x42, 0x41, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x20, 0x20, 0x20,
0x20, 0x20, 0x20, 0x20, 0x20, 0x10, 0x10, 0xf0, 0x88, 0x08, 0x04, 0x02, 0x01, 0x00, 0x00, 0x00,
0x00, 0x00, 0x01, 0x03, 0x06, 0x08, 0x71, 0x81, 0x02, 0x06, 0x04, 0x08, 0x10, 0x20, 0xc0, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x18, 0x08, 0x0c, 0x04, 0x04,
0x04, 0x07, 0x0a, 0x92, 0xf9, 0xc9, 0x05, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x39,
0x1c, 0x00, 0x00, 0x00, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
0xe0, 0xf0, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc1, 0x46, 0x78, 0x20, 0x20, 0x20, 0x20, 0x40,
0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x20, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x01,
0x06, 0x38, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x40, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00,
0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x1e, 0x64, 0xc4, 0x04, 0x04,
0x00, 0x02, 0x03, 0x01, 0x00, 0x01, 0x03, 0x0c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xe0, 0x40, 0x40, 0x40, 0x40, 0x40, 0x60, 0x20,
0x30, 0x19, 0x0e, 0x38, 0xcc, 0x08, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0d, 0xf3, 0x06, 0x0c, 0x10,
0x20, 0x20, 0xc1, 0xff, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x04, 0x05, 0x07, 0x0e,
0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x10, 0x10,
0x10, 0x10, 0x10, 0x10, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x40, 0x40, 0x40, 0x40, 0x40,
0x40, 0x40, 0x00, 0x80, 0x80, 0x80, 0x80, 0x80, 0x83, 0x87, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x01, 0x0e, 0x38, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01,
0x01, 0x01, 0x03, 0x02, 0x02, 0x02, 0x03, 0x03, 0x02, 0x04, 0x04, 0x04, 0x04, 0x07, 0x04, 0x00,
0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x10, 0x10, 0x10, 0x10, 0x10,
0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00

		}
	};
	static const char PROGMEM tap[TAP_FRAMES][ANIM_SIZE] = {
		{
	  // 'bongo3', 128x56px
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x40, 0x60, 0x20, 0x30, 0x30, 0x10,
0x18, 0x08, 0x0c, 0x04, 0x04, 0x04, 0x86, 0x82, 0xc2, 0x42, 0x22, 0x12, 0x12, 0x12, 0x12, 0x14,
0x14, 0x14, 0x14, 0x14, 0x18, 0x19, 0x19, 0x11, 0x11, 0x10, 0x10, 0x00, 0x20, 0xf0, 0x2f, 0x24,
0x66, 0xda, 0xd1, 0x11, 0x91, 0x91, 0x11, 0x11, 0x11, 0x12, 0x1e, 0x04, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x80, 0x80, 0x80, 0x80, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0xc0, 0x20, 0x18, 0x04, 0x02, 0x01, 0x80, 0x80, 0x00, 0x40, 0x40, 0xc0, 0x40, 0x30,
0x90, 0x88, 0x44, 0x42, 0x42, 0x41, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x20, 0x20, 0x20,
0x20, 0x20, 0x20, 0x20, 0x20, 0x10, 0x10, 0xf0, 0x88, 0x08, 0x04, 0x02, 0x01, 0x00, 0x00, 0x00,
0x00, 0x00, 0x01, 0x03, 0x06, 0x08, 0x71, 0x81, 0x02, 0x06, 0x04, 0x08, 0x10, 0x20, 0xc0, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x1f, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x03, 0x82, 0xe2, 0x39, 0x09, 0x05, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x39,
0x1c, 0x00, 0x00, 0x00, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
0xe0, 0xf0, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc1, 0x46, 0x78, 0x20, 0x20, 0x20, 0x20, 0x40,
0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x20, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x01,
0x06, 0x38, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x40, 0x80, 0x80, 0x80, 0xe0, 0xf8, 0xf8, 0xf0,
0xe0, 0xe0, 0xc0, 0xc0, 0x80, 0x01, 0x03, 0x1f, 0x0f, 0x07, 0x00, 0x80, 0xc0, 0x40, 0x20, 0x30,
0x18, 0x0e, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xe0, 0x40, 0x40, 0x40, 0x40, 0x40, 0x60, 0x20,
0x30, 0x19, 0x0e, 0x38, 0xcc, 0x08, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0d, 0xf3, 0x06, 0x0c, 0x10,
0x20, 0x20, 0xc1, 0xff, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x7f, 0x3f, 0x3f, 0x3f,
0x1f, 0x1f, 0x0f, 0x07, 0x01, 0x02, 0x02, 0xfa, 0x06, 0x06, 0x03, 0x03, 0x02, 0x02, 0x04, 0x18,
0xf0, 0x80, 0xc0, 0x60, 0x20, 0x10, 0x18, 0x0c, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x10, 0x10,
0x10, 0x10, 0x10, 0x10, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x40, 0x40, 0x40, 0x40, 0x40,
0x40, 0x40, 0x00, 0x80, 0x80, 0x80, 0x80, 0x80, 0x83, 0x87, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x01, 0x0e, 0x38, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x80, 0xe0, 0xf0, 0xf8, 0xf9, 0xe1, 0x81, 0x02, 0x02, 0x02, 0x03, 0x01, 0x01,
0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01,
0x01, 0x01, 0x03, 0x02, 0x02, 0x02, 0x03, 0x03, 0x02, 0x04, 0x04, 0x04, 0x04, 0x07, 0x04, 0x00,
0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x10, 0x10, 0x10, 0x10, 0x10,
0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x08, 0x1c, 0x1f, 0x1f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
		},
		{
			// 'bongo4', 128x56px
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x40, 0x60, 0x20, 0x30, 0x30, 0x10,
0x18, 0x08, 0x0c, 0x04, 0x04, 0x04, 0x86, 0x82, 0xc2, 0x42, 0x22, 0x12, 0x12, 0x12, 0x12, 0x14,
0x14, 0x14, 0x14, 0x14, 0x18, 0x19, 0x19, 0x11, 0x11, 0x10, 0x10, 0x00, 0x20, 0xf0, 0x2f, 0x24,
0x66, 0xda, 0xd1, 0x11, 0x91, 0x91, 0x11, 0x11, 0x11, 0x12, 0x1e, 0x04, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0xc0, 0x20, 0x18, 0x04, 0x02, 0x01, 0x80, 0x80, 0x00, 0x40, 0x40, 0xc0, 0x40, 0x30,
0x90, 0x88, 0x44, 0x42, 0x42, 0x41, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x20, 0x20, 0x20,
0x20, 0x20, 0x20, 0x20, 0x20, 0x10, 0x10, 0xf0, 0x88, 0x08, 0x04, 0x02, 0x01, 0x00, 0x00, 0x00,
0x00, 0x00, 0x01, 0x03, 0x06, 0x08, 0x71, 0x81, 0x02, 0x06, 0x04, 0x08, 0x10, 0x20, 0xc0, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x18, 0x08, 0x0c, 0x04, 0x04,
0x04, 0x07, 0x0a, 0x92, 0xf9, 0xc9, 0x05, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x39,
0x1c, 0x00, 0x00, 0x00, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
0xe0, 0xf0, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x06, 0x18, 0x20, 0x40, 0x80, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x20, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x01,
0x06, 0x38, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x40, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00,
0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x1e, 0x64, 0xc4, 0x04, 0x04,
0x00, 0x02, 0x03, 0x01, 0x00, 0x01, 0x03, 0x0c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02,
0x02, 0x04, 0x08, 0x08, 0x10, 0x08, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0d, 0xf3, 0x06, 0x0c, 0x10,
0x20, 0x20, 0xc1, 0xff, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x04, 0x05, 0x07, 0x0e,
0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x10, 0x10,
0x10, 0x10, 0x10, 0x10, 0x20, 0xa0, 0xa0, 0x20, 0x20, 0x20, 0x20, 0x40, 0x40, 0x40, 0x40, 0x40,
0x40, 0x40, 0x80, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x7c, 0x7f, 0x7f, 0x7f, 0x3f, 0x3f, 0x3e, 0x1e, 0x1e, 0x1c, 0x0c, 0x00, 0xf8,
0x8e, 0x03, 0x01, 0x01, 0x01, 0x03, 0x06, 0x9c, 0xf0, 0x80, 0xc0, 0x40, 0x60, 0x20, 0x30, 0x10,
0x08, 0x0d, 0x07, 0x03, 0x03, 0x02, 0x02, 0x02, 0x02, 0x04, 0x04, 0x04, 0x04, 0x07, 0x04, 0x00,
0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x10, 0x10, 0x10, 0x10, 0x10,
0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x30, 0x38, 0x38, 0x78, 0x7c, 0x7e, 0xff, 0xff, 0xff, 0xfc,
0xf1, 0xe1, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x03, 0x07, 0x06, 0x0e,
0x1e, 0x3e, 0x3e, 0x7e, 0x7e, 0x1e, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
		},
		{
// 'bongo5', 128x56px
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x40, 0x60, 0x20, 0x30, 0x30, 0x10,
0x18, 0x08, 0x0c, 0x04, 0x04, 0x04, 0x86, 0x82, 0xc2, 0x42, 0x22, 0x12, 0x12, 0x12, 0x12, 0x14,
0x14, 0x14, 0x14, 0x14, 0x18, 0x19, 0x19, 0x11, 0x11, 0x10, 0x10, 0x00, 0x20, 0xf0, 0x2f, 0x24,
0x66, 0xda, 0xd1, 0x11, 0x91, 0x91, 0x11, 0x11, 0x11, 0x12, 0x1e, 0x04, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x80, 0x80, 0x80, 0x80, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0xc0, 0x20, 0x18, 0x04, 0x02, 0x01, 0x80, 0x80, 0x00, 0x40, 0x40, 0xc0, 0x40, 0x30,
0x90, 0x88, 0x44, 0x42, 0x42, 0x41, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x20, 0x20, 0x20,
0x20, 0x20, 0x20, 0x20, 0x20, 0x10, 0x10, 0xf0, 0x88, 0x08, 0x04, 0x02, 0x01, 0x00, 0x00, 0x00,
0x00, 0x00, 0x01, 0x03, 0x06, 0x08, 0x71, 0x81, 0x02, 0x06, 0x04, 0x08, 0x10, 0x20, 0xc0, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x1f, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x03, 0x82, 0xe2, 0x39, 0x09, 0x05, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x39,
0x1c, 0x00, 0x00, 0x00, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
0xe0, 0xf0, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x06, 0x18, 0x20, 0x40, 0x80, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x20, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x01,
0x06, 0x38, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x40, 0x40, 0x80, 0x80, 0x80, 0xe0, 0xf8, 0xf8, 0xf0,
0xe0, 0xe0, 0xc0, 0xc0, 0x80, 0x01, 0x03, 0x1f, 0x0f, 0x07, 0x00, 0x80, 0xc0, 0x40, 0x20, 0x30,
0x18, 0x0e, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02,
0x02, 0x04, 0x08, 0x08, 0x10, 0x08, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0d, 0xf3, 0x06, 0x0c, 0x10,
0x20, 0x20, 0xc1, 0xff, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x7f, 0x3f, 0x3f, 0x3f,
0x1f, 0x1f, 0x0f, 0x07, 0x01, 0x02, 0x02, 0xfa, 0x06, 0x06, 0x03, 0x03, 0x02, 0x02, 0x04, 0x18,
0xf0, 0x80, 0xc0, 0x60, 0x20, 0x10, 0x18, 0x0c, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x10, 0x10,
0x10, 0x10, 0x10, 0x10, 0x20, 0xa0, 0xa0, 0x20, 0x20, 0x20, 0x20, 0x40, 0x40, 0x40, 0x40, 0x40,
0x40, 0x40, 0x80, 0x40, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x80, 0xe0, 0xf0, 0xf8, 0xf9, 0xe1, 0x81, 0x02, 0x02, 0x02, 0x03, 0x01, 0x01,
0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x7c, 0x7f, 0x7f, 0x7f, 0x3f, 0x3f, 0x3e, 0x1e, 0x1e, 0x1c, 0x0c, 0x00, 0xf8,
0x8e, 0x03, 0x01, 0x01, 0x01, 0x03, 0x06, 0x9c, 0xf0, 0x80, 0xc0, 0x40, 0x60, 0x20, 0x30, 0x10,
0x08, 0x0d, 0x07, 0x03, 0x03, 0x02, 0x02, 0x02, 0x02, 0x04, 0x04, 0x04, 0x04, 0x07, 0x04, 0x00,
0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x10, 0x10, 0x10, 0x10, 0x10,
0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x08, 0x1c, 0x1f, 0x1f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x30, 0x38, 0x38, 0x78, 0x7c, 0x7e, 0xff, 0xff, 0xff, 0xfc,
0xf1, 0xe1, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x03, 0x07, 0x06, 0x0e,
0x1e, 0x3e, 0x3e, 0x7e, 0x7e, 0x1e, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
		}
	};

	//assumes 1 frame prep stage
	void animation_phase(void) {
		if(get_current_wpm() <= IDLE_SPEED){
			current_idle_frame = (current_idle_frame + 1) % IDLE_FRAMES;
			oled_write_raw_P(idle[abs((IDLE_FRAMES-1)-current_idle_frame)], ANIM_SIZE);
		}
		else {
			// if  is true 2 frames in a row make it false;
			if(left && !lastl && right && !lastr) {
				oled_write_raw_P(tap[2], ANIM_SIZE);
			}
			else if (left && !lastl) {
				oled_write_raw_P(tap[0], ANIM_SIZE);
			}
			else if (right && !lastr) {
				oled_write_raw_P(tap[1], ANIM_SIZE);
			}
			else {
				oled_write_raw_P(prep[0], ANIM_SIZE);
				left = false;
				right = false;
			}
			lastl = left;
			lastr = right;
			left = false;
			right = false;
		}
	}
	if(get_current_wpm() != 000) {
		oled_on(); // not essential but turns on animation OLED with any alpha keypress
		if(timer_elapsed32(anim_timer) > ANIM_FRAME_DURATION) {
			anim_timer = timer_read32();
			animation_phase();
		}
		anim_sleep = timer_read32();
	} else {
		if(timer_elapsed32(anim_sleep) > OLED_TIMEOUT) {
			oled_off();
		} else {
			if(timer_elapsed32(anim_timer) > ANIM_FRAME_DURATION) {
				anim_timer = timer_read32();
				animation_phase();
			}
		}
	}
}
bool wpm_keycode_user(uint16_t keycode) {
	return true;
}
static void render_logo(void) {
	oled_set_cursor(0,0);
	oled_write_P(PSTR(" play asc "), false);
	oled_set_cursor(3,3);
	oled_write_P(PSTR(" play asc "), false);
	oled_set_cursor(6,6);
	oled_write_P(PSTR(" play asc "), false);
}
bool oled_task_user(void) {
	if (is_keyboard_master()) {
		render_anim();
		oled_set_cursor(0,7);
        uint8_t n = get_current_wpm();
        wpm_str[3] = '\0';
        wpm_str[2] = '0' + n % 10;
        wpm_str[1] = '0' + (n /= 10) % 10;
        wpm_str[0] = '0' + n / 10 ;
        oled_write_P(PSTR("WPM: "), false);
        oled_write(wpm_str, false);
        oled_write_P(PSTR("   "), false);
    		switch (get_highest_layer(layer_state)) {
			case _QWERTY:
				oled_write_ln_P(PSTR("base"), false);
				break;
			case _LOWER:
				oled_write_ln_P(PSTR("LOWER"), false);
				break;
			case _RAISE:
				oled_write_ln_P(PSTR("RAISE"), false);
				break;
			case _ADJUST:
				oled_write_ln_P(PSTR("ADJUST"), false);
				break;
			default:
				oled_write_ln_P(PSTR("Undefined"), false);
		}
	} else {
		render_logo();
		oled_scroll_left();

	}
    return false;
}
#endif
