/*
  Copyright (c) 2020 Fred Silberberg

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in all
  copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/

#include QMK_KEYBOARD_H
#include "debug.h"
#include "action_layer.h"
#include "version.h"
#include "333fred.h"

// NOTE: Cells marked with ACCESS must remain transparent, they're the keys that actually get to that layer

const uint16_t PROGMEM keymaps[][MATRIX_ROWS][MATRIX_COLS] = {
/* Keymap 0: Basic layer
 *
 * ,--------------------------------------------------.           ,--------------------------------------------------.
 * |   `    |   1  |   2  |   3  |   4  |   5  |  F5  |           |  F6  |   6  |   7  |   8  |   9  |   0  |   -    |
 * |--------+------+------+------+------+-------------|           |------+------+------+------+------+------+--------|
 * | TAB    |   Q  |   W  |   E  |   R  |   T  | GAME |           |GAMEA |   Y  |   U  |   I  |   O  |   P  |   \    |
 * |--------+------+------+------+------+------|      |           |      |------+------+------+------+------+--------|
 * | Esc    |   A  |   S  |   D  |   F  |   G  |------|           |------|   H  |   J  |   K  |   L  |   ;  |   '    |
 * |--------+------+------+------+------+------| SYMB |           |MEDIA |------+------+------+------+------+--------|
 * | LShift |Z/Ctrl|   X  |   C  |   V  |   B  |      |           |      |   N  |   M  |   ,  |   .  |//Ctrl| RShift |
 * `--------+------+------+------+------+-------------'           `-------------+------+------+------+------+--------'
 *   |LCTRL |  F4  |  F5  | LGUI | LALT |                                       | Left | Down |  =   | Right| RGUI |
 *   `----------------------------------'                                       `----------------------------------'
 *                                        ,-------------.       ,-------------.
 *                                        | Copy | Paste|       | Copy | Paste  |
 *                                 ,------|------|------|       |------+--------+------.
 *                                 |      |      | PgUp |       |  Up  |        |      |
 *                                 | Bcksp|OSL(2)|------|       |------|  Ent   |Space |
 *                                 |      |  VIM | Del  |       | Down |        |      |
 *                                 `--------------------'       `----------------------'
 */
[BASE] = LAYOUT_ergodox_pretty(  // layer 0 : default
        // left hand
        KC_GRV,        KC_1,        KC_2,  KC_3,    KC_4,    KC_5,    KC_F5,                                      KC_F6,          KC_6, KC_7,    KC_8,    KC_9,   KC_0,           KC_MINS,
        KC_TAB,        KC_Q,        KC_W,  KC_E,    KC_R,    KC_T,    TG(GAME),                                   TG(GAME_ARROW), KC_Y, KC_U,    KC_I,    KC_O,   KC_P,           KC_BSLS,
        KC_ESC,        KC_A,        KC_S,  KC_D,    KC_F,    KC_G,                                                                KC_H, KC_J,    KC_K,    KC_L,   KC_SCLN,        KC_QUOT,
        OSM(MOD_LSFT), CTL_T(KC_Z), KC_X,  KC_C,    KC_V,    KC_B,    TG(SYMB),                                   MO(MDIA),       KC_N, KC_M,    KC_COMM, KC_DOT, CTL_T(KC_SLSH), OSM(MOD_RSFT),
        OSM(MOD_LCTL), KC_F4,       KC_F5, KC_LGUI, KC_LALT,                                                                            KC_LEFT, KC_DOWN, KC_EQL, KC_RIGHT,       KC_RGUI,
                                                                      LCTL(KC_C),     LCTL(KC_V),     LCTL(KC_C), LCTL(KC_V),
                                                                                      KC_PGUP,        KC_UP,
                                                             KC_BSPC, TD(TD_SYM_VIM), KC_DEL,         KC_DOWN,    KC_ENT,         KC_SPC
    ),

/* Keymap 2: Codeflow Layer
 *
 * ,--------------------------------------------------.           ,--------------------------------------------------.
 * |        |      |      |      |      |      |      |           |      |      |      |      |      |      |        |
 * |--------+------+------+------+------+-------------|           |------+------+------+------+------+------+--------|
 * |        |      |      |      |      |      |      |           |      |      |      |      |      |      |        |
 * |--------+------+------+------+------+------|      |           |      |------+------+------+------+------+--------|
 * |        |      |      |      |      |      |------|           |------|      |      |      |      |      |        |
 * |--------+------+------+------+------+------|      |           |      |------+------+------+------+------+--------|
 * |        |      |      |      |      |      |      |           |      |      |      |      |      |      |        |
 * `--------+------+------+------+------+-------------'           `-------------+------+------+------+------+--------'
 *   |      |      |      |      |      |                                       |  F7  |  F8  |      |      |      |
 *   `----------------------------------'                                       `----------------------------------'
 *                                        ,-------------.       ,---------------.
 *                                        |      |ACCESS|       |      |        |
 *                                 ,------|------|------|       |------+--------+------.
 *                                 |      |      |      |       |      |        |      |
 *                                 |      |      |------|       |------|        |      |
 *                                 |      |      |      |       |      |        |      |
 *                                 `--------------------'       `----------------------'
 */
// If it accepts an argument (i.e, is a function), it doesn't need KC_.
// Otherwise, it needs KC_*
[CODEFLOW] = LAYOUT_ergodox_pretty(  // layer 1 : code
        // left hand
        _______, _______, _______, _______, _______, _______, _______,  _______, _______, _______, _______, _______, _______, _______,
        _______, _______, _______, _______, _______, _______, _______,  _______, _______, _______, _______, _______, _______, _______,
        _______, _______, _______, _______, _______, _______,                    _______, _______, _______, _______, _______, _______,
        _______, _______, _______, _______, _______, _______, _______,  _______, _______, _______, _______, _______, _______, _______,
        _______, _______, _______, _______, _______,                                      KC_F7,   KC_F8,   _______, _______, _______,
                                                     _______, _______,  _______, _______,
                                                              _______,  _______,
                                            _______, _______, _______,  _______, _______, _______
    ),
/* Keymap 3: Symbol Layer
 *
 * ,---------------------------------------------------.           ,--------------------------------------------------.
 * |         |  F1  |  F2  |  F3  |  F4  |  F5  |      |           |      |  F6  |  F7  |  F8  |  F9  |  F10 |   F11  |
 * |---------+------+------+------+------+------+------|           |------+------+------+------+------+------+--------|
 * |         |   !  |   @  |   (  |   )  |   |  |      |           |      |   Up |   7  |   8  |   9  |   *  |   F12  |
 * |---------+------+------+------+------+------|      |           |      |------+------+------+------+------+--------|
 * |         |   #  |   $  |   {  |   }  |   `  |------|           |------| Down |   4  |   5  |   6  |   +  |        |
 * |---------+------+------+------+------+------|ACCESS|           |      |------+------+------+------+------+--------|
 * |         |   %  |   ^  |   [  |   ]  |   ~  |      |           |      |   &  |   1  |   2  |   3  |   \  |        |
 * `---------+------+------+------+------+-------------'           `-------------+------+------+------+------+--------'
 *   |       |      |      |      |      |                                       |   0  |   0  |   .  |   =  |      |
 *   `-----------------------------------'                                       `----------------------------------'
 *                                        ,-------------.       ,-------------.
 *                                        |      |      |       |      |      |
 *                                 ,------|------|------|       |------+------+------.
 *                                 |      |      |APScr |       |      |      |      |
 *                                 |      |ACCESS|------|       |------|      |      |
 *                                 |      |      | PScr |       |      |      |      |
 *                                 `--------------------'       `--------------------'
 */
[SYMB] = LAYOUT_ergodox_pretty(
       // left hand
       _______, KC_F1,   KC_F2,   KC_F3,   KC_F4,   KC_F5,   _______,                              _______, KC_F6,   KC_F7,  KC_F8,   KC_F9,   KC_F10,  KC_F11,
       _______, KC_EXLM, KC_AT,   KC_LPRN, KC_RPRN, KC_PIPE, _______,                              _______, KC_UP,   KC_7,   KC_8,    KC_9,    KC_ASTR, KC_F12,
       _______, KC_HASH, KC_DLR,  KC_LCBR, KC_RCBR, KC_GRV,                                                 KC_DOWN, KC_4,   KC_5,    KC_6,    KC_PLUS, _______,
       _______, KC_PERC, KC_CIRC, KC_LBRC, KC_RBRC, KC_TILD, _______,                              _______, KC_AMPR, KC_1,   KC_2,    KC_3,    KC_BSLS, _______,
       _______, _______, _______, _______, _______,                                                                  KC_0,   KC_0,    KC_DOT,  KC_EQL,  _______,
                                                             _______, _______,            _______, _______,
                                                                      PSCREEN_APP,     _______,
                                                    _______, _______, KC_PSCR,            _______, _______, _______
),
/* Keymap 3: Media and mouse keys
 *
 * ,--------------------------------------------------.           ,--------------------------------------------------.
 * |        |      |      |      |      |      |      |           |      |      |      |      |      |      |        |
 * |--------+------+------+------+------+-------------|           |------+------+------+------+------+------+--------|
 * |        |      |      | MsUp |      |      |      |           |      |      |      |      |      |      |        |
 * |--------+------+------+------+------+------|      |           |      |------+------+------+------+------+--------|
 * |        |      |MsLeft|MsDown|MsRght|      |------|           |------|      |      |      |      |      |        |
 * |--------+------+------+------+------+------|      |           |ACCESS|------+------+------+------+------+--------|
 * |        |      |      |      |      |      |      |           |      |      |      |      |      |      |        |
 * `--------+------+------+------+------+-------------'           `-------------+------+------+------+------+--------'
 *   |      |      |      | Lclk | Rclk |                                       |      |      |      |      |      |
 *   `----------------------------------'                                       `----------------------------------'
 *                                        ,-------------.       ,-------------.
 *                                        | Back+| Back-|       | Vol+ |      |
 *                                 ,------|------|------|       |------+------+------.
 *                                 |      |      |BackTg|       | Vol- |      |      |
 *                                 |      |      |------|       |------| PL/PS| Next |
 *                                 |      |      |      |       | Back |      |      |
 *                                 `--------------------'       `--------------------'
 */
[MDIA] = LAYOUT_ergodox_pretty(
       _______, _______, _______, _______, _______, _______, _______,                       _______, _______, _______, _______, _______, _______, _______,
       _______, _______, _______, KC_MS_U, _______, _______, _______,                       _______, _______, _______, _______, _______, _______, _______,
       _______, _______, KC_MS_L, KC_MS_D, KC_MS_R, _______,                                         _______, _______, _______, _______, _______, _______,
       _______, _______, _______, _______, _______, _______, _______,                       _______, _______, _______, _______, _______, _______, _______,
       _______, _______, _______, KC_BTN1, KC_BTN2,                                                           _______, _______, _______, _______, _______,
                                                             BL_UP,   BL_DOWN,     KC_VOLU, _______,
                                                                      BL_TOGG,     KC_VOLD,
                                                    _______, _______, _______,     KC_MPRV, KC_MPLY, KC_MNXT
),
/* Keymap 4: Movement
 *
 * ,--------------------------------------------------.           ,--------------------------------------------------.
 * |        |      |      |      |      |      |      |           |      |      |      |      |      |      |        |
 * |--------+------+------+------+------+-------------|           |------+------+------+------+------+------+--------|
 * |        |KOpen |KType |TERMCP|TERMPT|      |      |           |      | Copy |      |      |      | Paste|        |
 * |--------+------+------+------+------+------|      |           |      |------+------+------+------+------+--------|
 * |        |DLeft |DRight|LCTRL | LGUI |      |------|           |------| Left | Down |  Up  | Right|      |        |
 * |--------+------+------+------+------+------|      |           |      |------+------+------+------+------+--------|
 * |        |SFT_TB| TAB  |      |      |      |      |           |      |      |      |      |      |      |        |
 * `--------+------+------+------+------+-------------'           `-------------+------+------+------+------+--------'
 *   |      |      |      |      |      |                                       |      |      |      |      |      |
 *   `----------------------------------'                                       `----------------------------------'
 *                                        ,-------------.       ,-------------.
 *                                        |      |      |       | Home | End  |
 *                                 ,------|------|------|       |------+------+------.
 *                                 |      |      |      |       |      |      |      |
 *                                 |      |ACCESS|------|       |------|      |      |
 *                                 |      |      |      |       |      |      |      |
 *                                 `--------------------'       `--------------------'
 */
// MEDIA AND MOUSE
[VIM] = LAYOUT_ergodox_pretty(
       _______, _______, _______, _______, _______, _______, _______,     _______,  _______, _______, _______, _______,  _______, _______,
       _______, _______, _______, _______, _______, _______, _______,     _______,  _______, _______, _______, _______,  _______, _______,
       _______, DLEFT,   DRIGHT,  KC_LCTL, KC_LGUI, _______,                        KC_LEFT, KC_DOWN, KC_UP,   KC_RIGHT, _______, _______,
       _______, _______, KC_TAB,  _______, _______, _______, _______,     _______,  _______, _______, _______, _______,  _______, _______,
       _______, _______, _______, _______, _______,                                          _______, _______, _______,  _______, _______,
                                                    _______, _______,     KC_HOME, KC_END,
                                                             _______,     _______,
                                           _______, _______, _______,     _______, _______, _______
),
/* Keymap 1: Game Layer
 *
 * ,--------------------------------------------------.           ,--------------------------------------------------.
 * |  ESC   |      |      |      |      |      |      |           |      |      |      |      |      |      |        |
 * |--------+------+------+------+------+-------------|           |------+------+------+------+------+------+--------|
 * |        |      |      |      |      |      |ACCESS|           |      |      |      |      |      |      |        |
 * |--------+------+------+------+------+------|      |           |      |------+------+------+------+------+--------|
 * |  CTRL  |      |      |      |      |      |------|           |------|      |      |      |      |      |        |
 * |--------+------+------+------+------+------|      |           |      |------+------+------+------+------+--------|
 * | SHIFT  |  Z   |      |      |      |      |      |           |      |      |      |      |      |      |        |
 * `--------+------+------+------+------+-------------'           `-------------+------+------+------+------+--------'
 *   |ENTER |      |      | LOCK | BSPC |                                       |      |      |      |      |      |
 *   `----------------------------------'                                       `----------------------------------'
 *                                        ,-------------.       ,---------------.
 *                                        |  F5  |  F6  |       |      |        |
 *                                 ,------|------|------|       |------+--------+------.
 *                                 |      |      |      |       |      |        |      |
 *                                 | ALT  | SPC  |------|       |------|        |      |
 *                                 |      |      | SYMB |       |      |        |      |
 *                                 `--------------------'       `----------------------'
 */
// If it accepts an argument (i.e, is a function), it doesn't need KC_.
// Otherwise, it needs KC_*
[GAME] = LAYOUT_ergodox_pretty(  // layer 1 : code
        // left hand
        KC_ESC,  _______, _______, _______, _______, _______, _______,                            _______, _______, _______, _______, _______, _______, _______,
        _______, _______, _______, _______, _______, _______, _______,                            _______, _______, _______, _______, _______, _______, _______,
        KC_LCTL, _______, _______, _______, _______, _______,                                              _______, _______, _______, _______, _______, _______,
        KC_LSFT, KC_Z,    _______, _______, _______, _______, _______,                            _______, _______, _______, _______, _______, _______, _______,
        KC_ENT,  _______, _______, QK_LOCK, KC_BSPC,                                                                _______, _______, _______, _______, _______,
                                                              KC_F5,   KC_F6,         LCTL(KC_C), LCTL(KC_V),
                                                                       _______,       KC_UP,
                                                     KC_LALT, KC_SPC,  OSM(SYMB),     KC_DOWN,    _______, _______
    ),
/* Keymap 1: Game Arrow Layer
 *
 * ,--------------------------------------------------.           ,--------------------------------------------------.
 * |  ESC   |      |      |      |      |      |      |           |      |      |      |      |      |      |        |
 * |--------+------+------+------+------+-------------|           |------+------+------+------+------+------+--------|
 * |        |      |  Up  |      |      |      |      |           |ACCESS|      |      |      |      |      |        |
 * |--------+------+------+------+------+------|      |           |      |------+------+------+------+------+--------|
 * |  CTRL  | Left | Down |Right |      |      |------|           |------|      |      |      |      |      |        |
 * |--------+------+------+------+------+------|      |           |      |------+------+------+------+------+--------|
 * | SHIFT  |  Z   |      |      |      |      |      |           |      |      |      |      |      |      |        |
 * `--------+------+------+------+------+-------------'           `-------------+------+------+------+------+--------'
 *   |ENTER |      |      | LOCK | BSPC |                                       |      |      |      |      |      |
 *   `----------------------------------'                                       `----------------------------------'
 *                                        ,-------------.       ,---------------.
 *                                        |  F5  |  F6  |       |      |        |
 *                                 ,------|------|------|       |------+--------+------.
 *                                 |      |      |      |       |      |        |      |
 *                                 | ALT  | SPC  |------|       |------|        |      |
 *                                 |      |      | SYMB |       |      |        |      |
 *                                 `--------------------'       `----------------------'
 */
// If it accepts an argument (i.e, is a function), it doesn't need KC_.
// Otherwise, it needs KC_*
[GAME_ARROW] = LAYOUT_ergodox_pretty(  // layer 1 : code
        // left hand
        KC_ESC,  _______, _______, _______,  _______, _______, _______,                            _______, _______, _______, _______, _______, _______, _______,
        _______, _______, KC_UP,   _______,  _______, _______, _______,                            _______, _______, _______, _______, _______, _______, _______,
        KC_LCTL, KC_LEFT, KC_DOWN, KC_RIGHT, _______, _______,                                              _______, _______, _______, _______, _______, _______,
        KC_LSFT, KC_Z,    _______, _______,  _______, _______, _______,                            _______, _______, _______, _______, _______, _______, _______,
        KC_ENT,  _______, _______, QK_LOCK,  KC_BSPC,                                                                _______, _______, _______, _______, _______,
                                                               KC_F5,   KC_F6,         LCTL(KC_C), LCTL(KC_V),
                                                                        _______,       KC_UP,
                                                      KC_LALT, KC_SPC,  OSM(SYMB),     KC_DOWN,    _______, _______
    )
};

bool process_record_user(uint16_t keycode, keyrecord_t *record) {
  tap_dance_process_keycode(keycode);
  return true;
}

// Runs constantly in the background, in a loop.
void matrix_scan_user(void) {
    ergodox_board_led_on();
    ergodox_led_all_on();
}

#ifdef ST7565_ENABLE

void st7565_task_user(void) {
    // The colors will need to be ported over to the quantum painter API when
    // https://github.com/qmk/qmk_firmware/pull/10174 is merged.

    st7565_clear();
    switch (get_highest_layer(layer_state)) {
        case BASE:
            //state->target_lcd_color = LCD_COLOR(84, saturation, 0xFF);
            st7565_write_P(PSTR("Default\n"), false);
            break;
        case CODEFLOW:
            //state->target_lcd_color = LCD_COLOR(216, 90, 0xFF);
            st7565_write_P(PSTR("Code\n"), false);
            break;
        case SYMB:
            //state->target_lcd_color = LCD_COLOR(168, saturation, 0xFF);
            st7565_write_P(PSTR("Symbol\n"), false);
            break;
        case MDIA:
            //state->target_lcd_color = LCD_COLOR(0, saturation, 0xFF);
            st7565_write_P(PSTR("Media\n"), false);
            break;
        case VIM:
            //state->target_lcd_color = LCD_COLOR(140, 100, 60);
            st7565_write_P(PSTR("Movement\n"), false);
            break;
        case GAME:
            //state->target_lcd_color = LCD_COLOR(0, 255, 60);
            st7565_write_P(PSTR("Game\n"), false);
            break;
        case GAME_ARROW:
            //state->target_lcd_color = LCD_COLOR(0, 255, 60);
            st7565_write_P(PSTR("Game Arrow\n"), false);
            break;
    }
}

#endif
