/* Copyright 2021 Simon Arlott
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gtest/gtest.h"

#include "debounce_test_common.h"

TEST_F(DebounceTest, OneKeyShort1) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
        /* Release key after 1ms delay */
        {1, {{0, 1, UP}}, {}},

        /*
         * Until the eager timer on DOWN is observed to finish, the defer timer
         * on UP can't start. There's no workaround for this because it's not
         * possible to debounce an event that isn't being tracked.
         *
         * sym_defer_pk has the same problem but the test has to track that the
         * key changed state so the DOWN timer is always allowed to finish
         * before starting the UP timer.
         */
        {5, {}, {}},

        {10, {}, {{0, 1, UP}}}, /* 5ms+5ms after DOWN at time 0 */
        /* Press key again after 1ms delay */
        {11, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyShort2) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
        /* Release key after 2ms delay */
        {2, {{0, 1, UP}}, {}},

        {5, {}, {}}, /* See OneKeyShort1 */

        {10, {}, {{0, 1, UP}}}, /* 5ms+5ms after DOWN at time 0 */
        /* Press key again after 1ms delay */
        {11, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyShort3) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
        /* Release key after 3ms delay */
        {3, {{0, 1, UP}}, {}},

        {5, {}, {}}, /* See OneKeyShort1 */

        {10, {}, {{0, 1, UP}}}, /* 5ms+5ms after DOWN at time 0 */
        /* Press key again after 1ms delay */
        {11, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyShort4) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
        /* Release key after 4ms delay */
        {4, {{0, 1, UP}}, {}},

        {5, {}, {}}, /* See OneKeyShort1 */

        {10, {}, {{0, 1, UP}}}, /* 5ms+5ms after DOWN at time 0 */
        /* Press key again after 1ms delay */
        {11, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyShort5) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        /* Release key after 5ms delay */
        {5, {{0, 1, UP}}, {}},

        {10, {}, {{0, 1, UP}}}, /* 5ms+5ms after DOWN at time 0 */
        /* Press key again after 1ms delay */
        {11, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyShort6) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        /* Release key after 6ms delay */
        {6, {{0, 1, UP}}, {}},

        {11, {}, {{0, 1, UP}}}, /* 5ms after UP at time 6 */
        /* Press key again after 1ms delay */
        {12, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyShort7) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        /* Release key after 7ms delay */
        {7, {{0, 1, UP}}, {}},

        {12, {}, {{0, 1, UP}}}, /* 5ms after UP at time 7 */
        /* Press key again after 1ms delay */
        {13, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyShort8) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
        /* Release key after 1ms delay */
        {1, {{0, 1, UP}}, {}},

        {5, {}, {}}, /* See OneKeyShort1 */

        {10, {}, {{0, 1, UP}}}, /* 5ms after UP at time 7 */
        /* Press key again after 0ms delay (scan 2) */
        {10, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyShort9) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
        /* Release key after 1ms delay */
        {1, {{0, 1, UP}}, {}},

        {5, {}, {}}, /* See OneKeyShort1 */

        /* Press key again after 0ms delay (same scan) before debounce finishes */
        {10, {{0, 1, DOWN}}, {}},
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyBouncing1) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
        {1, {{0, 1, UP}}, {}},
        {2, {{0, 1, DOWN}}, {}},
        {3, {{0, 1, UP}}, {}},
        {4, {{0, 1, DOWN}}, {}},
        {5, {{0, 1, UP}}, {}},
        {6, {{0, 1, DOWN}}, {}},
        {7, {{0, 1, UP}}, {}},
        {8, {{0, 1, DOWN}}, {}},
        {9, {{0, 1, UP}}, {}},
        {10, {{0, 1, DOWN}}, {}},
        {11, {{0, 1, UP}}, {}},
        {12, {{0, 1, DOWN}}, {}},
        {13, {{0, 1, UP}}, {}},
        {14, {{0, 1, DOWN}}, {}},
        {15, {{0, 1, UP}}, {}},

        {20, {}, {{0, 1, UP}}},
        /* Press key again after 1ms delay */
        {21, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyBouncing2) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
        /* Change twice in the same time period */
        {1, {{0, 1, UP}}, {}},
        {1, {{0, 1, DOWN}}, {}},
        /* Change three times in the same time period */
        {2, {{0, 1, UP}}, {}},
        {2, {{0, 1, DOWN}}, {}},
        {2, {{0, 1, UP}}, {}},
        /* Change twice in the same time period */
        {6, {{0, 1, DOWN}}, {}},
        {6, {{0, 1, UP}}, {}},
        /* Change three times in the same time period */
        {7, {{0, 1, DOWN}}, {}},
        {7, {{0, 1, UP}}, {}},
        {7, {{0, 1, DOWN}}, {}},
        /* Change twice in the same time period */
        {8, {{0, 1, UP}}, {}},
        {8, {{0, 1, DOWN}}, {}},
        /* Change three times in the same time period */
        {9, {{0, 1, UP}}, {}},
        {9, {{0, 1, DOWN}}, {}},
        {9, {{0, 1, UP}}, {}},

        {14, {}, {{0, 1, UP}}},
        /* Press key again after 1ms delay */
        {15, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyLong) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        {25, {{0, 1, UP}}, {}},

        {30, {}, {{0, 1, UP}}},

        {50, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        {75, {{0, 1, UP}}, {}},

        {80, {}, {{0, 1, UP}}},

        {100, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    runEvents();
}

TEST_F(DebounceTest, TwoKeysShort) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
        {1, {{0, 2, DOWN}}, {{0, 2, DOWN}}},
        /* Release key after 2ms delay */
        {2, {{0, 1, UP}}, {}},
        {3, {{0, 2, UP}}, {}},

        {5, {}, {}}, /* See OneKeyShort1 */
        {6, {}, {}}, /* See OneKeyShort1 */

        {10, {}, {{0, 1, UP}}}, /* 5ms+5ms after DOWN at time 0 */
        /* Press key again after 1ms delay */
        {11, {{0, 1, DOWN}}, {{0, 1, DOWN}, {0, 2, UP}}}, /* 5ms+5ms after DOWN at time 0 */
        {12, {{0, 2, DOWN}}, {{0, 2, DOWN}}},             /* 5ms+5ms after DOWN at time 0 */
    });
    runEvents();
}

TEST_F(DebounceTest, OneKeyDelayedScan1) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        /* Processing is very late, immediately release key */
        {300, {{0, 1, UP}}, {}},

        {305, {}, {{0, 1, UP}}},
    });
    time_jumps_ = true;
    runEvents();
}

TEST_F(DebounceTest, OneKeyDelayedScan2) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        /* Processing is very late, immediately release key */
        {300, {{0, 1, UP}}, {}},

        /* Processing is very late again */
        {600, {}, {{0, 1, UP}}},
    });
    time_jumps_ = true;
    runEvents();
}

TEST_F(DebounceTest, OneKeyDelayedScan3) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        /* Processing is very late */
        {300, {}, {}},
        /* Release key after 1ms */
        {301, {{0, 1, UP}}, {}},

        {306, {}, {{0, 1, UP}}},
    });
    time_jumps_ = true;
    runEvents();
}

TEST_F(DebounceTest, OneKeyDelayedScan4) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        /* Processing is very late */
        {300, {}, {}},
        /* Release key after 1ms */
        {301, {{0, 1, UP}}, {}},

        /* Processing is very late again */
        {600, {}, {{0, 1, UP}}},
    });
    time_jumps_ = true;
    runEvents();
}

TEST_F(DebounceTest, OneKeyDelayedScan5) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        {5, {{0, 1, UP}}, {}},

        /* Processing is very late */
        {300, {}, {{0, 1, UP}}},
        /* Immediately press key again */
        {300, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    time_jumps_ = true;
    runEvents();
}

TEST_F(DebounceTest, OneKeyDelayedScan6) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        {5, {{0, 1, UP}}, {}},

        /* Processing is very late */
        {300, {}, {{0, 1, UP}}},

        /* Press key again after 1ms */
        {301, {{0, 1, DOWN}}, {{0, 1, DOWN}}},
    });
    time_jumps_ = true;
    runEvents();
}

TEST_F(DebounceTest, OneKeyDelayedScan7) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        {5, {{0, 1, UP}}, {}},

        /* Press key again before debounce expires */
        {300, {{0, 1, DOWN}}, {}},
    });
    time_jumps_ = true;
    runEvents();
}

TEST_F(DebounceTest, OneKeyDelayedScan8) {
    addEvents({
        /* Time, Inputs, Outputs */
        {0, {{0, 1, DOWN}}, {{0, 1, DOWN}}},

        /* Processing is a bit late */
        {50, {}, {}},
        /* Release key after 1ms */
        {51, {{0, 1, UP}}, {}},

        /* Processing is a bit late again */
        {100, {}, {{0, 1, UP}}},
    });
    time_jumps_ = true;
    runEvents();
}
