#include "encoder_plugin_vorbis.h"

#include <vorbis/vorbisenc.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <inttypes.h>

#include "version.h"
#include "vorbis_mappings.h"

#define BLOCK_SIZE 64 /* lowest allowed vorbis blocksize */

#define LOG_PREFIX "[encoder:vorbis]"
#include "logger.h"

#define MAX_CHANNELS 8

#define LOGS(s, a) log_error(s, (int)(a).len, (char *)(a).x);
#define LOGERRNO(s) log_error(s": %s", strerror(errno))
#define LOGSERRNO(s,a) log_error(s": %s", (int)a.len, (char *)(a).x, strerror(errno))

static STRBUF_CONST(plugin_name,"vorbis");

typedef enum BITRATE_MODE {
    BITRATE_MODE_ABR = 0,
    BITRATE_MODE_CBR = 1,
    BITRATE_MODE_VBR_QUALITY = 2,
    BITRATE_MODE_VBR_TARGET = 3,
} BITRATE_MODE;

struct encoder_plugin_vorbis_userdata {
    vorbis_info v_info;
    vorbis_dsp_state v_dsp;
    vorbis_block v_block;
    vorbis_comment vc;

    int info_init;
    int dsp_init;
    int block_init;
    int vc_init;

    BITRATE_MODE mode;
    int quality; /* [-1, 10], divided by 10.0f to get -.1, 1.0 */
    int min_bitrate;
    int target_bitrate;
    int max_bitrate;

    strbuf name;
    packet_source me;
    frame buffer;
};

typedef struct encoder_plugin_vorbis_userdata encoder_plugin_vorbis_userdata;

static size_t encoder_plugin_vorbis_size(void) {
    return sizeof(encoder_plugin_vorbis_userdata);
}

static int encoder_plugin_vorbis_create(void* ud) {
    encoder_plugin_vorbis_userdata* userdata = (encoder_plugin_vorbis_userdata*)ud;

    /* default to 128kbps ABR */
    userdata->mode = BITRATE_MODE_ABR;
    userdata->min_bitrate = -1;
    userdata->max_bitrate = -1;
    userdata->target_bitrate = 128000;
    userdata->quality = 4; /* about 128kbps */
    userdata->info_init = 0;
    userdata->dsp_init = 0;
    userdata->block_init = 0;
    userdata->vc_init = 0;

    strbuf_init(&userdata->name);
    frame_init(&userdata->buffer);
    userdata->me = packet_source_zero;

    return 0;
}

static int encoder_plugin_vorbis_reset(void* ud) {
    encoder_plugin_vorbis_userdata* userdata = (encoder_plugin_vorbis_userdata*)ud;

    if(userdata->info_init) {
        vorbis_info_clear(&userdata->v_info);
        userdata->info_init = 0;
    }

    if(userdata->dsp_init) {
        vorbis_dsp_clear(&userdata->v_dsp);
        userdata->dsp_init = 0;
    }

    if(userdata->block_init) {
        vorbis_block_clear(&userdata->v_block);
        userdata->block_init = 0;
    }

    if(userdata->vc_init) {
        vorbis_comment_clear(&userdata->vc);
        userdata->vc_init = 0;
    }

    strbuf_free(&userdata->name);
    frame_free(&userdata->buffer);
    packet_source_free(&userdata->me);
    return 0;
}

static void encoder_plugin_vorbis_close(void* ud) {
    encoder_plugin_vorbis_userdata* userdata = (encoder_plugin_vorbis_userdata*)ud;
    encoder_plugin_vorbis_reset(userdata);
}

static int encoder_plugin_vorbis_config(void* ud, const strbuf* key, const strbuf* value) {
    encoder_plugin_vorbis_userdata* userdata = (encoder_plugin_vorbis_userdata*)ud;
    int mult = 1;

    if(strbuf_equals_cstr(key,"quality")) {
        errno = 0;
        userdata->quality = strbuf_strtol(value,10);
        if(errno != 0) {
            LOGSERRNO("error parsing quality value %.*s",(*value));
            return -1;
        }
        if(userdata->quality < -1 || userdata->quality > 10) {
            LOGS("invalid value for quality: %.*s (max 10)",(*value));
            return -1;
        }
        userdata->mode = BITRATE_MODE_VBR_QUALITY;
        return 0;
    }

    if(strbuf_equals_cstr(key,"target-bitrate") ||
       strbuf_equals_cstr(key,"target bitrate") ||
       strbuf_equals_cstr(key,"bitrate") ||
       strbuf_equals_cstr(key,"b")) {
        if(strbuf_caseends_cstr(value,"k")) {
            mult = 1000;
        }
        if(strbuf_caseends_cstr(value,"kbps")) {
            mult = 1000;
        }
        errno = 0;
        userdata->target_bitrate = strbuf_strtol(value,10);
        if(errno != 0) {
            LOGSERRNO("error parsing bitrate value %.*s",(*value));
            return -1;
        }
        userdata->target_bitrate *= mult;
        return 0;
    }

    if(strbuf_equals_cstr(key,"max-bitrate") || strbuf_equals_cstr(key,"max bitrate")) {
        if(strbuf_caseends_cstr(value,"k")) {
            mult = 1000;
        }
        if(strbuf_caseends_cstr(value,"kbps")) {
            mult = 1000;
        }
        errno = 0;
        userdata->max_bitrate = strbuf_strtol(value,10);
        if(errno != 0) {
            LOGSERRNO("error parsing bitrate value %.*s",(*value));
            return -1;
        }
        userdata->max_bitrate *= mult;
        return 0;
    }

    if(strbuf_equals_cstr(key,"min-bitrate") || strbuf_equals_cstr(key,"min bitrate")) {
        if(strbuf_caseends_cstr(value,"k")) {
            mult = 1000;
        }
        if(strbuf_caseends_cstr(value,"kbps")) {
            mult = 1000;
        }
        errno = 0;
        userdata->min_bitrate = strbuf_strtol(value,10);
        if(errno != 0) {
            LOGSERRNO("error parsing bitrate value %.*s",(*value));
            return -1;
        }
        userdata->min_bitrate *= mult;
        return 0;
    }

    if(strbuf_equals_cstr(key,"bitrate mode") || strbuf_equals_cstr(key,"bitrate-mode")) {
        if(strbuf_caseequals_cstr(value,"abr")) {
            userdata->mode = BITRATE_MODE_ABR;
        } else if(strbuf_caseequals_cstr(value,"cbr")) {
            userdata->mode = BITRATE_MODE_CBR;
        } else if(strbuf_caseequals_cstr(value,"vbr")) {
            userdata->mode = BITRATE_MODE_VBR_QUALITY;
        } else if(strbuf_caseequals_cstr(value,"quality")) {
            userdata->mode = BITRATE_MODE_VBR_QUALITY;
        } else if(strbuf_caseequals_cstr(value,"target")) {
            userdata->mode = BITRATE_MODE_VBR_TARGET;
        } else {
            LOGS("invalid value for mode: %.*s [abr | cbr | quality | target]",(*value));
            return -1;
        }
        return 0;
    }

    LOGS("unknown key: %.*s\n",*key);
    return -1;
}

static int encoder_plugin_vorbis_open(void* ud, const frame_source* source, const packet_receiver* dest) {
    int r = -1;
    unsigned int i = 0;
    uint8_t* dsi = NULL;
    size_t dsi_len = 0;
    encoder_plugin_vorbis_userdata* userdata = (encoder_plugin_vorbis_userdata*)ud;
    ogg_packet header_packet[3];

    int channels = 0;
    switch(source->channel_layout) {
        case LAYOUT_MONO: channels = 1; break;
        case LAYOUT_STEREO: channels = 2; break;
        case LAYOUT_3_0: channels = 3; break;
        case LAYOUT_4_0: channels = 4; break;
        case LAYOUT_5_0: channels = 5; break;
        case LAYOUT_5_1: channels = 6; break;
        case LAYOUT_6_1: channels = 7; break;
        case LAYOUT_7_1: channels = 8; break;
        default: {
            log_error("unsupported channel layout 0x%" PRIx64, source->channel_layout);
            return -1;
        }
    }

    userdata->buffer.format = SAMPLEFMT_FLOATP;
    userdata->buffer.channels = channels;
    userdata->buffer.duration = 0;
    userdata->buffer.sample_rate = source->sample_rate;

    if( (r = frame_ready(&userdata->buffer)) != 0) {
        LOGERRNO("error allocating samples buffer");
        return r;
    }

    vorbis_info_init(&userdata->v_info);
    userdata->info_init = 1;

    switch(userdata->mode) {
        case BITRATE_MODE_ABR: {
            r = vorbis_encode_init(&userdata->v_info, channels, source->sample_rate, userdata->max_bitrate, userdata->target_bitrate, userdata->min_bitrate);
            break;
        }
        case BITRATE_MODE_CBR: {
            r = vorbis_encode_init(&userdata->v_info, channels, source->sample_rate, userdata->target_bitrate, userdata->target_bitrate, userdata->target_bitrate);
            break;
        }
        case BITRATE_MODE_VBR_QUALITY: {
            r = vorbis_encode_init_vbr(&userdata->v_info, channels, source->sample_rate, ((float)userdata->quality) / 10.0f);
            break;
        }
        case BITRATE_MODE_VBR_TARGET: {
            if( (r = vorbis_encode_setup_managed(&userdata->v_info, channels, source->sample_rate, userdata->max_bitrate, userdata->target_bitrate, userdata->min_bitrate)) != 0) break;
            if( (r = vorbis_encode_ctl(&userdata->v_info, OV_ECTL_RATEMANAGE2_SET, NULL)) != 0) break;
            r = vorbis_encode_setup_init(&userdata->v_info);
            break;
        }
    }
    if(r != 0) {
        logs_error("error setting up vorbis bitrate");
        return r;
    }

    if( (r = vorbis_analysis_init(&userdata->v_dsp, &userdata->v_info)) != 0) {
        logs_error("error initializing vorbis analysis");
        return r;
    }
    userdata->dsp_init = 1;

    vorbis_comment_init(&userdata->vc);
    userdata->vc_init = 1;

    if( (r = vorbis_analysis_headerout(&userdata->v_dsp, &userdata->vc, &header_packet[0], &header_packet[1], &header_packet[2])) != 0) {
        logs_error("error calling vorbis_analysis_headerot");
        return r;
    }

    if( (r = vorbis_block_init(&userdata->v_dsp, &userdata->v_block)) != 0) {
        logs_error("error calling vorbis_block_init");
        return r;
    }
    userdata->block_init = 1;

    if( (r = strbuf_append_cstr(&userdata->name, vorbis_version_string())) != 0) {
        LOGERRNO("error appending name string");
        return r;
    }

    if( (r = strbuf_append_cstr(&userdata->name, ", ")) != 0) {
        LOGERRNO("error appending name string");
        return r;
    }

    if( (r = strbuf_append_cstr(&userdata->name, icecast_hls_version_string())) != 0) {
        LOGERRNO("error appending name string");
        return r;
    }

    /* Most other codecs only have 1 piece of DSI, but Ogg contains 3!
     * inspecting the extradata spit out by avcodec, it looks like they do
     * the xiph-style lacing, where:
     *   byte 0: number of packets - 1 (since there always needs to be at least 1 packet)
     *   then 2 sizes, encoded similar to ogg page packet sizes - always at least 1 byte,
     *   if a byte is 255 you read another
     *   The size of the third packet is implied.
     *   Then the packets concatenated together.
     */

    dsi_len = 3 + (header_packet[0].bytes/255) + (header_packet[1].bytes/255);
    dsi_len += header_packet[0].bytes;
    dsi_len += header_packet[1].bytes;
    dsi_len += header_packet[2].bytes;
    if( (r = membuf_ready(&userdata->me.dsi, dsi_len)) != 0) {
        LOGERRNO("error allocating dsi");
        return 01;
    }
    userdata->me.dsi.len = dsi_len;

    dsi = userdata->me.dsi.x;

    *dsi = 2; /* two packets */
    dsi++;

    for(i=0;i<2;i++) {
        dsi_len = header_packet[i].bytes;
        while(dsi_len >= 255) {
            *dsi = 255;
            dsi++;
            dsi_len -= 255;
        }
        *dsi = (uint8_t)dsi_len;
        dsi++;
    }

    for(i=0;i<3;i++) {
        memcpy(dsi, header_packet[i].packet, header_packet[i].bytes);
        dsi += header_packet[i].bytes;
    }

    userdata->me.codec = CODEC_TYPE_VORBIS;
    userdata->me.name = &userdata->name;
    userdata->me.channel_layout = source->channel_layout;
    userdata->me.sample_rate = source->sample_rate;
    userdata->me.sync_flag = 1;
    userdata->me.handle = userdata;

    return dest->open(dest->handle,&userdata->me);
}

static int encoder_plugin_vorbis_drain(encoder_plugin_vorbis_userdata* userdata, const packet_receiver* dest) {
    int r = -1;
    int rr = 0;
    int rrr = 0;
    ogg_packet op;

    packet p = packet_zero;

    p.sync = 1;
    p.sample_group = 1;
    p.sample_rate = userdata->buffer.sample_rate;
    int bcount = 0;

    while( (r = vorbis_analysis_blockout(&userdata->v_dsp, &userdata->v_block)) == 1) {
        bcount++;

        if( (rr = vorbis_analysis(&userdata->v_block, NULL)) != 0) {
            logs_error("error calling vorbis_analysis");
            return rr;
        }

        if( (rr = vorbis_bitrate_addblock(&userdata->v_block)) != 0) {
            logs_error("error calling vorbis_bitrate_addblock");
            return rr;
        }

        while( (rr = vorbis_bitrate_flushpacket(&userdata->v_dsp, &op)) == 1) {
            p.data.x = op.packet;
            p.data.len = op.bytes;
            p.pts = op.granulepos;
            p.duration = vorbis_packet_blocksize(&userdata->v_info, &op);

            if( (rrr = dest->submit_packet(dest->handle, &p)) != 0) {
                logs_error("error sending packet to muxer");
                return rrr;
            }
        }

        if(rr < 0) {
            logs_error("error calling vorbis_bitrate_flushpacket");
            return rr;
        }
    }

    if(r < 0) {
        logs_error("error calling vorbis_analysis_blockout");
        return r;
    }

    return 0;
}

static int encoder_plugin_vorbis_buffer_frame(encoder_plugin_vorbis_userdata* userdata, unsigned int duration) {
    int r;
    size_t i;
    size_t c;
    float* samples;
    float** buffer;

    buffer = vorbis_analysis_buffer(&userdata->v_dsp, duration);

    for(i=0; i < userdata->buffer.channels; i++) {
        c = (size_t)vorbis_channel_layout[userdata->buffer.channels][i];
        samples = (float*)frame_get_channel_samples(&userdata->buffer, i);
        memcpy(buffer[c], samples, sizeof(float) * duration);
    }

    frame_trim(&userdata->buffer, duration);

    if( (r = vorbis_analysis_wrote(&userdata->v_dsp, duration)) != 0) {
        logs_error("error calling vorbis_analysis_wrote");
        return r;
    }

    return 0;
}

static int encoder_plugin_vorbis_submit_frame(void* ud, const frame* frame, const packet_receiver* dest) {
    int r;
    encoder_plugin_vorbis_userdata* userdata = (encoder_plugin_vorbis_userdata*)ud;

    if( (r = frame_append(&userdata->buffer,frame)) != 0) {
        log_error("error appending frame to internal buffer: %d",r);
        return r;
    }

    while(userdata->buffer.duration >= BLOCK_SIZE) {
        if( (r = encoder_plugin_vorbis_buffer_frame(userdata, BLOCK_SIZE)) != 0) return r;
        if( (r = encoder_plugin_vorbis_drain(userdata, dest)) != 0) return r;
    }

    return 0;
}

static int encoder_plugin_vorbis_flush(void* ud, const packet_receiver* dest) {
    int r;
    encoder_plugin_vorbis_userdata* userdata = (encoder_plugin_vorbis_userdata*)ud;

    if(userdata->buffer.duration) {
        if( (r = encoder_plugin_vorbis_buffer_frame(userdata,userdata->buffer.duration)) != 0) return r;
        if( (r = encoder_plugin_vorbis_drain(userdata, dest)) != 0) return r;
    }

    if( (r = vorbis_analysis_wrote(&userdata->v_dsp, 0)) != 0) {
        logs_error("error calling vorbis_analysis_wrote");
        return r;
    }

    return encoder_plugin_vorbis_drain(userdata, dest);
}

static int encoder_plugin_vorbis_init(void) {
    return 0;
}

static void encoder_plugin_vorbis_deinit(void) {
    return;
}

const encoder_plugin encoder_plugin_vorbis = {
    &plugin_name,
    encoder_plugin_vorbis_size,
    encoder_plugin_vorbis_init,
    encoder_plugin_vorbis_deinit,
    encoder_plugin_vorbis_create,
    encoder_plugin_vorbis_config,
    encoder_plugin_vorbis_open,
    encoder_plugin_vorbis_close,
    encoder_plugin_vorbis_submit_frame,
    encoder_plugin_vorbis_flush,
    encoder_plugin_vorbis_reset,
};
