#ifndef VORBIS_COMMENT_INCLUDE_GUARD
#define VORBIS_COMMENT_INCLUDE_GUARD

#include <string.h>
#include <inttypes.h>

#include "strbuf.h"
#include "pack_u32le.h"

struct vc {
    strbuf data;
    size_t tagpos;
    size_t count;
};

typedef struct vc vc;

static inline void vc_reset(vc* v) {
    strbuf_reset(&v->data);
    v->tagpos = 0;
    v->count = 0;
}

static inline void vc_init(vc* v) {
    strbuf_init(&v->data);
    vc_reset(v);
}

static inline void vc_free(vc* v) {
    strbuf_free(&v->data);
}

static inline int vc_append_str(vc* v, const char* str, size_t len) {
    return strbuf_append(&v->data, str, len);
}

static inline int vc_append_cstr(vc* v, const char* str) {
    return strbuf_append(&v->data, str, strlen(str));
}

static inline int vc_prep_str(vc* v, size_t len) {
    int r;
    if( (r = strbuf_readyplus(&v->data, 4+len)) != 0) return r;
    pack_u32le(&v->data.x[v->data.len],len);
    v->data.len += 4;
    return 0;
}

static inline int vc_add_str(vc* v, const char* str, size_t len) {
    int r;
    if( (r = vc_prep_str(v,len)) != 0) return r;
    return strbuf_append(&v->data,str,len);
}

static inline int vc_add_cstr(vc* v, const char* str) {
    return vc_add_str(v,str,strlen(str));
}

static inline int vc_add_strbuf(vc* v, const strbuf* str) {
    return vc_add_str(v,(const char *)str->x, str->len);
}

static inline int vc_vendor_str(vc* v, const char* str, size_t len) {
    int r;

    if( (r = vc_add_str(v,str,len)) != 0) return r;
    v->tagpos = v->data.len;
    if( (r = vc_prep_str(v,0)) != 0) return r;
    v->count = 0;

    return 0;
}

static inline int vc_vendor_cstr(vc* v, const char* str) {
    return vc_vendor_str(v, str, strlen(str));
}

static inline int vc_vendor_strbuf(vc* v, const strbuf* str) {
    return vc_vendor_str(v, (const char *)str->x, str->len);
}

static inline int vc_add_tag(vc* v, const strbuf* key, const strbuf* value) {
    int r;

    if( (r = vc_prep_str(v, key->len + value->len + 1)) != 0) return r;
    if( (r = strbuf_cat(&v->data, key)) != 0) return r;
    if( (r = strbuf_append(&v->data, "=", 1)) != 0) return r;
    if( (r = strbuf_cat(&v->data, value)) != 0) return r;

    ++v->count;
    pack_u32le(&v->data.x[v->tagpos], v->count);
    return 0;
}

/* rewinds back to just after the vendor tag */
static inline int vc_restart(vc* v) {
    /* sanify check - this shouldn't ever happen */
    if(v->tagpos < 4) return -1;

    v->count = 0;

    v->data.len = v->tagpos;
    return vc_prep_str(v, 0);
}


#endif
