// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System.Threading;
using System.Diagnostics;
using System.Runtime.InteropServices;
using System.Runtime.CompilerServices;

namespace System.Xml
{
    unsafe internal struct XmlCharType
    {
        // Surrogate constants
        internal const int SurHighStart = 0xd800;    // 1101 10xx
        internal const int SurHighEnd = 0xdbff;
        internal const int SurLowStart = 0xdc00;    // 1101 11xx
        internal const int SurLowEnd = 0xdfff;
        internal const int SurMask = 0xfc00;    // 1111 11xx

        // Characters defined in the XML 1.0 Fourth Edition
        // Whitespace chars -- Section 2.3 [3]
        // Letters -- Appendix B [84]
        // Starting NCName characters -- Section 2.3 [5] (Starting Name characters without ':')
        // NCName characters -- Section 2.3 [4]          (Name characters without ':')
        // Character data characters -- Section 2.2 [2]
        // PubidChar ::=  #x20 | #xD | #xA | [a-zA-Z0-9] | [-'()+,./:=?;!*#@$_%] Section 2.3 of spec
        internal const int fWhitespace = 1;
        internal const int fLetter = 2;
        internal const int fNCStartNameSC = 4;
        internal const int fNCNameSC = 8;
        internal const int fCharData = 16;
        internal const int fNCNameXml4e = 32;
        internal const int fText = 64;
        internal const int fAttrValue = 128;

        // bitmap for public ID characters - 1 bit per character 0x0 - 0x80; no character > 0x80 is a PUBLIC ID char
        private const string s_PublicIdBitmap = "\u2400\u0000\uffbb\uafff\uffff\u87ff\ufffe\u07ff";

        private static readonly byte[] s_xmlCharTypeData = new byte[] {
            /* 0x0    */
                  0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x11, 0x11, 0x0 , 0x0 , 0x11, 0x0 , 0x0 ,
            /* 0x10   */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x20   */
                      0xD1, 0xD0, 0x50, 0xD0, 0xD0, 0xD0, 0x10, 0x50, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xD0,
            /* 0x30   */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0x10, 0xD0, 0x50, 0xD0,
            /* 0x40   */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x50   */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0x90, 0xD0, 0xFC,
            /* 0x60   */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x70   */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x80   */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x90   */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xA0   */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xB0   */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xC0   */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xD0   */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xE0   */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xF0   */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,

            /* 0x100  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x110  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x120  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x130  */
                      0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0,
            /* 0x140  */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x150  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x160  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x170  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0,
            /* 0x180  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x190  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1A0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1B0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1C0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE,
            /* 0x1D0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1E0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1F0  */
                      0xFE, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,

            /* 0x200  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x210  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x220  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x230  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x240  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x250  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x260  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x270  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x280  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x290  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x2A0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x2B0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x2C0  */
                      0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x2D0  */
                      0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x2E0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x2F0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x300  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x310  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x320  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x330  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x340  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x350  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x360  */
                      0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x370  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x380  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xF8, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE, 0xFE,
            /* 0x390  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x3A0  */
                      0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x3B0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x3C0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0,
            /* 0x3D0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE, 0xD0,
            /* 0x3E0  */
                      0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x3F0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x400  */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE,
            /* 0x410  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x420  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x430  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x440  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x450  */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE,
            /* 0x460  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x470  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x480  */
                      0xFE, 0xFE, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x490  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x4A0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x4B0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x4C0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0,
            /* 0x4D0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x4E0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE,
            /* 0x4F0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x500  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x510  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x520  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x530  */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x540  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x550  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x560  */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x570  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x580  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x590  */
                      0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x5A0  */
                      0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x5B0  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8,
            /* 0x5C0  */
                      0xD0, 0xF8, 0xF8, 0xD0, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x5D0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x5E0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x5F0  */
                      0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x600  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x610  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x620  */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x630  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x640  */
                      0xF8, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x650  */
                      0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x660  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x670  */
                      0xF8, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x680  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x690  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x6A0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x6B0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0,
            /* 0x6C0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0,
            /* 0x6D0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x6E0  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xFE, 0xFE, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0x6F0  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x700  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x710  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x720  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x730  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x740  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x750  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x760  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x770  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x780  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x790  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x7A0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x7B0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x7C0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x7D0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x7E0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x7F0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x800  */
                      0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x810  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x820  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x830  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xF8, 0xFE, 0xF8, 0xF8,
            /* 0x840  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0x850  */
                      0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x860  */
                      0xFE, 0xFE, 0xF8, 0xF8, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x870  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x880  */
                      0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE,
            /* 0x890  */
                      0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x8A0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x8B0  */
                      0xFE, 0xD0, 0xFE, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xF8, 0xD0, 0xF8, 0xF8,
            /* 0x8C0  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xF8, 0xF8, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0x8D0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE,
            /* 0x8E0  */
                      0xFE, 0xFE, 0xF8, 0xF8, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x8F0  */
                      0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x900  */
                      0xD0, 0xD0, 0xF8, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE,
            /* 0x910  */
                      0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x920  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x930  */
                      0xFE, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xD0, 0xD0, 0xF8, 0xD0, 0xF8, 0xF8,
            /* 0x940  */
                      0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0x950  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0,
            /* 0x960  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x970  */
                      0xF8, 0xF8, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x980  */
                      0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE,
            /* 0x990  */
                      0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x9A0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x9B0  */
                      0xFE, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xF8, 0xFE, 0xF8, 0xF8,
            /* 0x9C0  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0x9D0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x9E0  */
                      0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x9F0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0xA00  */
                      0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE,
            /* 0xA10  */
                      0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xA20  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xA30  */
                      0xFE, 0xD0, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xF8, 0xFE, 0xF8, 0xF8,
            /* 0xA40  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0xA50  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE,
            /* 0xA60  */
                      0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0xA70  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xA80  */
                      0xD0, 0xD0, 0xF8, 0xF8, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE,
            /* 0xA90  */
                      0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE, 0xFE,
            /* 0xAA0  */
                      0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE,
            /* 0xAB0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8,
            /* 0xAC0  */
                      0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0xAD0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xAE0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0xAF0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0xB00  */
                      0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE,
            /* 0xB10  */
                      0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xB20  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xB30  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8,
            /* 0xB40  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0xB50  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xB60  */
                      0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0xB70  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xB80  */
                      0xD0, 0xD0, 0xF8, 0xF8, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE,
            /* 0xB90  */
                      0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xBA0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xBB0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8,
            /* 0xBC0  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0xBD0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xD0,
            /* 0xBE0  */
                      0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0xBF0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0xC00  */
                      0xD0, 0xD0, 0xF8, 0xF8, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE,
            /* 0xC10  */
                      0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xC20  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xC30  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8,
            /* 0xC40  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0xC50  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xC60  */
                      0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0xC70  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xC80  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xC90  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xCA0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xCB0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xCC0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xCD0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xCE0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xCF0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0xD00  */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xD10  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xD20  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0,
            /* 0xD30  */
                      0xFE, 0xF8, 0xFE, 0xFE, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xD40  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0,
            /* 0xD50  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xD60  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xD70  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xD80  */
                      0xD0, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0, 0xD0, 0xFE, 0xD0, 0xD0,
            /* 0xD90  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xDA0  */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xD0,
            /* 0xDB0  */
                      0xFE, 0xF8, 0xFE, 0xFE, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xFE, 0xD0, 0xD0,
            /* 0xDC0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0xDD0  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xDE0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xDF0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0xE00  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xE10  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xE20  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xE30  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xD0, 0xF8, 0xD0, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8,
            /* 0xE40  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xE50  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xE60  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xE70  */
                      0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0xE80  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xE90  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0xEA0  */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0,
            /* 0xEB0  */
                      0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xEC0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xED0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xEE0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xEF0  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0xF00  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xF10  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xF20  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xF30  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xF40  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xF50  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xF60  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xF70  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xF80  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xF90  */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xFA0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xFB0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xFC0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0xFD0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xFE0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0xFF0  */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x1000 */
                      0xFE, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE,
            /* 0x1010 */
                      0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1020 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1030 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xD0, 0xFE, 0xD0,
            /* 0x1040 */
                      0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xD0, 0xFE, 0xD0,
            /* 0x1050 */
                      0xFE, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE,
            /* 0x1060 */
                      0xFE, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0,
            /* 0x1070 */
                      0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1080 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1090 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xD0,
            /* 0x10A0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xD0, 0xD0, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE,
            /* 0x10B0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x10C0 */
                      0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x10D0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x10E0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x10F0 */
                      0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x1100 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1110 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1120 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1130 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1140 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1150 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1160 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1170 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1180 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1190 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x11A0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x11B0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x11C0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x11D0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x11E0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x11F0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x1200 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1210 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0,
            /* 0x1220 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1230 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1240 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0,
            /* 0x1250 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE, 0xD0, 0xFE,
            /* 0x1260 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1270 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0,
            /* 0x1280 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1290 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x12A0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x12B0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xD0,
            /* 0x12C0 */
                      0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0,
            /* 0x12D0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x12E0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0,
            /* 0x12F0 */
                      0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0,

            /* 0x1300 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1310 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1320 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1330 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1340 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1350 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1360 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1370 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1380 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1390 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x13A0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x13B0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x13C0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x13D0 */
                      0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0,
            /* 0x13E0 */
                      0xD0, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x13F0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x1400 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1410 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1420 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xD0, 0xD0, 0xFE, 0xD0,
            /* 0x1430 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1440 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1450 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1460 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1470 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1480 */
                      0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1490 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x14A0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x14B0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x14C0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x14D0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x14E0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x14F0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x1500 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xD0, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1510 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1520 */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
            /* 0x1530 */
                      0xD0, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1540 */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1550 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1560 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1570 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1580 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1590 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xF8, 0xF8, 0xD0, 0xD0, 0xF8, 0xF8, 0xD0,
            /* 0x15A0 */
                      0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x15B0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x15C0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x15D0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x15E0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x15F0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xF8, 0xF8, 0xF8, 0xD0,

            /* 0x1600 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1610 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1620 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0,
            /* 0x1630 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1640 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1650 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1660 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1670 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1680 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1690 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x16A0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x16B0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x16C0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x16D0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x16E0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x16F0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x1700 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1710 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1720 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1730 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1740 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1750 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1760 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1770 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1780 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1790 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x17A0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x17B0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x17C0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x17D0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x17E0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x17F0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,

            /* 0x1800 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1810 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1820 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1830 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1840 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1850 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1860 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1870 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1880 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1890 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x18A0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x18B0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x18C0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x18D0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x18E0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x18F0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x1900 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1910 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1920 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1930 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1940 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1950 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1960 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1970 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1980 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x1990 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
            /* 0x19A0 */
                      0xFE, 0xFE, 0xFE, 0xFE, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x19B0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x19C0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x19D0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x19E0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x19F0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,

            /* 0x1A00 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1A10 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1A20 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1A30 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1A40 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1A50 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1A60 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1A70 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1A80 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1A90 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1AA0 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1AB0 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1AC0 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1AD0 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1AE0 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,
            /* 0x1AF0 */
                      0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 , 0x0 ,

            /* 0x1B00 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1B10 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1B20 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1B30 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1B40 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1B50 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1B60 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1B70 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1B80 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1B90 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1BA0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1BB0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1BC0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1BD0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1BE0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
            /* 0x1BF0 */
                      0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0x0 , 0x0
        };

        private static short[] s_Indexes = new short[256] {
            /* 0x0  */  0x0   , 0x100 , 0x200 , 0x300 , 0x400 , 0x500 , 0x600 , 0x700 ,
            /* 0x8  */
                        0x700 , 0x800 , 0x900 , 0xA00 , 0xB00 , 0xC00 , 0xD00 , 0xE00 ,
            /* 0x10 */
                        0xF00 , 0x1000, 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 ,
            /* 0x18 */
                        0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x1100, 0x1200,
            /* 0x20 */
                        0x1300, 0x1400, 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 ,
            /* 0x28 */
                        0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 ,
            /* 0x30 */
                        0x1500, 0x1600, 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 ,
            /* 0x38 */
                        0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 ,
            /* 0x40 */
                        0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 ,
            /* 0x48 */
                        0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x1700, 0x1700,
            /* 0x50 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0x58 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0x60 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0x68 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0x70 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0x70 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0x80 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0x80 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0x90 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0x98 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1800,
            /* 0xA0 */
                        0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 ,
            /* 0xA8 */
                        0x700 , 0x700 , 0x700 , 0x700 , 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0xB0 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0xB8 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0xC0 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0xC8 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700,
            /* 0xD0 */
                        0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1700, 0x1900,
            /* 0xD8 */
                        0x1A00, 0x1A00, 0x1A00, 0x1A00, 0x1A00, 0x1A00, 0x1A00, 0x1A00,
            /* 0xE0 */
                        0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 ,
            /* 0xE8 */
                        0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 ,
            /* 0xF0 */
                        0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 ,
            /* 0xF8 */
                        0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x700 , 0x1B00
        };

        private static readonly IntPtr[] s_PageIndexes = new IntPtr[256];
        private static readonly byte** s_CharProperties = InitializeCharProperties();
        private byte** _charProperties;

        private static byte** InitializeCharProperties()
        {
            byte* s_CharData = (byte*)GCHandle.Alloc(s_xmlCharTypeData, GCHandleType.Pinned).AddrOfPinnedObject();

            // we never free this memory. we avoid allocate this memory in the managed heap and pinning it to reduce 
            // the pressure on the GC
            for (int i = 0; i < 256; i++)
            {
                s_PageIndexes[i] = (IntPtr)(s_CharData + s_Indexes[i]);
            }

            return (byte**)GCHandle.Alloc(s_PageIndexes, GCHandleType.Pinned).AddrOfPinnedObject();
        }

        private XmlCharType(byte** charProperties)
        {
            Debug.Assert(s_CharProperties != null);
            _charProperties = charProperties;
        }

        public static XmlCharType Instance
        {
            get { return new XmlCharType(s_CharProperties); }
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsWhiteSpace(char ch)
        {
            return (_charProperties[ch >> 8][ch & 0xff] & fWhitespace) != 0;
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsNCNameSingleChar(char ch)
        {
            return (_charProperties[ch >> 8][ch & 0xff] & fNCNameSC) != 0;
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsStartNCNameSingleChar(char ch)
        {
            return (_charProperties[ch >> 8][ch & 0xff] & fNCStartNameSC) != 0;
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsNameSingleChar(char ch)
        {
            return IsNCNameSingleChar(ch) || ch == ':';
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsStartNameSingleChar(char ch)
        {
            return IsStartNCNameSingleChar(ch) || ch == ':';
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsCharData(char ch)
        {
            return (_charProperties[ch >> 8][ch & 0xff] & fCharData) != 0;
        }

        // [13] PubidChar ::=  #x20 | #xD | #xA | [a-zA-Z0-9] | [-'()+,./:=?;!*#@$_%] Section 2.3 of spec
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsPubidChar(char ch)
        {
            if (ch < (char)0x80)
            {
                return (s_PublicIdBitmap[ch >> 4] & (1 << (ch & 0xF))) != 0;
            }
            return false;
        }

        // TextChar = CharData - { 0xA, 0xD, '<', '&', ']' }
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        internal bool IsTextChar(char ch)
        {
            return (_charProperties[ch >> 8][ch & 0xff] & fText) != 0;
        }

        // AttrValueChar = CharData - { 0xA, 0xD, 0x9, '<', '>', '&', '\'', '"' }
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        internal bool IsAttributeValueChar(char ch)
        {
            return (_charProperties[ch >> 8][ch & 0xff] & fAttrValue) != 0;
        }

        // XML 1.0 Fourth Edition definitions
        //
        // NOTE: This method will not be inlined (because it uses byte* charProperties)
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsLetter(char ch)
        {
            return (_charProperties[ch >> 8][ch & 0xff] & fLetter) != 0;
        }

        // NOTE: This method will not be inlined (because it uses byte* charProperties)
        // This method uses the XML 4th edition name character ranges
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsNCNameCharXml4e(char ch)
        {
            return (_charProperties[ch >> 8][ch & 0xff] & fNCNameXml4e) != 0;
        }

        // This method uses the XML 4th edition name character ranges
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsStartNCNameCharXml4e(char ch)
        {
            return IsLetter(ch) || ch == '_';
        }

        // This method uses the XML 4th edition name character ranges
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsNameCharXml4e(char ch)
        {
            return IsNCNameCharXml4e(ch) || ch == ':';
        }

        // This method uses the XML 4th edition name character ranges
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool IsStartNameCharXml4e(char ch)
        {
            return IsStartNCNameCharXml4e(ch) || ch == ':';
        }

        // Digit methods
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static bool IsDigit(char ch)
        {
            return InRange(ch, 0x30, 0x39);
        }

        // Surrogate methods
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        internal static bool IsHighSurrogate(int ch)
        {
            return InRange(ch, SurHighStart, SurHighEnd);
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        internal static bool IsLowSurrogate(int ch)
        {
            return InRange(ch, SurLowStart, SurLowEnd);
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        internal static bool IsSurrogate(int ch)
        {
            return InRange(ch, SurHighStart, SurLowEnd);
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        internal static int CombineSurrogateChar(int lowChar, int highChar)
        {
            return (lowChar - SurLowStart) | ((highChar - SurHighStart) << 10) + 0x10000;
        }

        internal static void SplitSurrogateChar(int combinedChar, out char lowChar, out char highChar)
        {
            int v = combinedChar - 0x10000;
            lowChar = (char)(SurLowStart + v % 1024);
            highChar = (char)(SurHighStart + v / 1024);
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        internal bool IsOnlyWhitespace(string str)
        {
            return IsOnlyWhitespaceWithPos(str) == -1;
        }

        // Character checking on strings
        internal int IsOnlyWhitespaceWithPos(string str)
        {
            if (str != null)
            {
                for (int i = 0; i < str.Length; i++)
                {
                    char ch = str[i];
                    if ((_charProperties[ch >> 8][ch & 0xff] & fWhitespace) == 0)
                    {
                        return i;
                    }
                }
            }
            return -1;
        }

        internal int IsOnlyCharData(string str)
        {
            if (str != null)
            {
                for (int i = 0; i < str.Length; i++)
                {
                    char ch = str[i];
                    if ((_charProperties[ch >> 8][ch & 0xff] & fCharData) == 0)
                    {
                        if (i + 1 >= str.Length || !(XmlCharType.IsHighSurrogate(str[i]) && XmlCharType.IsLowSurrogate(str[i + 1])))
                        {
                            return i;
                        }
                        else
                        {
                            i++;
                        }
                    }
                }
            }
            return -1;
        }

        static internal bool IsOnlyDigits(string str, int startPos, int len)
        {
            Debug.Assert(str != null);
            Debug.Assert(startPos + len <= str.Length);
            Debug.Assert(startPos <= str.Length);

            for (int i = startPos; i < startPos + len; i++)
            {
                if (!IsDigit(str[i]))
                {
                    return false;
                }
            }
            return true;
        }

        static internal bool IsOnlyDigits(char[] chars, int startPos, int len)
        {
            Debug.Assert(chars != null);
            Debug.Assert(startPos + len <= chars.Length);
            Debug.Assert(startPos <= chars.Length);

            for (int i = startPos; i < startPos + len; i++)
            {
                if (!IsDigit(chars[i]))
                {
                    return false;
                }
            }
            return true;
        }

        internal int IsPublicId(string str)
        {
            if (str != null)
            {
                for (int i = 0; i < str.Length; i++)
                {
                    if (!IsPubidChar(str[i]))
                    {
                        return i;
                    }
                }
            }
            return -1;
        }

        // This method tests whether a value is in a given range with just one test; start and end should be constants
        private static bool InRange(int value, int start, int end)
        {
            Debug.Assert(start <= end);
            return (uint)(value - start) <= (uint)(end - start);
        }
    }
}

