/*
 * Copyright 2019 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.quarkus.gradle.tasks;

import java.io.File;
import java.util.List;

import org.gradle.api.tasks.Optional;
import org.gradle.api.tasks.TaskAction;
import org.gradle.api.tasks.options.Option;

import io.quarkus.creator.AppArtifact;
import io.quarkus.creator.AppCreator;
import io.quarkus.creator.AppCreatorException;
import io.quarkus.creator.AppDependency;
import io.quarkus.creator.phase.augment.AugmentPhase;
import io.quarkus.creator.phase.curate.CurateOutcome;
import io.quarkus.creator.phase.runnerjar.RunnerJarOutcome;
import io.quarkus.creator.phase.runnerjar.RunnerJarPhase;
import io.quarkus.gradle.ResolvedGradleArtifactDeps;

/**
 * @author <a href="mailto:stalep@gmail.com">Ståle Pedersen</a>
 */
public class QuarkusBuild extends QuarkusTask {

    private String transformedClassesDirectory;

    private String wiringClassesDirectory;

    private String buildDir;

    private String libDir;

    private String mainClass = "io.quarkus.runner.GeneratedMain";

    private boolean useStaticInit = true;

    private boolean uberJar = false;

    public QuarkusBuild() {
        super("Quarkus builds a runner jar based on the build jar");
    }

    public File getTransformedClassesDirectory() {
        if (transformedClassesDirectory == null)
            return extension().transformedClassesDirectory();
        else
            return new File(transformedClassesDirectory);
    }

    @Option(description = "The directory for application classes transformed by processing.", option = "transformed-classes-directory")
    @Optional
    public void setTransformedClassesDirectory(String transformedClassesDirectory) {
        this.transformedClassesDirectory = transformedClassesDirectory;
    }

    public File getWiringClassesDirectory() {
        if (wiringClassesDirectory == null)
            return extension().wiringClassesDirectory();
        else
            return new File(wiringClassesDirectory);
    }

    @Option(description = "The directory for classes generated by processing", option = "wiring-classes-directory")
    @Optional
    public void setWiringClassesDirectory(String wiringClassesDirectory) {
        this.wiringClassesDirectory = wiringClassesDirectory;
    }

    public File getBuildDir() {
        if (buildDir == null)
            return extension().buildDir();
        else
            return new File(buildDir);
    }

    @Option(description = "The directory where built classes are stored", option = "build-dir")
    @Optional
    public void setBuildDir(String buildDir) {
        this.buildDir = buildDir;
    }

    public File getLibDir() {
        if (libDir == null)
            return extension().libDir();
        else
            return new File(libDir);
    }

    @Option(description = "The directory for library jars", option = "lib-dir")
    @Optional
    public void setLibDir(String libDir) {
        this.libDir = libDir;
    }

    public String getMainClass() {
        return mainClass;
    }

    @Option(description = "Name of the main class generated by the quarkus build process", option = "main-class")
    @Optional
    public void setMainClass(String mainClass) {
        this.mainClass = mainClass;
    }

    public boolean isUseStaticInit() {
        return useStaticInit;
    }

    @Option(description = "", option = "use-static-init")
    @Optional
    public void setUseStaticInit(boolean useStaticInit) {
        this.useStaticInit = useStaticInit;
    }

    public boolean isUberJar() {
        return uberJar;
    }

    @Option(description = "Set to true if the build task should build an uberjar", option = "uber-jar")
    @Optional
    public void setUberJar(boolean uberJar) {
        this.uberJar = uberJar;
    }

    @TaskAction
    public void buildQuarkus() {
        getLogger().lifecycle("building quarkus runner");

        try (AppCreator appCreator = AppCreator.builder()
                // configure the build phases we want the app to go through
                .addPhase(new AugmentPhase()
                        .setAppClassesDir(extension().outputDirectory().toPath())
                        .setTransformedClassesDir(getTransformedClassesDirectory().toPath())
                        .setWiringClassesDir(getWiringClassesDirectory().toPath()))
                .addPhase(new RunnerJarPhase()
                        .setLibDir(getLibDir().toPath())
                        .setFinalName(extension().finalName())
                        .setMainClass(getMainClass())
                        .setUberJar(isUberJar()))
                .setWorkDir(extension().buildDir().toPath())
                .build()) {

            final AppArtifact appArtifact = new AppArtifact(extension().groupId(), extension().artifactId(),
                    extension().version());
            try {
                ResolvedGradleArtifactDeps resolvedGradleArtifactDeps = new ResolvedGradleArtifactDeps(getProject());
                final List<AppDependency> appDeps = resolvedGradleArtifactDeps.collectDependencies(appArtifact);

                // push resolved application state
                appCreator.pushOutcome(CurateOutcome.builder()
                        .setAppArtifact(appArtifact)
                        .setArtifactResolver(resolvedGradleArtifactDeps)
                        .setInitialDeps(appDeps)
                        .build());

                // resolve the outcome we need here
                appCreator.resolveOutcome(RunnerJarOutcome.class);
            } catch (AppCreatorException e) {
                e.printStackTrace();
            }

        } catch (AppCreatorException e) {
            e.printStackTrace();
        }

    }
}
