<?php
/**
 * Subclass for representing a row from the 'asset' table, used for caption_assets
 *
 * @package plugins.caption
 * @subpackage model
 */ 
class CaptionAsset extends asset
{
	const CUSTOM_DATA_FIELD_LANGUAGE = "language";
	const CUSTOM_DATA_FIELD_DEFAULT = "default";
	const CUSTOM_DATA_FIELD_LABEL = "label";
	const CUSTOM_DATA_PARENT_ID = "parentId";
	const CUSTOM_DATA_ACCURACY = "accuracy";
	const CUSTOM_DATA_DISPLAY_ON_PLAYER = "display_on_player";
	const CUSTOM_DATA_ASSOCIATED_TRANSCRIPT_ASSET_IDS = "associated_transcript_ids";
	const CUSTOM_DATA_SOURCE = 'source';
	const CUSTOM_DATA_USAGE = 'usage';
	
	const MULTI_LANGUAGE = 'Multilingual';

	/* (non-PHPdoc)
	 * @see Baseasset::applyDefaultValues()
	 */
	public function applyDefaultValues()
	{
		parent::applyDefaultValues();
		$this->setType(CaptionPlugin::getAssetTypeCoreValue(CaptionAssetType::CAPTION));
	}

	public function getLanguage()		    {return $this->getFromCustomData(self::CUSTOM_DATA_FIELD_LANGUAGE);}
	public function getDefault()		    {return $this->getFromCustomData(self::CUSTOM_DATA_FIELD_DEFAULT);}
	public function getLabel()			    {return $this->getFromCustomData(self::CUSTOM_DATA_FIELD_LABEL);}
	public function getParentId()           {return $this->getFromCustomData(self::CUSTOM_DATA_PARENT_ID);}
	public function getAccuracy()           {return $this->getFromCustomData(self::CUSTOM_DATA_ACCURACY);}
	public function getDisplayOnPlayer()    {return $this->getFromCustomData(self::CUSTOM_DATA_DISPLAY_ON_PLAYER, null, true);}
	public function getAssociatedTranscriptIds()    {return $this->getFromCustomData(self::CUSTOM_DATA_ASSOCIATED_TRANSCRIPT_ASSET_IDS);}
	public function getSource()             {return $this->getFromCustomData(self::CUSTOM_DATA_SOURCE);}
	public function getUsage() 				{return $this->getFromCustomData(self::CUSTOM_DATA_USAGE, null, CaptionUsage::CAPTION);}


	public function setLanguage($v)		    {$this->putInCustomData(self::CUSTOM_DATA_FIELD_LANGUAGE, $v);}
	public function setDefault($v)		    {$this->putInCustomData(self::CUSTOM_DATA_FIELD_DEFAULT, (bool)$v);}
	public function setLabel($v)		    {$this->putInCustomData(self::CUSTOM_DATA_FIELD_LABEL, $v);}
	public function setParentId($v)         {$this->putInCustomData(self::CUSTOM_DATA_PARENT_ID, $v);}
	public function setAccuracy($v)         {$this->putInCustomData(self::CUSTOM_DATA_ACCURACY, $v);}
	public function setDisplayOnPlayer($v)  {$this->putInCustomData(self::CUSTOM_DATA_DISPLAY_ON_PLAYER, $v);}
	public function setAssociatedTranscriptIds($v)    {$this->putInCustomData(self::CUSTOM_DATA_ASSOCIATED_TRANSCRIPT_ASSET_IDS, $v);}
	public function setSource($v)           {$this->putInCustomData(self::CUSTOM_DATA_SOURCE, $v);}
	public function setUsage($v)			{$this->putInCustomData(self::CUSTOM_DATA_USAGE, $v);}
	
	public function getFinalDownloadUrlPathWithoutKs()
	{
		$finalPath = '/api_v3/index.php/service/caption_captionAsset/action/serve';
		$finalPath .= '/captionAssetId/' . $this->getId();
		if($this->getVersion() > 1)
		{
			$finalPath .= '/v/' . $this->getVersion();
		}
		
		$partner = PartnerPeer::retrieveByPK($this->getPartnerId());
		$entry = $this->getentry();
		
		$partnerVersion = $partner->getFromCustomData('cache_caption_version');
		$entryVersion = $entry->getFromCustomData('cache_caption_version');
		
		$finalPath .= ($partnerVersion ? "/pv/$partnerVersion" : '');
		$finalPath .= ($entryVersion ? "/ev/$entryVersion" : '');
		
		return $finalPath;
	}
	
	public function setFromAssetParams($dbAssetParams)
	{
		parent::setFromAssetParams($dbAssetParams);
		
		$this->setLanguage($dbAssetParams->getLanguage());
		$this->setLabel($dbAssetParams->getLabel());
	}
	
	public function getName()
	{
		return $this->getLanguage();
	}

	protected function validateCaptionContent()
	{
		KalturaLog::debug("Validate caption content for caption asset ID {$this->getId()}");

		//Get the Content Manager
		$captionsContentManager = kCaptionsContentManager::getCoreContentManager($this->getContainerFormat());
		if (!$captionsContentManager)
		{
			KalturaLog::warning("Failed to find content manager for caption ID {$this->getId()} and container format {$this->getContainerFormat()}");
			return false;
		}

		//Get the content
		$content = kCaptionsContentManager::extractContent($this);
		if(!$content)
		{
			return false;
		}

		//Validate
		$validatedCaption = $captionsContentManager->validate($content);
		if (!$validatedCaption)
		{
			KalturaLog::info("Failed to validate the content for caption ID {$this->getId()}");
		}
		return $validatedCaption;
	}

	public function preUpdate(PropelPDO $con = null)
	{
		$ret = parent::preUpdate($con); // TODO: Change the autogenerated stub

		if( ($this->getStatus() == CaptionAsset::ASSET_STATUS_READY)
			&& ($this->isColumnModified(assetPeer::VERSION) || $this->isColumnModified(assetPeer::STATUS)) )
		{
			if(!$this->validateCaptionContent())
			{
				$this->setStatus(CaptionAsset::ASSET_STATUS_ERROR);
			}
		}

		if( $this->getAccuracy() && ($this->getStatus() == CaptionAsset::ASSET_STATUS_READY &&
			($this->isColumnModified(assetPeer::STATUS) && $this->getLanguage()) ||
			($this->isCustomDataModified(self::CUSTOM_DATA_ACCURACY) && $this->getLanguage()) ||
			($this->isCustomDataModified(self::CUSTOM_DATA_FIELD_LANGUAGE)))
		)
		{
			//Check if we have duplication in language in this case the one that should be marked as displayOnPlayer = true is:
			// The newest caption - if it has accuracy >= 99 OR if the oldest caption had accuracy and his accuracy is smaller than the newest.
			// otherwise - The oldest caption will remain marked.
			$captionAssets = assetPeer::retrieveByEntryId($this->getEntryId(), array(CaptionPlugin::getAssetTypeCoreValue(CaptionAssetType::CAPTION)));
			foreach ($captionAssets as $captionAsset)
			{
				/* @var $captionAsset CaptionAsset */
				if( ($this->getId() === $captionAsset->getId()) ||
					($captionAsset->getLanguage() != $this->getLanguage()) ||
					!$captionAsset->getDisplayOnPlayer() ||
					!$this->getDisplayOnPlayer() )
				{
					continue;
				}

				if ($this->getAccuracy() >= 99 ||
					($captionAsset->getAccuracy() && ($this->getAccuracy() >= $captionAsset->getAccuracy())) )
				{
					$captionAsset->setDisplayOnPlayer(false);
					$this->decideDefaultCaption($captionAsset);
				}
				else
				{
					$this->setDisplayOnPlayer(false);
				}

				$this->save();
				$captionAsset->save();
			}
		}
		
		return $ret;
	}
	
	public function shouldCopyOnReplacement() {return false;}

	protected function decideDefaultCaption($captionAsset)
	{
		if ($captionAsset->getDefault())
		{
			$this->setDefault(true);
			$captionAsset->setDefault(false);
		}
	}

	/**
	 * (non-PHPdoc)
	 * @see lib/model/ISyncableFile#getTypeFolderName()
	 */
	public function getTypeFolderName()
	{
		return 'captions';
	}

}
