<?php
/**
 * @package plugins.elasticSearch
 * @subpackage model
 */

function build_sorter($objectsOrder) {
	return function ($a, $b) use ($objectsOrder) {
		return ($objectsOrder[$a->getId()] > $objectsOrder[$b->getId()]) ? 1 : -1;
	};
}

class kESearchCoreAdapter
{

	const HITS_KEY = 'hits';
	const INNER_HITS_KEY = 'inner_hits';
	const ID_KEY = '_id';
	const TOTAL_KEY = 'total';
	const TOTAL_COUNT_KEY = 'totalCount';
	const ITEMS_KEY = 'items';
	const HIGHLIGHT_KEY = 'highlight';
	const SOURCE = '_source';

	private static $innerHitsObjectType = array(
		'caption' => ESearchItemDataType::CAPTION,
		'metadata' => ESearchItemDataType::METADATA,
		'cue_points' => ESearchItemDataType::CUE_POINTS
	);

	public static function transformElasticToCoreObject($elasticResults, $coreSearchObject)
	{
		list($objectData, $objectOrder, $objectCount, $objectHighlight, $aggregations) = self::getElasticResultAsArray($elasticResults,
			$coreSearchObject->getQueryAttributes()->getQueryHighlightsAttributes());
		$objects = $coreSearchObject->fetchCoreObjectsByIds(array_keys($objectData));
		$coreResults = self::getCoreESearchResults($objects, $objectData, $objectOrder, $objectHighlight);
		return array($coreResults, $objectCount, $aggregations);
	}

	public static function getElasticResultAsArray($elasticResults, $queryHighlightsAttribute)
	{
		$objectData = array();
		$objectOrder = array();
		$objectHighlight = array();
		$objectCount = 0;
		foreach ($elasticResults[self::HITS_KEY][self::HITS_KEY] as $key => $elasticObject)
		{
			$itemData = array();
			if (isset($elasticObject[self::INNER_HITS_KEY]))
				self::getItemDataFromInnerHits($elasticObject, $itemData, $queryHighlightsAttribute);
			
			$objectData[$elasticObject[self::ID_KEY]] = $itemData;
			$objectOrder[$elasticObject[self::ID_KEY]] = $key;
			if(array_key_exists(self::HIGHLIGHT_KEY, $elasticObject))
				$objectHighlight[$elasticObject[self::ID_KEY]] = self::elasticHighlightToCoreHighlight($elasticObject[self::HIGHLIGHT_KEY], $queryHighlightsAttribute);
		}
		
		if(isset($elasticResults[self::HITS_KEY][self::TOTAL_KEY]))
			$objectCount = $elasticResults[self::HITS_KEY][self::TOTAL_KEY];

		$aggregations = isset($elasticResults['aggregations']) ? $elasticResults['aggregations'] : null;

		return array($objectData, $objectOrder, $objectCount, $objectHighlight, $aggregations );
	}

	private static function getCoreESearchResults($coreObjects, $objectsData, $objectsOrder, $objectHighlight)
	{
		$resultsObjects = array();
		usort($coreObjects, build_sorter($objectsOrder));
		foreach ($coreObjects as $coreObject)
		{
			$resultObj = new ESearchResult();
			$resultObj->setObject($coreObject);
			$itemsData = array();
			foreach ($objectsData[$coreObject->getId()] as $innerHitKey => $values)
			{
				$itemsDataResult = self::getItemsDataResult($innerHitKey, $values);
				if($itemsDataResult)
					$itemsData[] = $itemsDataResult;
			}

			$resultObj->setItemsData($itemsData);
			if(array_key_exists($coreObject->getId(), $objectHighlight))
				$resultObj->setHighlight($objectHighlight[$coreObject->getId()]);

			$resultsObjects[] = $resultObj;
		}

		return $resultsObjects;
	}

	private static function getItemDataFromInnerHits($elasticObject, &$itemData, $queryHighlightsAttribute)
	{
		foreach ($elasticObject[self::INNER_HITS_KEY] as $innerHitsKey => $hits)
		{
			list($objectType, $queryName, $queryIndex) = self::getDataFromInnerHitsKey($innerHitsKey);

			$itemData[$innerHitsKey] = array();
			$itemData[$innerHitsKey][self::TOTAL_COUNT_KEY] = self::getInnerHitsTotalCountForObject($hits, $objectType);
			foreach ($hits[self::HITS_KEY][self::HITS_KEY] as $itemResult)
			{
				$currItemData = KalturaPluginManager::loadObject('ESearchItemData', $objectType);
				if ($currItemData)
				{
					if(array_key_exists(self::HIGHLIGHT_KEY, $itemResult))
						$itemResult[self::HIGHLIGHT_KEY] = self::elasticHighlightToCoreHighlight($itemResult[self::HIGHLIGHT_KEY], $queryHighlightsAttribute);

					$currItemData->loadFromElasticHits($itemResult);
					$itemData[$innerHitsKey][self::ITEMS_KEY][] = $currItemData;
				}
			}
		}
	}

	private static function getDataFromInnerHitsKey($innerHitsKey)
	{
		$queryNames = explode(ESearchNestedObjectItem::QUERY_NAME_DELIMITER, $innerHitsKey);
		$objectType = $queryNames[0];
		$queryName = null;
		if ($queryNames[1] != ESearchNestedObjectItem::DEFAULT_GROUP_NAME)
			$queryName = $queryNames[1];

		$queryIndex =  $queryNames[2];

		if(isset(self::$innerHitsObjectType[$objectType]))
			return array(self::$innerHitsObjectType[$objectType], $queryName, $queryIndex);

		KalturaLog::err('Unsupported inner object key in elastic results['.$innerHitsKey.']');
		return array($objectType, $queryName, $queryIndex);
	}

	private static function getItemsDataResult($innerHitsKey, $values)
	{
		if(!isset($values[self::ITEMS_KEY]))
			return null;

		list($objectType, $queryName, $queryIndex) = self::getDataFromInnerHitsKey($innerHitsKey);

		$result = new ESearchItemsDataResult();
		$result->setTotalCount($values[self::TOTAL_COUNT_KEY]);
		$result->setItems($values[self::ITEMS_KEY]);
		$result->setItemsType($objectType);

		return $result;
	}

	private static function getInnerHitsTotalCountForObject($objectResult, $objectType)
	{
		switch ($objectType)
		{
			case ESearchItemDataType::CAPTION:
			case ESearchItemDataType::METADATA:
			case ESearchItemDataType::CUE_POINTS:
				return $objectResult[self::HITS_KEY][self::TOTAL_KEY];
			default:
			{
				KalturaLog::err('Unsupported inner object type in elastic results['.$objectType.']');
				return 0;
			}
		}
	}


	/**
	 * @param array $eHighlight
	 * @param ESearchQueryHighlightsAttributes $queryHighlightsAttribute
	 * @return array|null
	 */
	private static function elasticHighlightToCoreHighlight($eHighlight, $queryHighlightsAttribute)
	{
		if(isset($eHighlight))
		{
			$result = array();
			$eHighlight = $queryHighlightsAttribute->removeDuplicateHits($eHighlight);
			foreach ($eHighlight as $key => $value)
			{
				$resultType = new ESearchHighlight();
				$resultType->setFieldName($key);
				if(self::shouldRemoveRedundantTags($key))
					self::removeRedundantTags($value);
				$resultType->setHits($value);
				$result[] = $resultType;
			}

			return $result;
		}

		return null;
	}

	private static function shouldRemoveRedundantTags($fieldName)
	{
		$suffix = elasticSearchUtils::DOT_FIELD_DELIMITER . kESearchQueryManager::NGRAMS_FIELD_SUFFIX;
		return (strpos($fieldName ,$suffix) !== false);
	}

	private static function removeRedundantTags(&$values)
	{
		foreach ($values as &$value)
			$value = str_replace ("</em><em>", "" ,$value);
	}

	public static function getObjectIdsFromElasticResults($elasticResults)
	{
		$ids = array();
		foreach ($elasticResults[self::HITS_KEY][self::HITS_KEY] as $elasticObject)
		{
			$ids[] = $elasticObject[self::ID_KEY];
		}
		return $ids;
	}

}
