<?php
/**
 * @service eSearch
 * @package plugins.elasticSearch
 * @subpackage api.services
 */
class ESearchService extends KalturaBaseService
{
	const MAX_RESULT_WINDOW = 10000;
	/**
	 *
	 * @action searchEntry
	 * @param KalturaESearchEntryParams $searchParams
	 * @param KalturaPager $pager
	 * @return KalturaESearchEntryResponse
	 */
	function searchEntryAction(KalturaESearchEntryParams $searchParams, KalturaPager $pager = null)
	{
		$entrySearch = new kEntrySearch();
		list($coreResults, $objectCount, $aggregations) = $this->initAndSearch($entrySearch, $searchParams, $pager);
		$response = new KalturaESearchEntryResponse();
		$response->objects = KalturaESearchEntryResultArray::fromDbArray($coreResults, $this->getResponseProfile());
		$response->totalCount = $objectCount;
		if($aggregations)
		{
			$aggregationResponse = new KalturaESearchAggregationResponse();
			$response->aggregations = $aggregationResponse->resultToApi($aggregations);
		}
		return $response;
	}

	/**
	 *
	 * @action searchCategory
	 * @param KalturaESearchCategoryParams $searchParams
	 * @param KalturaPager $pager
	 * @return KalturaESearchCategoryResponse
	 */
	function searchCategoryAction(KalturaESearchCategoryParams $searchParams, KalturaPager $pager = null)
	{
		$categorySearch = new kCategorySearch();
		list($coreResults, $objectCount) = $this->initAndSearch($categorySearch, $searchParams, $pager);
		$response = new KalturaESearchCategoryResponse();
		$response->objects = KalturaESearchCategoryResultArray::fromDbArray($coreResults, $this->getResponseProfile());
		$response->totalCount = $objectCount;
		return $response;
	}

	/**
	 *
	 * @action searchUser
	 * @param KalturaESearchUserParams $searchParams
	 * @param KalturaPager $pager
	 * @return KalturaESearchUserResponse
	 */
	function searchUserAction(KalturaESearchUserParams $searchParams, KalturaPager $pager = null)
	{
		$userSearch = new kUserSearch();
		list($coreResults, $objectCount) = $this->initAndSearch($userSearch, $searchParams, $pager);
		$response = new KalturaESearchUserResponse();
		$response->objects = KalturaESearchUserResultArray::fromDbArray($coreResults, $this->getResponseProfile());
		$response->totalCount = $objectCount;
		return $response;
	}

	/**
	 * Creates a batch job that sends an email with a link to download a CSV containing a list of entries
	 *
	 * @action entryExportToCsv
	 * @actionAlias media.exportToCsv
	 * @param KalturaMediaEsearchExportToCsvJobData $data job data indicating filter to pass to the job
	 * @return string
	 *
	 * @throws APIErrors::USER_EMAIL_NOT_FOUND
	 */
	public function entryExportToCsvAction (KalturaMediaEsearchExportToCsvJobData $data)
	{
		if(!$data->userName || !$data->userMail)
			throw new KalturaAPIException(APIErrors::USER_EMAIL_NOT_FOUND, '');
		
		$kJobdData = $data->toObject(new kMediaEsearchExportToCsvJobData());
		
		kJobsManager::addExportCsvJob($kJobdData, $this->getPartnerId(), ElasticSearchPlugin::getExportTypeCoreValue(EsearchMediaEntryExportObjectType::ESEARCH_MEDIA));
		
		return $data->userMail;
	}

	/**
	 * @param kBaseSearch $coreSearchObject
	 * @param $searchParams
	 * @param $pager
	 * @return array
	 */
	protected function initAndSearch($coreSearchObject, $searchParams, $pager)
	{
		list($coreSearchOperator, $objectStatusesArr, $objectIdsCsvStr, $objectIdsNotIn, $kPager, $coreOrder, $aggregations, $ignoreSynonym) =
			self::initSearchActionParams($searchParams, $pager);
		if (isset($ignoreSynonym))
		{
			$coreSearchObject->setIgnoreSynonymFromQuery(true);
			$coreSearchObject->getQueryAttributes()->setIgnoreSynonymOnPartner($ignoreSynonym);
		}
		$elasticResults = $coreSearchObject->doSearch($coreSearchOperator, $kPager, $objectStatusesArr, $objectIdsCsvStr, $coreOrder, $aggregations, $objectIdsNotIn);

		list($coreResults, $objectCount, $aggregationsResult) = kESearchCoreAdapter::transformElasticToCoreObject($elasticResults, $coreSearchObject);
		return array($coreResults, $objectCount, $aggregationsResult);
	}

	protected static function initSearchActionParams($searchParams, KalturaPager $pager = null)
	{
		/**
		 * @var ESearchParams $coreParams
		 */
		$coreParams = $searchParams->toObject();

		$objectStatusesArr = array();
		$objectStatuses = $coreParams->getObjectStatuses();
		if (isset($objectStatuses))
		{
			$objectStatusesArr = explode(',', $objectStatuses);
		}

		$kPager = null;
		if ($pager)
		{
			/* @var kPager $kPager */
			$kPager = $pager->toObject();
			if( ($kPager->getPageSize() * $kPager->getPageIndex()) > self::MAX_RESULT_WINDOW )
			{
				throw new KalturaAPIException(KalturaESearchErrors::CRITERIA_EXCEEDED_MAX_MATCHES_ALLOWED);
			}
		}
		
		$ignoreSynonym = false;
		if ($coreParams->getIgnoreSynonym())
		{
			$ignoreSynonym = true;
		}

		return array($coreParams->getSearchOperator(), $objectStatusesArr, $coreParams->getObjectId(), $coreParams->getObjectIdsNotIn(), $kPager, $coreParams->getOrderBy(), $coreParams->getAggregations(), $ignoreSynonym);
	}

}
