#!/bin/sh
##
## docker-entrypoint.sh
##   Prepares and starts php-fpm in non-daemonized mode
##   All php errors to logged to terminal 
##   Generates a phpinfo-{RANDOM}.php file in the DOCUMENT_ROOT. 
## 
## Author: D. Bird <doug@katmore.com> 
##

##
## script localization
##
ME_NAME='docker-entrypoint.sh'

##
## infinite sleep loop
##
if [ "$START_FPM_DAEMON" != "1" ]; then
   echo "$ME_NAME: set the environment variable 'START_FPM_DAEMON' to '1' to enable php-fpm on startup"
   echo "$ME_NAME: php-fpm not enabled on startup, sleeping..."
   while :; do sleep 2073600; done
   exit 0
fi

##
## runtime start
##
STARTTIME=$(date '+%FT%T')

##
## php-fpm paths
##
FPM_DAEMON=php-fpm7
FPM_CONF=/etc/php7/php-fpm.conf
FPM_START_COMMAND="$FPM_DAEMON --nodaemonize --force-stderr --fpm-config $FPM_CONF"
FPM_TEST_COMMAND="$FPM_DAEMON --test --fpm-config $FPM_CONF"

##
## sanity check status
##
SANITY_STATUS=0

##
## DOCUMENT_ROOT sanity enforcement
##
[ -z "$DOCUMENT_ROOT" ] && { >&2 echo "$ME_NAME: missing expected 'DOCUMENT_ROOT' environment variable"; SANITY_STATUS=1; }
[ -d "$DOCUMENT_ROOT" ] || { >&2 echo "$ME_NAME: document root not found ($DOCUMENT_ROOT)"; SANITY_STATUS=1; }

##
## enforce php-fpm.conf exists
##   and that 'pid' setting value could be extracted
##
if  [ -f "$FPM_CONF" ]; then
   FPM_PIDFILE=$(sed -n 's/^pid[ =]*//p' $FPM_CONF 2> /dev/null)
   if [ -z "$FPM_PIDFILE" ]; then
      >&2 echo "$ME_NAME: could not find value for 'pid' setting in php-fpm config ($FPM_CONF)"
      SANITY_STATUS=1
   fi
else
   >&2 echo "$ME_NAME: php-fpm config file not found ($FPM_CONF)"
   SANITY_STATUS=1
fi


##
## enforce FPM_DAEMON exists
##
$FPM_DAEMON -v > /dev/null || { >&2 echo "$ME_NAME: system missing '$FPM_DAEMON' command"; SANITY_STATUS=1; }

##
## terminate on sanity check failure
##
[ "$SANITY_STATUS" -ne "0" ] && { >&2 echo "$ME_NAME: (FATAL) one or more sanity checks failed"; exit $SANITY_STATUS; }

##
## resolve FPM LISTEN
##
[ -f "$FPM_CONF" ] && CONF_FPM_LISTEN=$(sed -n 's/^listen[ =]*//p' $FPM_CONF 2> /dev/null)
if [ -n "$FPM_PORT" ]; then
   FPM_LISTEN=$FPM_PORT
else
   [ -n "$CONF_FPM_LISTEN" ] || {
      >&2 echo "$ME_NAME: (FATAL) cannot determine php-fpm port; both the FPM_PORT environment variable and the php-fpm.conf listen setting are missing"
      exit 1
   }
   FPM_LISTEN=$CONF_FPM_LISTEN
fi
if [ "$CONF_FPM_LISTEN" != "$FPM_LISTEN" ]; then
   [ -f "$FPM_CONF" ] || {
      >&2 echo "$ME_NAME: (FATAL) php-fpm listen directive differs from FPM_PORT environment variable, but the FPM_CONF environment variable is missing or is an invalid path"
      exit 1
   }
   sed -i -e '/^listen[ =]*/d' $FPM_CONF || {
      >&2 echo "$ME_NAME: (FATAL) failed to remove existing php-fpm listen setting from '$FPM_CONF'"
      exit 1
   }
   echo "listen = $FPM_LISTEN" >> $FPM_CONF || {
      >&2 echo "$ME_NAME: (FATAL) failed to append the updated php-fpm listen setting to '$FPM_CONF'"
      exit 1
   }
   echo "$ME_NAME: changed listen setting to '$FPM_LISTEN' "
fi
##
## concatenate and resolve path for
##   phpinfo-{RANDOM-STRING}.php
##
PHPINFO_BASENAME="phpinfo-"$(cat /dev/urandom | tr -dc 'a-zA-Z0-9' | fold -w 12 | head -n 1)".php"
PHPINFO_PATH="$DOCUMENT_ROOT/$PHPINFO_BASENAME"

##
## ensure php-fpm pid file directory exists
##
if [ ! -d $(dirname "${FPM_PIDFILE}") ]; then
   mkdir -p $(dirname "${FPM_PIDFILE}")
   CMD_STATUS=$?
   [ "$CMD_STATUS" -ne "0" ] && { >&2 echo "$ME_NAME: failed to create php-fpm pid file directory " \
      $(dirname "${FPM_PIDFILE}")"', 'mkdir' terminated with exit status $CMD_STATUS"; exit $CMD_STATUS; }
   echo "$ME_NAME: created php-fpm pid file directory ("$(dirname "${FPM_PIDFILE}")")"
fi

##
## test php-fpm configuration
##
$FPM_TEST_COMMAND
CMD_STATUS=$?
[ "$CMD_STATUS" -ne "0" ] && { >&2 echo "$ME_NAME: php-fpm config test failed"; exit $CMD_STATUS; }

##
## remove any stale phpinfo-{RANDOM-STRING}.php files
##
for file in $DOCUMENT_ROOT/phpinfo-* ; do         
  if [ -f "$file" ]; then
     echo "$ME_NAME: removing $(basename $file)"
     rm $file
     CMD_STATUS=$?
     if [ "$CMD_STATUS" -ne "0" ]; then
        >&2 echo "$ME_NAME: failed to remove '$(basename $file)' $CMD_STATUS ($file)"
        exit $CMD_STATUS
     fi        
  fi
done

##
## 'finish' function to invoke on EXIT
##
finish() {
   ##
   ## cleanup phpinfo-{RANDOM-STRING}.php on exit
   ##
   if [ -f "$PHPINFO_PATH" ]; then
      rm $PHPINFO_PATH
      if [ "$CMD_STATUS" -eq "0" ]; then
         echo "$ME_NAME: removed $PHPINFO_BASENAME"
      else
         >&2 echo "$ME_NAME: failed to remove $PHPINFO_PATH, 'rm' exit status $CMD_STATUS"
      fi
   fi
}
trap finish EXIT

##
## generate phpinfo-{RANDOM-STRING}.php
##
echo "<?php" > $PHPINFO_PATH
echo "//" >> $PHPINFO_PATH
echo "// $PHPINFO_BASENAME" >> $PHPINFO_PATH
echo "// -- AUTOMATICALLY GENERATED BY $ME_NAME -- " >> $PHPINFO_PATH
echo "// -- $STARTTIME --" >> $PHPINFO_PATH
echo "//" >> $PHPINFO_PATH
echo "phpinfo();" >> $PHPINFO_PATH
echo "" >> $PHPINFO_PATH

##
## display some diagnostic info
##
echo "$ME_NAME: diagnostic info:"
echo "  php-fpm daemon: $FPM_DAEMON"
echo "  php-fpm start command: $FPM_START_COMMAND"
echo "  php-fpm port: $FPM_LISTEN"
echo "  php modules: "$($FPM_DAEMON -m | sed --expression=':a;N;$!ba;s/\n/ /g')
echo "  php version: "$($FPM_DAEMON -v | sed 's/[^0-9.]*\([0-9.]*\).*/\1/' | head -1)
echo "  phpinfo: $PHPINFO_BASENAME"
echo ""

##
## start php-fpm
##
echo "$ME_NAME: starting "$(basename "$FPM_DAEMON")"..."
$FPM_START_COMMAND
CMD_STATUS=$?
[ "$CMD_STATUS" -ne "0" ] && {>&2 echo "$ME_NAME: '$FPM_DAEMON' terminated with exit status $CMD_STATUS"; exit $CMD_STATUS; }
exit 0









