﻿// Copyright (c) .NET Foundation and contributors. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

using Microsoft.Build.Framework;
using Microsoft.Build.Utilities;
using System.Runtime.InteropServices;

namespace Microsoft.DotNet.Cli.Build
{
    public class GenerateBuildVersionInfo : ToolTask
    {
        [Required]
        public string RepoRoot { get; set; }

        [Output]
        public int VersionMajor { get; set; }

        [Output]
        public int VersionMinor { get; set; }

        [Output]
        public int VersionPatch { get; set; }

        [Output]
        public string CommitCount { get; set; }

        [Output]
        public string ReleaseSuffix { get; set; }

        [Output]
        public string MsiVersion { get; set; }

        [Output]
        public string VersionBadgeMoniker { get; set; }

        [Output]
        public string Channel { get; set; }

        [Output]
        public string BranchName { get; set; }

        private int _commitCount;

        public override bool Execute()
        {
            base.Execute();

            var branchInfo = new BranchInfo(RepoRoot);

            var buildVersion = new Version
            {
                Major = int.Parse(branchInfo.Entries["MAJOR_VERSION"]),
                Minor = int.Parse(branchInfo.Entries["MINOR_VERSION"]),
                Patch = int.Parse(branchInfo.Entries["PATCH_VERSION"]),
                ReleaseSuffix = branchInfo.Entries["RELEASE_SUFFIX"],
                CommitCount = _commitCount
            };

            VersionMajor = buildVersion.Major;
            VersionMinor = buildVersion.Minor;
            VersionPatch = buildVersion.Patch;
            CommitCount = buildVersion.CommitCountString;
            ReleaseSuffix = buildVersion.ReleaseSuffix;
            MsiVersion = buildVersion.GenerateMsiVersion();
            VersionBadgeMoniker = Monikers.GetBadgeMoniker();
            Channel = branchInfo.Entries["CHANNEL"];
            BranchName= branchInfo.Entries["BRANCH_NAME"];

            return true;
        }

        protected override string ToolName
        {
            get { return "git"; }
        }

        protected override MessageImportance StandardOutputLoggingImportance
        {
            get { return MessageImportance.High; } // or else the output doesn't get logged by default
        }

        protected override string GenerateFullPathToTool()
        {
            // Workaround: https://github.com/Microsoft/msbuild/issues/1215
            // There's a "git" folder on the PATH in VS 2017 Developer command prompt and it causes msbuild to fail to execute git.
            return RuntimeInformation.IsOSPlatform(OSPlatform.Windows) ? "git.exe" : "git";
        }

        protected override string GenerateCommandLineCommands()
        {
            return $"rev-list --count HEAD";
        }

        protected override void LogEventsFromTextOutput(string line, MessageImportance importance)
        {
            _commitCount = int.Parse(line);
        }
    }
}