package main

import (
	"fmt"
	"os"
	"testing"

	"github.com/kosli-dev/cli/internal/testHelpers"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
)

// Define the suite, and absorb the built-in basic suite
// functionality from testify - including a T() method which
// returns the current testing context
type AttestGitlabPRCommandTestSuite struct {
	flowName            string
	trailName           string
	tmpDir              string
	commitWithPR        string
	commitWithNoPR      string
	artifactFingerprint string
	suite.Suite
	defaultKosliArguments string
}

func (suite *AttestGitlabPRCommandTestSuite) SetupTest() {
	testHelpers.SkipIfEnvVarUnset(suite.Suite.T(), []string{"KOSLI_GITLAB_TOKEN"})

	suite.flowName = "attest-gitlab-pr"
	suite.trailName = "test-123"
	suite.commitWithPR = "f6d2c1a288f2c400c04e8451f4fdddb1f3b4ce01"
	suite.commitWithNoPR = "30a7a221184e88524adabbccabe8a3d63371e443"
	suite.artifactFingerprint = "7509e5bda0c762d2bac7f90d758b5b2263fa01ccbc542ab5e3df163be08e6ca9"
	global = &GlobalOpts{
		ApiToken: "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJpZCI6ImNkNzg4OTg5In0.e8i_lA_QrEhFncb05Xw6E_tkCHU9QfcY4OLTVUCHffY",
		Org:      "docs-cmd-test-user",
		Host:     "http://localhost:8001",
	}

	var err error
	suite.tmpDir, err = os.MkdirTemp("", "testDir")
	require.NoError(suite.Suite.T(), err)
	_, err = testHelpers.CloneGitRepo("https://gitlab.com/kosli-dev/merkely-gitlab-demo.git", suite.tmpDir)
	require.NoError(suite.Suite.T(), err)

	suite.defaultKosliArguments = fmt.Sprintf(" --flow %s --trail %s --repo-root %s --host %s --org %s --api-token %s", suite.flowName, suite.trailName, suite.tmpDir, global.Host, global.Org, global.ApiToken)
	CreateFlowWithTemplate(suite.flowName, "testdata/valid_template.yml", suite.Suite.T())
	BeginTrail(suite.trailName, suite.flowName, "", suite.Suite.T())
	CreateArtifactOnTrail(suite.flowName, suite.trailName, "cli", suite.artifactFingerprint, "file1", suite.Suite.T())
}

func (suite *AttestGitlabPRCommandTestSuite) TearDownSuite() {
	os.RemoveAll(suite.tmpDir)
}

func (suite *AttestGitlabPRCommandTestSuite) TestAttestGitlabPRCmd() {
	tests := []cmdTestCase{
		{
			wantError: true,
			name:      "01 fails when more arguments are provided",
			cmd:       fmt.Sprintf("attest pullrequest gitlab foo bar --commit %s %s", suite.commitWithPR, suite.defaultKosliArguments),
			golden:    "Error: accepts at most 1 arg(s), received 2 [foo bar]\n",
		},
		{
			wantError: true,
			name:      "02 fails when missing a required flags",
			cmd:       fmt.Sprintf("attest pullrequest gitlab foo -t file --commit %s %s", suite.commitWithPR, suite.defaultKosliArguments),
			golden:    "Error: required flag(s) \"gitlab-org\", \"name\", \"repository\" not set\n",
		},
		{
			wantError: true,
			name:      "03 fails when both --fingerprint and --artifact-type",
			cmd:       fmt.Sprintf("attest pullrequest gitlab testdata/file1 --fingerprint xxxx --artifact-type file --name bar --commit %s %s", suite.commitWithPR, suite.defaultKosliArguments),
			golden:    "Error: only one of --fingerprint, --artifact-type is allowed\n",
		},
		{
			wantError: true,
			name:      "04 fails when --fingerprint is not valid",
			cmd:       fmt.Sprintf("attest pullrequest gitlab --name foo --fingerprint xxxx --commit %s %s", suite.commitWithPR, suite.defaultKosliArguments),
			golden:    "Error: xxxx is not a valid SHA256 fingerprint. It should match the pattern ^([a-f0-9]{64})$\nUsage: kosli attest pullrequest gitlab [IMAGE-NAME | FILE-PATH | DIR-PATH] [flags]\n",
		},
		{
			wantError: true,
			name:      "05 attesting against an artifact that does not exist fails",
			cmd: fmt.Sprintf(`attest pullrequest gitlab --fingerprint 1234e5bda0c762d2bac7f90d758b5b2263fa01ccbc542ab5e3df163be08e6ca9 --name foo 
						--gitlab-org kosli-dev --repository merkely-gitlab-demo --commit %s %s`, suite.commitWithPR, suite.defaultKosliArguments),
			goldenRegex: "found 1 merge request\\(s\\) for commit: .*\nError: Artifact with fingerprint 1234e5bda0c762d2bac7f90d758b5b2263fa01ccbc542ab5e3df163be08e6ca9 does not exist in trail \"test-123\" of flow \"attest-gitlab-pr\" belonging to organization \"docs-cmd-test-user\"\n",
		},
		{
			name: "06 can attest gitlab pr against an artifact using artifact name and --artifact-type",
			cmd: fmt.Sprintf(`attest pullrequest gitlab testdata/file1 --artifact-type file --name foo 
					--gitlab-org kosli-dev --repository merkely-gitlab-demo  --commit %s %s`, suite.commitWithPR, suite.defaultKosliArguments),
			goldenRegex: "found 1 merge request\\(s\\) for commit: .*\ngitlab merge request attestation 'foo' is reported to trail: test-123\n",
		},
		{
			name: "07 can attest gitlab pr against an artifact using artifact name and --artifact-type when --name does not exist in the trail template",
			cmd: fmt.Sprintf(`attest pullrequest gitlab testdata/file1 --artifact-type file --name bar 
					--gitlab-org kosli-dev --repository merkely-gitlab-demo  --commit %s %s`, suite.commitWithPR, suite.defaultKosliArguments),
			goldenRegex: "found 1 merge request\\(s\\) for commit: .*\ngitlab merge request attestation 'bar' is reported to trail: test-123\n",
		},
		{
			name: "08 can attest gitlab pr against an artifact using --fingerprint",
			cmd: fmt.Sprintf(`attest pullrequest gitlab --fingerprint 7509e5bda0c762d2bac7f90d758b5b2263fa01ccbc542ab5e3df163be08e6ca9 --name foo 
					--gitlab-org kosli-dev --repository merkely-gitlab-demo  --commit %s %s`, suite.commitWithPR, suite.defaultKosliArguments),
			goldenRegex: "found 1 merge request\\(s\\) for commit: .*\ngitlab merge request attestation 'foo' is reported to trail: test-123\n",
		},
		{
			name: "09 can attest gitlab pr against a trail",
			cmd: fmt.Sprintf(`attest pullrequest gitlab --name bar 
				--gitlab-org kosli-dev --repository merkely-gitlab-demo --commit %s %s`, suite.commitWithPR, suite.defaultKosliArguments),
			goldenRegex: "found 1 merge request\\(s\\) for commit: .*\ngitlab merge request attestation 'bar' is reported to trail: test-123\n",
		},
		{
			name: "10 can attest gitlab pr against a trail when name is not found in the trail template",
			cmd: fmt.Sprintf(`attest pullrequest gitlab --name additional 
					--gitlab-org kosli-dev --repository merkely-gitlab-demo --commit %s %s`, suite.commitWithPR, suite.defaultKosliArguments),
			goldenRegex: "found 1 merge request\\(s\\) for commit: .*\ngitlab merge request attestation 'additional' is reported to trail: test-123\n",
		},
		{
			name: "11 can attest gitlab pr against an artifact it is created using dot syntax in --name",
			cmd: fmt.Sprintf(`attest pullrequest gitlab --name cli.foo 
				--gitlab-org kosli-dev --repository merkely-gitlab-demo --commit %s %s`, suite.commitWithPR, suite.defaultKosliArguments),
			goldenRegex: "found 1 merge request\\(s\\) for commit: .*\ngitlab merge request attestation 'foo' is reported to trail: test-123\n",
		},
		{
			wantError: true,
			name:      "12 assert fails with non-zero exit code when commit has no merge requests",
			cmd: fmt.Sprintf(`attest pullrequest gitlab --name bar 
				--gitlab-org kosli-dev --repository merkely-gitlab-demo --commit %s --assert %s`, suite.commitWithNoPR, suite.defaultKosliArguments),
			goldenRegex: "found 0 merge request\\(s\\) for commit: .*\ngitlab merge request attestation 'bar' is reported to trail: test-123\nError: assert failed: no merge request found for the given commit: .*\n",
		},
		{
			name: "13 assert works and has zero exit code when commit has merge request(s)",
			cmd: fmt.Sprintf(`attest pullrequest gitlab --name bar 
				--gitlab-org kosli-dev --repository merkely-gitlab-demo --commit %s --assert %s`, suite.commitWithPR, suite.defaultKosliArguments),
			goldenRegex: "found 1 merge request\\(s\\) for commit: .*\ngitlab merge request attestation 'bar' is reported to trail: test-123\n",
		},
		{
			wantError: true,
			name:      "14 if there is a server error, this is output even when assert fails",
			cmd: fmt.Sprintf(`attest pullrequest gitlab --fingerprint 1234e5bda0c762d2bac7f90d758b5b2263fa01ccbc542ab5e3df163be08e6ca9 --name foo 
				--gitlab-org kosli-dev --repository merkely-gitlab-demo --commit %s --assert %s`, suite.commitWithNoPR, suite.defaultKosliArguments),
			goldenRegex: "found 0 merge request\\(s\\) for commit: .*\nError: Artifact with fingerprint 1234e5bda0c762d2bac7f90d758b5b2263fa01ccbc542ab5e3df163be08e6ca9 does not exist in trail \"test-123\" of flow \"attest-gitlab-pr\" belonging to organization \"docs-cmd-test-user\"\nError: assert failed: no merge request found for the given commit: .*\n",
		},
		{
			name: "15 can attest github pr even if commit has no merge request",
			cmd: fmt.Sprintf(`attest pullrequest gitlab --name bar 
				--gitlab-org kosli-dev --repository merkely-gitlab-demo --commit %s %s`, suite.commitWithNoPR, suite.defaultKosliArguments),
			goldenRegex: "found 0 merge request\\(s\\) for commit: .*\ngitlab merge request attestation 'bar' is reported to trail: test-123\n",
		},
	}

	runTestCmd(suite.Suite.T(), tests)
}

// In order for 'go test' to run this suite, we need to create
// a normal test function and pass our suite to suite.Run
func TestAttestGitlabPRCommandTestSuite(t *testing.T) {
	suite.Run(t, new(AttestGitlabPRCommandTestSuite))
}
