package main

import (
	"fmt"
	"io"
	"net/http"
	"os"

	"github.com/kosli-dev/cli/internal/requests"
	"github.com/kosli-dev/cli/internal/sonar"
	"github.com/spf13/cobra"
)

type SonarAttestationPayload struct {
	*CommonAttestationPayload
	SonarResults *sonar.SonarResults `json:"sonar_results"`
}

type attestSonarOptions struct {
	*CommonAttestationOptions
	apiToken   string
	workingDir string
	ceTaskURL  string
	projectKey string
	serverURL  string
	revision   string
	maxWait    int
	payload    SonarAttestationPayload
}

const attestSonarShortDesc = `Report a SonarQube attestation to an artifact or a trail in a Kosli flow.  `

const attestSonarLongDesc = attestSonarShortDesc + `
Retrieves results for the specified scan from SonarQube Cloud or SonarQube Server and attests them to Kosli.
The results are parsed to find the status of the project's quality gate which is used to determine the attestation's compliance status.

The scan to be retrieved can be specified in two ways:
1. (Default) Using metadata created by the Sonar scanner. By default this is located within a temporary ^.scannerwork^ folder in the repo base directory.
If you have overriden the location of this folder by passing parameters to the Sonar scanner, or are running Kosli's CLI locally outside the repo's base directory,
you can provide the correct path using the ^--sonar-working-dir^ flag. This metadata is generated by a specific scan, allowing Kosli to retrieve the results of that scan.  
If there are delays in the scan processing (either because the scanned project is very large, or because SonarQube is experiencing processing delays), it may happen that 
the scan results are not available by the time the attest sonar command is executed. In this case you can use the ^--max-wait^ flag to retry the command while waiting for the scan to be processed. 
This flag takes the maximum number of seconds to wait for the results to be available. The Kosli CLI will then attempt to retrieve the scan results until the maximum wait time is reached, with 
exponential backoff between retries. Once the results are available they are attested to Kosli as usual.

2. Providing the Sonar project key and the revision of the scan (plus the SonarQube server URL if relevant). If running the Kosli CLI in some CI/CD pipeline, the revision
is defaulted to the commit SHA. If you are running the command locally, or have overriden the revision in SonarQube via parameters to the Sonar scanner, you can
provide the correct revision using the ^--sonar-revision^ flag. Kosli then finds the scan results for the specified project key and revision.

Note that if your project is very large and you are using SonarQube Cloud's automatic analysis, it is possible for the attest sonar command to run before the SonarQube Cloud scan is completed.
In this case, we recommend using Kosli's Sonar webhook integration ( https://docs.kosli.com/integrations/sonar/ ) rather than the CLI to attest the scan results.
` + attestationBindingDesc

const attestSonarExample = `
# report a SonarQube Cloud attestation about a trail using SonarQube's metadata, with no retries:
kosli attest sonar \
	--name yourAttestationName \
	--flow yourFlowName \
	--trail yourTrailName \
	--sonar-api-token yourSonarAPIToken \
	--sonar-working-dir yourSonarWorkingDirPath \
	--api-token yourAPIToken \
	--org yourOrgName \

# report a SonarQube Server attestation about a trail using SonarQube's metadata, waiting for up to 60 seconds for the results to be available:
kosli attest sonar \
	--name yourAttestationName \
	--flow yourFlowName \
	--trail yourTrailName \
	--sonar-api-token yourSonarAPIToken \
	--sonar-working-dir yourSonarWorkingDirPath \
	--api-token yourAPIToken \
	--org yourOrgName \
	--max-wait 60

# report a SonarQube Cloud attestation for a specific branch about a trail using key/revision:
kosli attest sonar \
	--name yourAttestationName \
	--flow yourFlowName \
	--trail yourTrailName \
	--sonar-api-token yourSonarAPIToken \
	--sonar-project-key yourSonarProjectKey \
	--sonar-revision yourSonarRevision \
	--branch-name yourBranchName \
	--api-token yourAPIToken \
	--org yourOrgName \

# report a SonarQube Server attestation for a pull-request about a trail using key/revision:
kosli attest sonar \
	--name yourAttestationName \
	--flow yourFlowName \
	--trail yourTrailName \
	--sonar-api-token yourSonarAPIToken \
	--sonarqube-url yourSonarQubeURL \
	--sonar-project-key yourSonarProjectKey \
	--sonar-revision yourSonarRevision \
	--pull-request-id yourPullRequestID \
	--api-token yourAPIToken \
	--org yourOrgName \

# report a SonarQube Cloud attestation about a trail with an attachment using SonarQube's metadata, waiting for up to 300 seconds for the results to be available:
kosli attest sonar \
	--name yourAttestationName \
	--flow yourFlowName \
	--trail yourTrailName \
	--sonar-api-token yourSonarAPIToken \
	--sonar-working-dir yourSonarWorkingDirPath \
	--attachment yourAttachmentPath \
	--api-token yourAPIToken \
	--org yourOrgName \
	--max-wait 300
`

func newAttestSonarCmd(out io.Writer) *cobra.Command {
	o := &attestSonarOptions{
		CommonAttestationOptions: &CommonAttestationOptions{
			fingerprintOptions: &fingerprintOptions{},
		},
		payload: SonarAttestationPayload{
			CommonAttestationPayload: &CommonAttestationPayload{},
		},
	}

	cmd := &cobra.Command{
		// Args:    cobra.MaximumNArgs(1),  // See CustomMaximumNArgs() below
		Use:     "sonar [IMAGE-NAME | FILE-PATH | DIR-PATH]",
		Short:   attestSonarShortDesc,
		Long:    attestSonarLongDesc,
		Example: attestSonarExample,
		PreRunE: func(cmd *cobra.Command, args []string) error {

			err := CustomMaximumNArgs(1, args)
			if err != nil {
				return err
			}

			err = RequireGlobalFlags(global, []string{"Org", "ApiToken"})
			if err != nil {
				return ErrorBeforePrintingUsage(cmd, err.Error())
			}

			err = MuXRequiredFlags(cmd, []string{"fingerprint", "artifact-type"}, false)
			if err != nil {
				return err
			}

			err = ValidateAttestationArtifactArg(args, o.fingerprintOptions.artifactType, o.payload.ArtifactFingerprint)
			if err != nil {
				return ErrorBeforePrintingUsage(cmd, err.Error())
			}

			return ValidateRegistryFlags(cmd, o.fingerprintOptions)

		},
		RunE: func(cmd *cobra.Command, args []string) error {
			return o.run(args)
		},
	}

	ci := WhichCI()
	addAttestationFlags(cmd, o.CommonAttestationOptions, o.payload.CommonAttestationPayload, ci)
	cmd.Flags().StringVar(&o.apiToken, "sonar-api-token", "", sonarAPITokenFlag)
	cmd.Flags().StringVar(&o.workingDir, "sonar-working-dir", ".scannerwork", sonarWorkingDirFlag)
	cmd.Flags().StringVar(&o.projectKey, "sonar-project-key", "", sonarProjectKeyFlag)
	cmd.Flags().StringVar(&o.serverURL, "sonar-server-url", "https://sonarcloud.io", sonarServerURLFlag)
	cmd.Flags().StringVar(&o.revision, "sonar-revision", o.commitSHA, sonarRevisionFlag)
	cmd.Flags().IntVar(&o.maxWait, "max-wait", 30, sonarMaxWaitFlag)

	err := RequireFlags(cmd, []string{"flow", "trail", "name", "sonar-api-token"})
	if err != nil {
		logger.Error("failed to configure required flags: %v", err)
	}

	return cmd
}

func (o *attestSonarOptions) run(args []string) error {
	url := fmt.Sprintf("%s/api/v2/attestations/%s/%s/trail/%s/sonar", global.Host, global.Org, o.flowName, o.trailName)

	err := o.CommonAttestationOptions.run(args, o.payload.CommonAttestationPayload)
	if err != nil {
		return err
	}

	sc := sonar.NewSonarConfig(o.apiToken, o.workingDir, o.ceTaskURL, o.projectKey, o.serverURL, o.revision, o.maxWait)

	o.payload.SonarResults, err = sc.GetSonarResults(logger)
	if err != nil {
		return err
	}

	form, cleanupNeeded, evidencePath, err := prepareAttestationForm(o.payload, o.attachments)
	if err != nil {
		return err
	}
	// if we created a tar package, remove it after uploading it
	if cleanupNeeded {
		defer os.Remove(evidencePath)
	}

	reqParams := &requests.RequestParams{
		Method: http.MethodPost,
		URL:    url,
		Form:   form,
		DryRun: global.DryRun,
		Token:  global.ApiToken,
	}
	_, err = kosliClient.Do(reqParams)
	if err == nil && !global.DryRun {
		logger.Info("sonar attestation '%s' is reported to trail: %s", o.payload.AttestationName, o.trailName)
	}

	return wrapAttestationError(err)

}
