package main

import (
	"fmt"
	"testing"

	"github.com/kosli-dev/cli/internal/testHelpers"
	"github.com/stretchr/testify/suite"
)

// Define the suite, and absorb the built-in basic suite
// functionality from testify - including a T() method which
// returns the current testing context
type CommitEvidencePRAzureCommandTestSuite struct {
	suite.Suite
	defaultKosliArguments string
	flowNames             string
}

func (suite *CommitEvidencePRAzureCommandTestSuite) SetupTest() {
	testHelpers.SkipIfEnvVarUnset(suite.Suite.T(), []string{"KOSLI_AZURE_TOKEN"})

	suite.flowNames = "azure-pr"
	global = &GlobalOpts{
		ApiToken: "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJpZCI6ImNkNzg4OTg5In0.e8i_lA_QrEhFncb05Xw6E_tkCHU9QfcY4OLTVUCHffY",
		Org:      "docs-cmd-test-user",
		Host:     "http://localhost:8001",
	}
	suite.defaultKosliArguments = fmt.Sprintf(" --host %s --org %s --api-token %s", global.Host, global.Org, global.ApiToken)

	CreateFlow(suite.flowNames, suite.Suite.T())
}

func (suite *CommitEvidencePRAzureCommandTestSuite) TestCommitEvidencePRAzureCmd() {
	tests := []cmdTestCase{
		{
			name: "report Azure PR evidence works",
			cmd: `report evidence commit pullrequest azure --name az-pr --flows ` + suite.flowNames + `
			          --build-url http://www.example.com --azure-org-url https://dev.azure.com/kosli --project kosli-azure --repository cli --commit 5f61be8f00a01c84e491922a630c9a418c684c7a` + suite.defaultKosliArguments,
			goldenRegex: "found 1 pull request\\(s\\) for commit: 5f61be8f00a01c84e491922a630c9a418c684c7a\nazure pull request evidence is reported to commit: 5f61be8f00a01c84e491922a630c9a418c684c7a\n",
		},
		{
			wantError: true,
			name:      "report Azure PR evidence fails when --org is missing",
			cmd: `report evidence commit pullrequest azure --name az-pr --flows ` + suite.flowNames + `
			          --build-url http://www.example.com --azure-org-url https://dev.azure.com/kosli --project kosli-azure --repository cli --commit 5f61be8f00a01c84e491922a630c9a418c684c7a --api-token foo --host bar`,
			goldenRegex: "Error: --org is not set\n" +
				"Usage: kosli report evidence commit pullrequest azure \\[flags\\]\n",
		},
		{
			wantError: true,
			name:      "report Azure PR evidence fails when --name is missing",
			cmd: `report evidence commit pullrequest azure --flows ` + suite.flowNames + ` --azure-org-url https://dev.azure.com/kosli --project kosli-azure
			          --build-url http://www.example.com --repository cli --commit 5f61be8f00a01c84e491922a630c9a418c684c7a` + suite.defaultKosliArguments,
			goldenRegex: "Error: required flag\\(s\\) \"name\" not set\n",
		},
		{
			wantError: true,
			name:      "report Azure PR evidence fails when --azure-org-url is missing",
			cmd: `report evidence commit pullrequest azure --name az-pr --flows ` + suite.flowNames + `
			          --build-url http://www.example.com --project kosli-azure --repository cli --commit 5f61be8f00a01c84e491922a630c9a418c684c7a` + suite.defaultKosliArguments,
			goldenRegex: "Error: required flag\\(s\\) \"azure-org-url\" not set\n",
		},
		{
			wantError: true,
			name:      "report Azure PR evidence fails when --project is missing",
			cmd: `report evidence commit pullrequest azure --name az-pr --flows ` + suite.flowNames + `
			          --build-url http://www.example.com --azure-org-url https://dev.azure.com/kosli --repository cli --commit 5f61be8f00a01c84e491922a630c9a418c684c7a` + suite.defaultKosliArguments,
			goldenRegex: "Error: required flag\\(s\\) \"project\" not set\n",
		},
		{
			wantError: true,
			name:      "report Azure PR evidence fails when --repository is missing",
			cmd: `report evidence commit pullrequest azure --name az-pr --flows ` + suite.flowNames + `
			          --build-url http://www.example.com --azure-org-url https://dev.azure.com/kosli --project kosli-azure --commit 5f61be8f00a01c84e491922a630c9a418c684c7a` + suite.defaultKosliArguments,
			goldenRegex: "Error: required flag\\(s\\) \"repository\" not set\n",
		},
		{
			wantError: true,
			name:      "report Azure PR evidence fails when --commit is missing",
			cmd: `report evidence commit pullrequest azure --name az-pr --flows ` + suite.flowNames + `
			          --build-url http://www.example.com --azure-org-url https://dev.azure.com/kosli --project kosli-azure --repository cli` + suite.defaultKosliArguments,
			goldenRegex: "Error: required flag\\(s\\) \"commit\" not set\n",
		},
		{
			name: "report Azure PR evidence does not fail when commit does not exist, empty evidence is reported instead",
			cmd: `report evidence commit pullrequest azure --name az-pr --flows ` + suite.flowNames + `
			          --build-url http://www.example.com --azure-org-url https://dev.azure.com/kosli --project kosli-azure --repository cli --commit 1111111111111111111111111111111111111111` + suite.defaultKosliArguments,
			goldenRegex: "found 0 pull request\\(s\\) for commit: 1111111111111111111111111111111111111111\n" +
				"azure pull request evidence is reported to commit: 1111111111111111111111111111111111111111\n",
		},
		{
			wantError: true,
			name:      "report Azure PR evidence fails when --assert is used and commit has no PRs",
			cmd: `report evidence commit pullrequest azure --name az-pr --flows ` + suite.flowNames + ` --assert
					--build-url http://www.example.com --azure-org-url https://dev.azure.com/kosli --project kosli-azure --repository cli --commit 1a877d0c3cf4644b4225bf3eb23ced26818d685a` + suite.defaultKosliArguments,
			goldenRegex: "found 0 pull request\\(s\\) for commit: .*\nazure pull request evidence is reported to commit: .*\nError: assert failed: no pull request found for the given commit: .*\n",
		},
		{
			name: "report Azure PR evidence does not fail when commit has no PRs",
			cmd: `report evidence commit pullrequest azure --name az-pr --flows ` + suite.flowNames + `
			          --build-url http://www.example.com --azure-org-url https://dev.azure.com/kosli --project kosli-azure --repository cli --commit 1a877d0c3cf4644b4225bf3eb23ced26818d685a` + suite.defaultKosliArguments,
			goldenRegex: "found 0 pull request\\(s\\) for commit: .*\nazure pull request evidence is reported to commit: .*\n",
		},
		{
			wantError: true,
			name:      "report Azure PR evidence fails when --user-data is not found",
			cmd: `report evidence commit pullrequest azure --name az-pr --flows ` + suite.flowNames + `
					  --user-data non-existing.json
			          --build-url http://www.example.com --azure-org-url https://dev.azure.com/kosli --project kosli-azure --repository cli --commit 5f61be8f00a01c84e491922a630c9a418c684c7a` + suite.defaultKosliArguments,
			goldenRegex: "Error: open non-existing.json: no such file or directory\n",
		},
	}

	runTestCmd(suite.Suite.T(), tests)
}

// In order for 'go test' to run this suite, we need to create
// a normal test function and pass our suite to suite.Run
func TestCommitEvidencePRAzureCommandTestSuite(t *testing.T) {
	suite.Run(t, new(CommitEvidencePRAzureCommandTestSuite))
}
