package main

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/suite"
)

// Define the suite, and absorb the built-in basic suite
// functionality from testify - including a T() method which
// returns the current testing context
type CommitEvidenceSnykCommandTestSuite struct {
	suite.Suite
	defaultKosliArguments string
	flowName              string
}

func (suite *CommitEvidenceSnykCommandTestSuite) SetupTest() {
	suite.flowName = "snyk-test"
	global = &GlobalOpts{
		ApiToken: "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJpZCI6ImNkNzg4OTg5In0.e8i_lA_QrEhFncb05Xw6E_tkCHU9QfcY4OLTVUCHffY",
		Org:      "docs-cmd-test-user-shared",
		Host:     "http://localhost:8001",
	}
	suite.defaultKosliArguments = fmt.Sprintf(" --host %s --org %s --api-token %s", global.Host, global.Org, global.ApiToken)

	CreateFlow(suite.flowName, suite.Suite.T())
}

func (suite *CommitEvidenceSnykCommandTestSuite) TestCommitEvidenceSnykCmd() {
	tests := []cmdTestCase{
		{
			name: "report Snyk test evidence works",
			cmd: `report evidence commit snyk --commit 239d7cee00ca341f124fa710fc694b67cdf8011b --name snyk-result --flows ` + suite.flowName + `
			          --build-url http://www.example.com --scan-results testdata/snyk_scan_example.json` + suite.defaultKosliArguments,
			goldenRegex: "snyk scan evidence is reported to commit: 239d7cee00ca341f124fa710fc694b67cdf8011b\n",
		},
		{
			name: "report Snyk test evidence works  when --evidence-url and --evidence-fingerprint are provided",
			cmd: `report evidence commit snyk --commit 239d7cee00ca341f124fa710fc694b67cdf8011b --name snyk-result --flows ` + suite.flowName + `
			          --build-url http://www.example.com --scan-results testdata/snyk_scan_example.json 
					  --evidence-url https://http://www.example.com --evidence-fingerprint 847411c6124e719a4e8da2550ac5c116b7ff930493ce8a061486b48db8a5aaa0` + suite.defaultKosliArguments,
			goldenRegex: "snyk scan evidence is reported to commit: 239d7cee00ca341f124fa710fc694b67cdf8011b\n",
		},
		{
			name: "report Snyk scan evidence with non-existing scan-results",
			cmd: `report evidence commit snyk --commit 239d7cee00ca341f124fa710fc694b67cdf8011b --name snyk-result --flows ` + suite.flowName + `
			          --build-url http://www.example.com --scan-results testdata/foo.json` + suite.defaultKosliArguments,
			wantError:   true,
			goldenRegex: "Error: failed to parse Snyk results file \\[testdata\\/foo.json\\]. Failed to parse as Sarif: open testdata\\/foo.json: no such file or directory. Fallen back to parse Snyk Json, but also failed: the provided file path doesn't have a file\n",
		},
		{
			name: "report Snyk scan evidence with missing scan-results flag",
			cmd: `report evidence commit snyk --commit 239d7cee00ca341f124fa710fc694b67cdf8011b --name snyk-result --flows ` + suite.flowName + `
			          --build-url http://www.example.com` + suite.defaultKosliArguments,
			wantError:   true,
			goldenRegex: "Error: required flag\\(s\\) \"scan-results\" not set\n",
		},
		{
			name: "report Snyk scan evidence with missing name flag",
			cmd: `report evidence commit snyk --commit 239d7cee00ca341f124fa710fc694b67cdf8011b --flows ` + suite.flowName + `
			          --build-url http://www.example.com --scan-results testdata/snyk_scan_example.json` + suite.defaultKosliArguments,
			wantError:   true,
			goldenRegex: "Error: required flag\\(s\\) \"name\" not set\n",
		},
		{
			name: "report Snyk scan evidence with a missing --flows flag works",
			cmd: `report evidence commit snyk --commit 239d7cee00ca341f124fa710fc694b67cdf8011b --name snyk-result
			          --build-url http://www.example.com --scan-results testdata/snyk_scan_example.json` + suite.defaultKosliArguments,
			goldenRegex: "snyk scan evidence is reported to commit: 239d7cee00ca341f124fa710fc694b67cdf8011b\n",
		},
		{
			name: "report Snyk scan evidence with a missing build-url fails",
			cmd: `report evidence commit snyk --commit 239d7cee00ca341f124fa710fc694b67cdf8011b --flows ` + suite.flowName + `
			         --name snyk-result --scan-results testdata/snyk_scan_example.json` + suite.defaultKosliArguments,
			wantError:   true,
			goldenRegex: "Error: required flag\\(s\\) \"build-url\" not set\n",
		},
		{
			name: "report Snyk scan with a sarif file and upload it",
			cmd: `report evidence commit snyk --commit 239d7cee00ca341f124fa710fc694b67cdf8011b --name snyk-result
			          --build-url http://www.example.com --scan-results testdata/snyk_sarif.json` + suite.defaultKosliArguments,
			goldenRegex: "snyk scan evidence is reported to commit: 239d7cee00ca341f124fa710fc694b67cdf8011b\n",
		},
		{
			name: "report Snyk scan with a sarif file and DO NOT upload it",
			cmd: `report evidence commit snyk --commit 239d7cee00ca341f124fa710fc694b67cdf8011b --name snyk-result
			          --build-url http://www.example.com --scan-results testdata/snyk_sarif.json --upload-results=false` + suite.defaultKosliArguments,
			goldenRegex: "snyk scan evidence is reported to commit: 239d7cee00ca341f124fa710fc694b67cdf8011b\n",
		},
	}
	runTestCmd(suite.Suite.T(), tests)
}

// In order for 'go test' to run this suite, we need to create
// a normal test function and pass our suite to suite.Run
func TestCommitEvidenceSnykCommandTestSuite(t *testing.T) {
	suite.Run(t, new(CommitEvidenceSnykCommandTestSuite))
}
