package io.quarkus.qute;

import io.quarkus.qute.SectionHelperFactory.BlockInfo;
import io.quarkus.qute.TemplateNode.Origin;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

/**
 * Each section consists of one or more blocks. The main block is always present. Additional blocks start with a label
 * definition: <code>{#label param1}</code>.
 */
public final class SectionBlock {

    public final Origin origin;

    /**
     * Id generated by the parser. {@code main} for the main block.
     */
    public final String id;
    /**
     * Label used for the given part. {@code main} for the main block.
     */
    public final String label;
    /**
     * An unmodifiable ordered map of parsed parameters.
     */
    public final Map<String, String> parameters;

    /**
     * An unmodifiable ordered map of parameter expressions.
     */
    public final Map<String, Expression> expressions;

    /**
     * Section content - an immutable list of template nodes.
     */
    List<TemplateNode> nodes;

    public SectionBlock(Origin origin, String id, String label, Map<String, String> parameters,
            Map<String, Expression> expressions,
            List<TemplateNode> nodes) {
        this.origin = origin;
        this.id = id;
        this.label = label;
        this.parameters = parameters;
        this.expressions = expressions;
        this.nodes = ImmutableList.copyOf(nodes);
    }

    public boolean isEmpty() {
        return nodes.isEmpty();
    }

    List<Expression> getExpressions() {
        List<Expression> expressions = new ArrayList<>();
        expressions.addAll(this.expressions.values());
        for (TemplateNode node : nodes) {
            expressions.addAll(node.getExpressions());
        }
        return expressions;
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("SectionBlock [origin=").append(origin).append(", id=").append(id).append(", label=").append(label)
                .append("]");
        return builder.toString();
    }

    void optimizeNodes(Set<TemplateNode> nodesToRemove) {
        List<TemplateNode> effectiveNodes = new ArrayList<>();
        boolean hasLineSeparator = false;
        boolean nodeIgnored = false;
        for (TemplateNode node : nodes) {
            if (node instanceof SectionNode) {
                effectiveNodes.add(node);
                ((SectionNode) node).optimizeNodes(nodesToRemove);
            } else if (node == Parser.COMMENT_NODE || (node instanceof ParameterDeclarationNode)
                    || nodesToRemove.contains(node)) {
                // Ignore comments, param declarations and nodes for removal
                nodeIgnored = true;
            } else {
                effectiveNodes.add(node);
                if (node instanceof LineSeparatorNode) {
                    hasLineSeparator = true;
                }
            }
        }

        if (!hasLineSeparator && !nodeIgnored) {
            // No optimizations are possible
            return;
        }

        if (hasLineSeparator) {
            List<TemplateNode> finalNodes;
            // Collapse adjacent text and line separator nodes
            finalNodes = new ArrayList<>();
            List<TextNode> textGroup = null;
            for (TemplateNode node : effectiveNodes) {
                if (node instanceof TextNode) {
                    if (textGroup == null) {
                        textGroup = new ArrayList<>();
                    }
                    textGroup.add((TextNode) node);
                } else {
                    if (textGroup != null) {
                        collapseGroup(textGroup, finalNodes);
                        textGroup = null;
                    }
                    finalNodes.add(node);
                }
            }
            if (textGroup != null) {
                collapseGroup(textGroup, finalNodes);
            }
            nodes = ImmutableList.copyOf(finalNodes);
        } else if (nodeIgnored) {
            nodes = ImmutableList.copyOf(effectiveNodes);
        }
    }

    private void collapseGroup(List<TextNode> group, List<TemplateNode> finalNodes) {
        if (group.size() > 1) {
            // Collapse the group...
            StringBuilder val = new StringBuilder();
            for (TextNode textNode : group) {
                val.append(textNode.getValue());
            }
            finalNodes.add(new TextNode(val.toString(), group.get(0).getOrigin()));
        } else {
            finalNodes.add(group.get(0));
        }
    }

    static SectionBlock.Builder builder(String id, Function<String, Expression> expressionFunc,
            Function<String, TemplateException> errorFun) {
        return new Builder(id, expressionFunc, errorFun).setLabel(id);
    }

    static class Builder implements BlockInfo {

        private final String id;
        private Origin origin;
        private String label;
        private Map<String, String> parameters;
        private final List<TemplateNode> nodes;
        private Map<String, Expression> expressions;
        private final Function<String, Expression> expressionFun;
        private final Function<String, TemplateException> errorFun;

        public Builder(String id, Function<String, Expression> expressionFun, Function<String, TemplateException> errorFun) {
            this.id = id;
            this.nodes = new ArrayList<>();
            this.expressionFun = expressionFun;
            this.errorFun = errorFun;
        }

        SectionBlock.Builder setOrigin(Origin origin) {
            this.origin = origin;
            return this;
        }

        SectionBlock.Builder addNode(TemplateNode node) {
            nodes.add(node);
            return this;
        }

        SectionBlock.Builder setLabel(String label) {
            this.label = label;
            return this;
        }

        SectionBlock.Builder addParameter(String name, String value) {
            if (parameters == null) {
                parameters = new LinkedHashMap<>();
            }
            parameters.put(name, value);
            return this;
        }

        @Override
        public Expression addExpression(String param, String value) {
            Expression expression = expressionFun.apply(value);
            if (expressions == null) {
                expressions = new LinkedHashMap<>();
            }
            expressions.put(param, expression);
            return expression;
        }

        public Map<String, String> getParameters() {
            return parameters == null ? Collections.emptyMap() : Collections.unmodifiableMap(parameters);
        }

        public String getLabel() {
            return label;
        }

        @Override
        public TemplateException createParserError(String message) {
            return errorFun.apply(message);
        }

        SectionBlock build() {
            Map<String, String> parameters = this.parameters;
            if (parameters == null) {
                parameters = Collections.emptyMap();
            } else if (parameters.size() == 1) {
                parameters = Map.copyOf(parameters);
            } else {
                parameters = Collections.unmodifiableMap(parameters);
            }
            Map<String, Expression> expressions = this.expressions;
            if (expressions == null) {
                expressions = Collections.emptyMap();
            } else if (expressions.size() == 1) {
                expressions = Map.copyOf(expressions);
            } else {
                expressions = Collections.unmodifiableMap(expressions);
            }
            return new SectionBlock(origin, id, label, parameters, expressions, nodes);
        }
    }

}
