package me.liuwj.ktorm.entity

import me.liuwj.ktorm.database.prepareStatement
import me.liuwj.ktorm.dsl.AliasRemover
import me.liuwj.ktorm.expression.*
import me.liuwj.ktorm.schema.*

/**
 * 将给定的实体对象插入的表中，如果插入后有主键生成，会将生成的主键设置到实体对象中的相应属性，返回受影响的记录数
 */
@Suppress("UNCHECKED_CAST")
fun <E : Entity<E>> Table<E>.add(entity: E): Int {
    entity.implementation.checkUnexpectedDiscarding(this)

    val assignments = findInsertColumns(entity).takeIf { it.isNotEmpty() } ?: return 0

    val expression = AliasRemover.visit(
        expr = InsertExpression(
            table = asExpression(),
            assignments = assignments.map { (col, argument) ->
                ColumnAssignmentExpression(
                    column = col.asExpression() as ColumnExpression<Any>,
                    expression = ArgumentExpression(argument, col.sqlType as SqlType<Any>)
                )
            }
        )
    )

    val useGeneratedKey = primaryKey?.binding != null && entity.implementation.getPrimaryKeyValue(this) == null

    expression.prepareStatement(autoGeneratedKeys = useGeneratedKey) { statement, logger ->
        val effects = statement.executeUpdate().also { logger.debug("Effects: {}", it) }

        if (useGeneratedKey) {
            statement.generatedKeys.use { rs ->
                if (rs.next()) {
                    val generatedKey = primaryKey?.sqlType?.getResult(rs, 1)
                    if (generatedKey != null) {
                        logger.debug("Generated Key: {}", generatedKey)
                        entity.implementation.setPrimaryKeyValue(this, generatedKey)
                    }
                }
            }
        }

        entity.implementation.fromTable = this
        entity.implementation.doDiscardChanges()
        return effects
    }
}

private fun Table<*>.findInsertColumns(entity: Entity<*>): Map<Column<*>, Any?> {
    val implementation = entity.implementation
    val assignments = LinkedHashMap<Column<*>, Any?>()

    for (column in columns) {
        if (column is SimpleColumn && column.binding != null) {
            val value = implementation.getColumnValue(column)
            if (value != null) {
                assignments[column] = value
            }
        }
    }

    return assignments
}

@Suppress("UNCHECKED_CAST")
internal fun EntityImplementation.doFlushChanges(): Int {
    val fromTable = this.fromTable?.takeIf { this.parent == null } ?: error("The entity is not associated with any table yet.")
    checkUnexpectedDiscarding(fromTable)

    val primaryKey = fromTable.primaryKey ?: error("Table ${fromTable.tableName} doesn't have a primary key.")
    val assignments = findChangedColumns(fromTable).takeIf { it.isNotEmpty() } ?: return 0

    val expression = AliasRemover.visit(
        expr = UpdateExpression(
            table = fromTable.asExpression(),
            assignments = assignments.map { (col, argument) ->
                ColumnAssignmentExpression(
                    column = col.asExpression() as ColumnExpression<Any>,
                    expression = ArgumentExpression(argument, col.sqlType as SqlType<Any>)
                )
            },
            where = BinaryExpression(
                type = BinaryExpressionType.EQUAL,
                left = primaryKey.asExpression(),
                right = ArgumentExpression(getPrimaryKeyValue(fromTable), primaryKey.sqlType as SqlType<Any>),
                sqlType = BooleanSqlType
            )
        )
    )

    expression.prepareStatement { statement, logger ->
        val effects =  statement.executeUpdate().also { logger.debug("Effects: {}", it) }
        doDiscardChanges()
        return effects
    }
}

private fun EntityImplementation.findChangedColumns(fromTable: Table<*>): Map<Column<*>, Any?> {
    val assignments = LinkedHashMap<Column<*>, Any?>()

    for (column in fromTable.columns) {
        val binding = column.binding?.takeIf { column is SimpleColumn } ?: continue

        when (binding) {
            is ReferenceBinding -> {
                if (binding.onProperty.name in changedProperties) {
                    val child = this.getProperty(binding.onProperty.name) as Entity<*>?
                    assignments[column] = child?.implementation?.getPrimaryKeyValue(binding.referenceTable)
                }
            }
            is NestedBinding -> {
                var anyChanged = false
                var curr: Any? = this

                for (prop in binding.properties) {
                    if (curr is Entity<*>) {
                        curr = curr.implementation
                    }

                    check(curr is EntityImplementation?)

                    if (curr != null && prop.name in curr.changedProperties) {
                        anyChanged = true
                    }

                    curr = curr?.getProperty(prop.name)
                }

                if (anyChanged) {
                    assignments[column] = curr
                }
            }
        }
    }

    return assignments
}

internal fun EntityImplementation.doDiscardChanges() {
    val fromTable = this.fromTable?.takeIf { this.parent == null } ?: error("The entity is not associated with any table yet.")

    for (column in fromTable.columns) {
        val binding = column.binding?.takeIf { column is SimpleColumn } ?: continue

        when (binding) {
            is ReferenceBinding -> {
                changedProperties.remove(binding.onProperty.name)
            }
            is NestedBinding -> {
                var curr: Any? = this

                for (prop in binding.properties) {
                    if (curr == null) {
                        break
                    }
                    if (curr is Entity<*>) {
                        curr = curr.implementation
                    }

                    check(curr is EntityImplementation)
                    curr.changedProperties.remove(prop.name)
                    curr = curr.getProperty(prop.name)
                }
            }
        }
    }
}

// Add check to avoid bug #10
private fun EntityImplementation.checkUnexpectedDiscarding(fromTable: Table<*>) {
    for (column in fromTable.columns) {
        val binding = column.binding?.takeIf { column is SimpleColumn } ?: continue

        if (binding is NestedBinding) {
            var curr: Any? = this

            for ((i, prop) in binding.properties.withIndex()) {
                if (curr == null) {
                    break
                }
                if (curr is Entity<*>) {
                    curr = curr.implementation
                }

                check(curr is EntityImplementation)

                if (i > 0 && prop.name in curr.changedProperties && curr.fromTable != null && curr.getRoot() != this) {
                    val propPath = binding.properties.subList(0, i + 1).joinToString(separator = ".", prefix = "this.") { it.name }
                    throw IllegalStateException("$propPath may be unexpectedly discarded, please save it to database first.")
                }

                curr = curr.getProperty(prop.name)
            }
        }
    }
}

private tailrec fun EntityImplementation.getRoot(): EntityImplementation {
    val parent = this.parent
    if (parent == null) {
        return this
    } else {
        return parent.getRoot()
    }
}

internal fun Entity<*>.clearChangesRecursively() {
    implementation.changedProperties.clear()

    for ((_, value) in properties) {
        if (value is Entity<*>) {
            value.clearChangesRecursively()
        }
    }
}

@Suppress("UNCHECKED_CAST")
internal fun EntityImplementation.doDelete(): Int {
    val fromTable = this.fromTable?.takeIf { this.parent == null } ?: error("The entity is not associated with any table yet.")
    val primaryKey = fromTable.primaryKey ?: error("Table ${fromTable.tableName} doesn't have a primary key.")

    val expression = AliasRemover.visit(
        expr = DeleteExpression(
            table = fromTable.asExpression(),
            where = BinaryExpression(
                type = BinaryExpressionType.EQUAL,
                left = primaryKey.asExpression(),
                right = ArgumentExpression(getPrimaryKeyValue(fromTable), primaryKey.sqlType as SqlType<Any>),
                sqlType = BooleanSqlType
            )
        )
    )

    expression.prepareStatement { statement, logger ->
        return statement.executeUpdate().also { logger.debug("Effects: {}", it) }
    }
}
