﻿using Orleans.Runtime;
using Orleans.Serialization;
using Orleans.Streams;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Threading.Tasks;
using Microsoft.Extensions.Logging;

namespace Orleans.Providers.GCP.Streams.PubSub
{
    internal class PubSubAdapter<TDataAdapter> : IQueueAdapter
        where TDataAdapter : IPubSubDataAdapter
    {
        protected readonly string DeploymentId;
        private readonly SerializationManager _serializationManager;
        protected readonly string ProjectId;
        protected readonly string TopicId;
        protected readonly TimeSpan? Deadline;
        private readonly HashRingBasedStreamQueueMapper _streamQueueMapper;
        protected readonly ConcurrentDictionary<QueueId, PubSubDataManager> Subscriptions = new ConcurrentDictionary<QueueId, PubSubDataManager>();
        protected readonly IPubSubDataAdapter _dataAdapter;
        private readonly ILogger _logger;
        private readonly ILoggerFactory loggerFactory;
        private readonly string _customEndpoint;
        public string Name { get; }
        public bool IsRewindable => false;
        public StreamProviderDirection Direction => StreamProviderDirection.ReadWrite;

        public PubSubAdapter(
            TDataAdapter dataAdapter,
            SerializationManager serializationManager,
            ILoggerFactory loggerFactory,
            HashRingBasedStreamQueueMapper streamQueueMapper,
            string projectId,
            string topicId,
            string deploymentId,
            string providerName,
            TimeSpan? deadline = null, 
            string customEndpoint = "")
        {
            if (string.IsNullOrEmpty(projectId)) throw new ArgumentNullException(nameof(projectId));
            if (string.IsNullOrEmpty(topicId)) throw new ArgumentNullException(nameof(topicId));
            if (string.IsNullOrEmpty(deploymentId)) throw new ArgumentNullException(nameof(deploymentId));

            _logger = loggerFactory.CreateLogger($"{this.GetType().FullName}.{providerName}");
            this.loggerFactory = loggerFactory;
            _serializationManager = serializationManager;
            ProjectId = projectId;
            TopicId = topicId;
            DeploymentId = deploymentId;
            Name = providerName;
            Deadline = deadline;
            _streamQueueMapper = streamQueueMapper;
            _dataAdapter = dataAdapter;
            _customEndpoint = customEndpoint;
        }

        public IQueueAdapterReceiver CreateReceiver(QueueId queueId)
        {
            return PubSubAdapterReceiver.Create(_serializationManager, this.loggerFactory, queueId, ProjectId, TopicId, DeploymentId, _dataAdapter, Deadline, _customEndpoint);
        }

        public async Task QueueMessageBatchAsync<T>(Guid streamGuid, string streamNamespace, IEnumerable<T> events, StreamSequenceToken token, Dictionary<string, object> requestContext)
        {
            if (token != null) throw new ArgumentException("Google PubSub stream provider currently does not support non-null StreamSequenceToken.", nameof(token));
            var queueId = _streamQueueMapper.GetQueueForStream(streamGuid, streamNamespace);

            PubSubDataManager pubSub;
            if (!Subscriptions.TryGetValue(queueId, out pubSub))
            {
                var tmpPubSub = new PubSubDataManager(this.loggerFactory, ProjectId, TopicId, queueId.ToString(), DeploymentId, Deadline);
                await tmpPubSub.Initialize();
                pubSub = Subscriptions.GetOrAdd(queueId, tmpPubSub);
            }

            var msg = _dataAdapter.ToPubSubMessage(streamGuid, streamNamespace, events, requestContext);
            await pubSub.PublishMessages(new[] { msg });
        }
    }
}
