#pragma once

#include <string>
#include <vector>
#include <filesystem>
#include <fstream>
#include "host_bindings.hpp"

namespace loongarch::script {

// API Generator - generates binding headers for guest projects
class APIGenerator {
public:
	// Generate C++ API header for guest code
	static std::string generate_cpp_api() {
		std::string output;

		// Header guard and includes
		output += "#pragma once\n\n";
		output += "#include <string>\n";
		output += "#include <vector>\n\n";

		// Add user-defined C++ header content
		output += HostBindings::get_cpp_header();
		output += "\n";

		// Generate extern "C" declarations
		output += "extern \"C\" {\n";
		output += HostBindings::generate_extern_declarations();
		output += "}\n\n";

		// Generate assembly stubs
		output += HostBindings::generate_asm_stubs();
		output += "\n";

		// Add fast_exit helper
		output += R"(
// Fast exit for vmcall support
__asm__(
	".pushsection .text\n"
	".global fast_exit\n"
	".type fast_exit, @function\n"
	"fast_exit:\n"
	"  move $zero, $zero\n"
	".popsection\n"
);
extern "C" __attribute__((noreturn)) void fast_exit(int code);
)";

		return output;
	}

	// Generate Rust API bindings
	static std::string generate_rust_api() {
		std::string output;

		output += "// Auto-generated Rust bindings for libloong scripting\n";
		output += "// DO NOT EDIT - Generated by libloong API generator\n\n";

		// Add user-defined Rust header content
		if (!HostBindings::get_rust_header().empty()) {
			output += HostBindings::get_rust_header();
			output += "\n";
		}

		// Add type definitions for complex types
		output += "// Opaque types for guest<->host communication\n";
		output += "#[repr(C)]\n";
		output += "pub struct GuestStdString {\n";
		output += "    _private: [u8; 0],\n";
		output += "}\n\n";
		output += "#[repr(C)]\n";
		output += "pub struct GuestRustString {\n";
		output += "    ptr: u64,\n";
		output += "    capacity: u64,\n";
		output += "    len: u64,\n";
		output += "}\n\n";

		// Generate assembly stubs for host functions
		output += generate_rust_asm_stubs();
		output += "\n";

		// Generate unsafe extern "C" declarations (sys_* functions)
		output += "// Private FFI declarations (unsafe)\n";
		output += "extern \"C\" {\n";
		for (const auto& [name, binding] : HostBindings::get_bindings()) {
			output += "    ";
			output += translate_cpp_to_rust_signature(binding.signature, true);
			output += "\n";
		}
		output += "}\n\n";

		// Generate safe public wrapper functions
		output += "// Public safe API wrappers\n";
		for (const auto& [name, binding] : HostBindings::get_bindings()) {
			output += generate_safe_wrapper(name, binding.signature);
			output += "\n";
		}
		output += "\n";

		// Add fast_exit
		output += R"(
// Fast exit for vmcall support
core::arch::global_asm!(
    ".pushsection .text\n",
    ".global fast_exit\n",
    ".type fast_exit, @function\n",
    "fast_exit:\n",
    "  move $zero, $zero\n",
    ".popsection\n"
);
extern "C" {
    pub fn fast_exit(code: i32) -> !;
}
)";

		return output;
	}

	// Write C++ API to file
	static void write_cpp_api(const std::filesystem::path& output_path) {
		std::ofstream file(output_path);
		if (!file) {
			throw std::runtime_error("Failed to write C++ API to: " + output_path.string());
		}
		file << generate_cpp_api();
	}

	// Write Rust API to file with optional project path for scanning
	static void write_rust_api(const std::filesystem::path& output_path,
	                           const std::filesystem::path& project_path = "") {
		std::ofstream file(output_path);
		if (!file) {
			throw std::runtime_error("Failed to write Rust API to: " + output_path.string());
		}

		// Generate base API
		std::string api_content = generate_rust_api();

		// If project path provided, scan for #[no_mangle] functions and generate keep-alive code
		if (!project_path.empty() && std::filesystem::exists(project_path)) {
			// Also write a .cargo/config.toml with linker flags to keep the symbols
			write_rust_cargo_config(output_path.parent_path(), project_path);
		}

		file << api_content;
	}

private:
	// Generate Rust global assembly stubs for host functions
	static std::string generate_rust_asm_stubs() {
		std::string output;

		output += "// Assembly stubs for host functions (will be live-patched at runtime)\n";
		output += "core::arch::global_asm!(\n";
		output += "    \".pushsection .text\\n\",\n";

		for (const auto& [name, binding] : HostBindings::get_bindings()) {
			// Generate sys_* function name for FFI
			std::string sys_name = "sys_" + name;
			output += "    \".global " + sys_name + "\\n\",\n";
			output += "    \".type " + sys_name + ", @function\\n\",\n";
			output += "    \"" + sys_name + ":\\n\",\n";
			output += "    \"  break 0\\n\",\n";
		}

		output += "    \".popsection\\n\"";
		output += "\n);\n";
		return output;
	}

	// Translate C++ function signature to Rust
	static std::string translate_cpp_to_rust_signature(const std::string& cpp_sig, bool add_sys_prefix = false) {
		// Parse C++ signature: "return_type function_name(args)"
		// Output Rust: "pub fn function_name(args) -> return_type;"
		// If add_sys_prefix is true, output: "fn sys_function_name(args) -> return_type;"

		std::string sig = cpp_sig;

		// Find opening parenthesis
		size_t paren_pos = sig.find('(');
		if (paren_pos == std::string::npos) {
			throw std::runtime_error("Invalid C++ signature: " + cpp_sig);
		}

		// Extract function name (search backwards from paren)
		size_t name_end = paren_pos;
		while (name_end > 0 && std::isspace(sig[name_end - 1])) {
			--name_end;
		}
		size_t name_start = name_end;
		while (name_start > 0 && (std::isalnum(sig[name_start - 1]) || sig[name_start - 1] == '_')) {
			--name_start;
		}
		std::string func_name = sig.substr(name_start, name_end - name_start);

		// Extract return type (everything before function name)
		std::string return_type_cpp = sig.substr(0, name_start);
		// Trim whitespace
		while (!return_type_cpp.empty() && std::isspace(return_type_cpp.back())) {
			return_type_cpp.pop_back();
		}

		// Extract arguments (between parentheses)
		size_t close_paren = sig.find(')', paren_pos);
		if (close_paren == std::string::npos) {
			throw std::runtime_error("Invalid C++ signature: " + cpp_sig);
		}
		std::string args_cpp = sig.substr(paren_pos + 1, close_paren - paren_pos - 1);

		// Translate return type
		std::string return_type_rust = translate_cpp_type_to_rust(return_type_cpp, func_name);

		// Translate arguments
		std::string args_rust = translate_cpp_args_to_rust(args_cpp, func_name);

		// Build Rust signature
		std::string actual_func_name = add_sys_prefix ? ("sys_" + func_name) : func_name;
		std::string visibility = add_sys_prefix ? "" : "pub ";
		std::string result = visibility + "fn " + actual_func_name + "(" + args_rust + ")";
		if (return_type_rust != "()") {
			result += " -> " + return_type_rust;
		}
		result += ";";

		return result;
	}

	// Generate a safe wrapper function that calls the unsafe sys_* function
	static std::string generate_safe_wrapper(const std::string& name, const std::string& cpp_sig) {
		// Parse the C++ signature to extract return type and arguments
		std::string sig = cpp_sig;

		// Find opening parenthesis
		size_t paren_pos = sig.find('(');
		if (paren_pos == std::string::npos) {
			return "";
		}

		// Extract function name
		size_t name_end = paren_pos;
		while (name_end > 0 && std::isspace(sig[name_end - 1])) {
			--name_end;
		}
		size_t name_start = name_end;
		while (name_start > 0 && (std::isalnum(sig[name_start - 1]) || sig[name_start - 1] == '_')) {
			--name_start;
		}

		// Extract return type
		std::string return_type_cpp = sig.substr(0, name_start);
		while (!return_type_cpp.empty() && std::isspace(return_type_cpp.back())) {
			return_type_cpp.pop_back();
		}

		// Extract arguments
		size_t close_paren = sig.find(')', paren_pos);
		if (close_paren == std::string::npos) {
			return "";
		}
		std::string args_cpp = sig.substr(paren_pos + 1, close_paren - paren_pos - 1);

		// Translate return type
		std::string return_type_rust = translate_cpp_type_to_rust(return_type_cpp, name);

		// Translate arguments
		std::string args_rust = translate_cpp_args_to_rust(args_cpp, name);

		// Build wrapper function
		std::string output = "#[inline]\npub fn " + name + "(" + args_rust + ")";
		if (return_type_rust != "()") {
			output += " -> " + return_type_rust;
		}
		output += " {\n";
		output += "    unsafe {\n";
		output += "        sys_" + name + "(";

		// Add argument forwarding
		if (!args_cpp.empty()) {
			std::vector<std::string> args = split_args(args_cpp);
			for (size_t i = 0; i < args.size(); ++i) {
				if (i > 0) output += ", ";

				// Extract argument name
				std::string arg = args[i];
				size_t last_space = arg.rfind(' ');
				std::string arg_name;
				if (last_space != std::string::npos) {
					arg_name = arg.substr(last_space + 1);
					while (!arg_name.empty() && std::isspace(arg_name.front())) {
						arg_name.erase(0, 1);
					}
				} else {
					arg_name = "arg" + std::to_string(i);
				}
				output += arg_name;
			}
		}

		output += ")\n";
		output += "    }\n";
		output += "}\n";

		return output;
	}

	static std::string translate_cpp_type_to_rust(const std::string& cpp_type, const std::string& func_name = "") {
		std::string type = cpp_type;

		// Track if this was a const reference
		bool was_const_ref = false;

		// Trim whitespace
		while (!type.empty() && std::isspace(type.front())) {
			type.erase(0, 1);
		}
		while (!type.empty() && std::isspace(type.back())) {
			type.pop_back();
		}

		// Handle const
		if (type.find("const ") == 0) {
			type = type.substr(6);
			while (!type.empty() && std::isspace(type.front())) {
				type.erase(0, 1);
			}
		}

		// Handle references (strip &)
		if (!type.empty() && type.back() == '&') {
			was_const_ref = true;
			type.pop_back();
			while (!type.empty() && std::isspace(type.back())) {
				type.pop_back();
			}
		}

		// Basic type mappings
		if (type == "void") return "()";
		if (type == "int") return "i32";
		if (type == "int32_t") return "i32";
		if (type == "unsigned int" || type == "unsigned") return "u32";
		if (type == "uint32_t") return "u32";
		if (type == "long") return "i64";
		if (type == "int64_t") return "i64";
		if (type == "unsigned long") return "u64";
		if (type == "uint64_t") return "u64";
		if (type == "short") return "i16";
		if (type == "int16_t") return "i16";
		if (type == "unsigned short") return "u16";
		if (type == "uint16_t") return "u16";
		if (type == "char") return "i8";
		if (type == "int8_t") return "i8";
		if (type == "unsigned char") return "u8";
		if (type == "uint8_t") return "u8";
		if (type == "float") return "f32";
		if (type == "double") return "f64";
		if (type == "bool") return "bool";

		// Handle std::string
		if (type.find("std::string") != std::string::npos) {
			if (was_const_ref) {
				// For Rust functions with const std::string&, use &String
				return "&String";
			} else {
				return "*const GuestRustString";
			}
		}

		// Handle vectors - extract the element type
		if (type.find("std::vector<") != std::string::npos) {
			size_t start = type.find('<') + 1;
			size_t end = type.rfind('>');
			if (start < end) {
				std::string element_type = type.substr(start, end - start);
				// Trim
				while (!element_type.empty() && std::isspace(element_type.front())) {
					element_type.erase(0, 1);
				}
				while (!element_type.empty() && std::isspace(element_type.back())) {
					element_type.pop_back();
				}

				// Translate element type
				std::string rust_element = translate_cpp_type_to_rust(element_type, func_name);

				if (was_const_ref) {
					// For Rust functions with const std::vector<T>&, use &Vec<T>
					return "&Vec<" + rust_element + ">";
				} else {
					return "*const core::ffi::c_void";
				}
			}
		}

		// Default: return as-is (may need manual fixing)
		return type;
	}

	static std::string translate_cpp_args_to_rust(const std::string& cpp_args, const std::string& func_name = "") {
		if (cpp_args.empty()) {
			return "";
		}

		std::string result;
		std::vector<std::string> args = split_args(cpp_args);

		for (size_t i = 0; i < args.size(); ++i) {
			if (i > 0) result += ", ";

			std::string arg = args[i];

			// Parse "type name" or just "type"
			// Find last space or identifier
			size_t last_space = arg.rfind(' ');
			std::string arg_type;
			std::string arg_name;

			if (last_space != std::string::npos) {
				arg_type = arg.substr(0, last_space);
				arg_name = arg.substr(last_space + 1);

				// Trim
				while (!arg_type.empty() && std::isspace(arg_type.back())) {
					arg_type.pop_back();
				}
				while (!arg_name.empty() && std::isspace(arg_name.front())) {
					arg_name.erase(0, 1);
				}
			} else {
				arg_type = arg;
				arg_name = "arg" + std::to_string(i);
			}

			// Translate type
			std::string rust_type = translate_cpp_type_to_rust(arg_type, func_name);

			result += arg_name + ": " + rust_type;
		}

		return result;
	}

	static std::vector<std::string> split_args(const std::string& args) {
		std::vector<std::string> result;
		std::string current;
		int depth = 0;

		for (char c : args) {
			if (c == '<' || c == '(') {
				++depth;
				current += c;
			} else if (c == '>' || c == ')') {
				--depth;
				current += c;
			} else if (c == ',' && depth == 0) {
				// Trim and add
				while (!current.empty() && std::isspace(current.front())) {
					current.erase(0, 1);
				}
				while (!current.empty() && std::isspace(current.back())) {
					current.pop_back();
				}
				if (!current.empty()) {
					result.push_back(current);
				}
				current.clear();
			} else {
				current += c;
			}
		}

		// Add last argument
		while (!current.empty() && std::isspace(current.front())) {
			current.erase(0, 1);
		}
		while (!current.empty() && std::isspace(current.back())) {
			current.pop_back();
		}
		if (!current.empty()) {
			result.push_back(current);
		}

		return result;
	}

	// Scan Rust source files for #[no_mangle] function declarations
	static std::vector<std::string> scan_rust_no_mangle_functions(const std::filesystem::path& project_path) {
		std::vector<std::string> functions;

		// Recursively scan all .rs files
		for (const auto& entry : std::filesystem::recursive_directory_iterator(project_path)) {
			if (!entry.is_regular_file()) continue;
			if (entry.path().extension() != ".rs") continue;

			std::ifstream file(entry.path());
			if (!file) continue;

			bool next_fn_is_no_mangle = false;
			std::string line;
			while (std::getline(file, line)) {
				// Trim whitespace
				std::string trimmed = line;
				while (!trimmed.empty() && std::isspace(trimmed.front())) {
					trimmed.erase(0, 1);
				}
				while (!trimmed.empty() && std::isspace(trimmed.back())) {
					trimmed.pop_back();
				}

				// Check for #[no_mangle]
				if (trimmed == "#[no_mangle]") {
					next_fn_is_no_mangle = true;
					continue;
				}

				// If we found #[no_mangle], look for the function declaration
				if (next_fn_is_no_mangle) {
					// Look for "pub extern "C" fn <name>" or "pub fn <name>"
					if (trimmed.find("fn ") != std::string::npos) {
						// Extract function name
						size_t fn_pos = trimmed.find("fn ");
						if (fn_pos != std::string::npos) {
							fn_pos += 3; // Skip "fn "
							size_t name_start = fn_pos;
							size_t name_end = name_start;

							// Find end of function name (either '(' or '<' for generics)
							while (name_end < trimmed.length() &&
							       (std::isalnum(trimmed[name_end]) || trimmed[name_end] == '_')) {
								++name_end;
							}

							if (name_end > name_start) {
								std::string fn_name = trimmed.substr(name_start, name_end - name_start);
								if (!fn_name.empty() && fn_name != "main") {
									functions.push_back(fn_name);
								}
							}
						}
						next_fn_is_no_mangle = false;
					}
				} else {
					// Reset if we hit a non-attribute, non-empty line
					if (!trimmed.empty() && trimmed[0] != '#') {
						next_fn_is_no_mangle = false;
					}
				}
			}
		}

		return functions;
	}

	// Write .cargo/config.toml with linker flags to prevent DCE
	static void write_rust_cargo_config(const std::filesystem::path& project_root,
	                                    const std::filesystem::path& src_path) {
		auto functions = scan_rust_no_mangle_functions(src_path);

		// Create .cargo directory if it doesn't exist
		std::filesystem::path cargo_dir = project_root / ".cargo";
		std::filesystem::create_directories(cargo_dir);

		// Write config.toml
		std::filesystem::path config_path = cargo_dir / "config.toml";
		std::ofstream file(config_path);
		if (!file) {
			throw std::runtime_error("Failed to write Cargo config to: " + config_path.string());
		}

		file << "# Auto-generated by libloong API generator\n";
		file << "# This config prevents dead code elimination of guest functions and host stubs\n\n";
		file << "[build]\n";
		file << "target = \"loongarch64-unknown-linux-gnu\"\n\n";
		file << "[target.loongarch64-unknown-linux-gnu]\n";
		file << "linker = \"loongarch64-linux-gnu-gcc-14\"\n";
		file << "rustflags = [\n";
		file << "  \"-C\", \"target-feature=+crt-static\",\n";
		file << "  \"-C\", \"link-arg=-static\",\n";
		file << "  \"-C\", \"link-arg=-Wl,-Ttext-segment=0x200000\",\n";

		// Add --undefined flag for guest functions to prevent DCE
		for (const auto& fn_name : functions) {
			file << "  \"-C\", \"link-arg=-Wl,--undefined=" << fn_name << "\",\n";
		}

		// Add --undefined flag for sys_* host function stubs to prevent DCE
		for (const auto& [name, binding] : HostBindings::get_bindings()) {
			file << "  \"-C\", \"link-arg=-Wl,--undefined=sys_" << name << "\",\n";
		}

		file << "]\n";
	}
};

} // loongarch::script
