/**
 * 剪贴板数据及操作上下文
 */

import {reactive, ref} from "vue";
import {selectedTagState} from "./TagDataContext.ts";
import ClipboardDBService from "../../../services/ClipboardDBService.ts";
import {error} from "@tauri-apps/plugin-log";
import {exists} from "@tauri-apps/plugin-fs";
import {copyFileToClipboard, copyToClipboard, initClipboardListener} from "../../../services/ClipboardService.ts";
import {LanguageConfig} from "../../../types/language.ts";
import {hideWindow} from "./WindowContext.ts";
import {checkFileExist, fileExistCache} from "./FileDataContext.ts";
import {emit, listen} from "@tauri-apps/api/event";
import DataClearService from "../../../services/DataClearService.ts";

// 剪贴板监听
let clipboardListener: any = null;

// 内容列表
export const clipboardItems = ref<ClipboardItem[]>([]);

// 搜索框状态
export const searchBoxState = reactive({
    visible: false,
    text: ''
})

// 无限滚动相关状态
export const scrollState = reactive({
    pageSize: 10,
    total: 1000,
    hasMore: true,
    isLoading: false
})

// 选中的内容id
export const selectedItemId = ref<number | undefined>();


/**
 * 加载剪贴板项目列表
 * @param reset 是否重新获取数据
 */
export async function loadClipboardItems(reset: boolean = true) {
    if (reset) {
        // 重置列表和分页状态
        clipboardItems.value = [];
        scrollState.hasMore = true;
        scrollState.total = 0;
    }
    // 如果没有更多数据或正在加载中，则不执行
    if (!scrollState.hasMore || scrollState.isLoading) return;

    try {
        scrollState.isLoading = true;
        // 使用选中的标签ID进行过滤
        const tagId = selectedTagState.selectedTagId;

        const db = await ClipboardDBService.getInstance();
        const {
            total,
            items
        } = await db.searchItems(searchBoxState.text, tagId, scrollState.pageSize, clipboardItems.value && clipboardItems.value.length > 0 ? clipboardItems.value[clipboardItems.value.length - 1].id : undefined);

        // 更新数据列表和分页信息
        if (reset) {
            clipboardItems.value = items;
        } else {
            // 追加数据而不是替换
            clipboardItems.value = [...clipboardItems.value, ...items]
            // items.forEach(item => clipboardItems.value.push(item));
        }

        scrollState.total = total;
        // 修改判断逻辑：只有当获取的数据条数小于pageSize或已加载的总数据等于总条数时，才认为没有更多数据
        scrollState.hasMore = items.length >= scrollState.pageSize && clipboardItems.value.length < total;

        // 预加载图片的base64数据
        for (const item of items) {
            if (item.type === 'file') {
                checkFileExist(item.file_path);
            }
        }
    } catch (err) {
        error('加载剪贴板项目失败:' + err);
    } finally {
        scrollState.isLoading = false;
    }
}

/**
 * 插入剪贴板项目
 * @param item 需要插入的内容
 */
export async function insertClipboardItem(item: ClipboardItem) {
    // 根据查询条件判断是否可以插入
    if (selectedTagState.selectedTagId) {
        // 新加的数据肯定没有绑定标签，可以忽略
        return;
    }
    if (searchBoxState.text
        && searchBoxState.text !== ''
        && !item.content.includes(searchBoxState.text)
        && !item.file_path.includes(searchBoxState.text)
    ) {
        return;
    }
    if (clipboardItems.value.length === 0) {
        clipboardItems.value.unshift({...item});
        return;
    }

    // 查找插入位置（第一个非置顶项的位置）
    let insertIndex = 0;
    for (let i = 0; i < clipboardItems.value.length; i++) {
        if (!clipboardItems.value[i].is_topped) {
            insertIndex = i;
            break;
        }
    }

    // 查找是否已存在该项目
    const existingIndex = clipboardItems.value.findIndex(data => data.id === item.id);

    if (existingIndex !== -1) {
        // 如果项目已存在，先移除再插入
        const [removedItem] = clipboardItems.value.splice(existingIndex, 1);
        // 调整插入位置（如果删除项在插入位置之前）
        const adjustedIndex = existingIndex < insertIndex ? insertIndex - 1 : insertIndex;
        clipboardItems.value.splice(adjustedIndex, 0, {...removedItem, ...item});
    } else {
        // 如果项目不存在，直接插入
        clipboardItems.value.splice(insertIndex, 0, {...item});
    }
}

// 加载更多数据
export function loadMoreItems() {
    // 如果有更多数据，增加页码
    if (!scrollState.isLoading && scrollState.hasMore) {
        loadClipboardItems(false);
    }
}

// 置顶项目
export async function onTop(id: number) {
    selectedItemId.value = id;
    const db = await ClipboardDBService.getInstance();
    await db.toggleTopClipboardItem(id, true);
    await loadClipboardItems(true);
    selectedItemId.value = undefined;
}

// 取消置顶项目
export async function onUnTop(id: number) {
    selectedItemId.value = id;
    const db = await ClipboardDBService.getInstance();
    await db.toggleTopClipboardItem(id, false);
    await loadClipboardItems(true);
    selectedItemId.value = undefined;
}

// 绑定标签
export async function bindTag(item: ClipboardItem, itemIndex: number, tag: TagItem) {
    const db = await ClipboardDBService.getInstance();
    await db.bindItemToTag(item.id, tag.id);
    // 添加标签
    clipboardItems.value[itemIndex].tags.push(tag);
    await emit('update-bind-quantity');
}

// 删除项目的标签
export async function removeItemTag(item: ClipboardItem, tag: TagItem, index: number) {
    selectedItemId.value = item.id;
    const db = await ClipboardDBService.getInstance();
    await db.deleteClipboardItemTag(item.id, tag.id);
    // 删除标签
    clipboardItems.value[index].tags.splice(clipboardItems.value[index].tags.findIndex(tag => tag.id === tag.id), 1);
    selectedItemId.value = undefined;
    await emit('update-bind-quantity');
}

// 删除项目
export async function removeItem(id: number, index: number) {
    selectedItemId.value = id;
    const db = await ClipboardDBService.getInstance();
    await db.deleteClipboardItem(id);
    clipboardItems.value.splice(index, 1);
    selectedItemId.value = undefined;
}

// 将内容复制到系统剪贴板
export async function onCopy(item: ClipboardItem, message: any, language: LanguageConfig) {
    if (selectedItemId.value === item.id) {
        return;
    }
    if (item) {
        let isSuccess;
        if (item.type === 'file') {
            const filePaths: Array<string> = JSON.parse(item.file_path);
            // 过滤出系统存在的文件
            const paths = filePaths.filter(async path => {
                return fileExistCache.get(path) && await exists(path);
            });
            item.file_path = JSON.stringify(paths);
            isSuccess = await copyToClipboard(item);
        } else {
            isSuccess = await copyToClipboard(item);
        }
        if (isSuccess) {
            message.success(language.pages.list.copySuccessMsg);
            toggleSearchBox();
            // 隐藏窗口
            await hideWindow();
        } else {
            message.error(language.pages.list.copyFailedMsg);
        }
    }
}

// 将单个文件复制到系统剪贴板
export async function onCopyFile(itemId: number, filePath: string, message: any, language: LanguageConfig) {
    if (filePath) {
        selectedItemId.value = itemId;
        if (fileExistCache.get(filePath) && await exists(filePath)) {
            const isSuccess = await copyFileToClipboard([filePath]);
            if (isSuccess) {
                message.success(language.pages.list.copySuccessMsg);
                toggleSearchBox();
                // 隐藏窗口
                await hideWindow();
            } else {
                message.error(language.pages.list.copyFailedMsg);
            }
        } else {
            message.error(language.pages.list.fileNotExistCopyFailedMsg);
        }
    }
    selectedItemId.value = undefined;
}

/**
 * 切换搜索框显示状态
 */
export function toggleSearchBox() {
    searchBoxState.visible = !searchBoxState.visible;
    if (searchBoxState.visible) {
        const input = document.querySelector(".n-input__input-el") as HTMLInputElement;
        if (input) {
            input.focus();
        }
    } else {
        // 当搜索框隐藏时，清空搜索内容并重新加载列表
        if (searchBoxState.text) {
            searchBoxState.text = '';
            loadClipboardItems(true);
        }
    }
}

// 检查项目是否已绑定标签
export function isItemTagged(itemId: number, tagId: number) {
    const item = clipboardItems.value.find(item => item.id === itemId);
    if (item && item.tags) {
        return item.tags.some((tag: any) => tag.id === tagId);
    }
    return false;
}

/**
 * 初始化更新数据保留限制监听
 */
let updateDataHistoryRestrictListener: any = null;

async function initUpdateDataHistoryRestrictListener() {
    return await listen('update-data-history-restrict', async (event: any) => {
        const dataRetentionDays = event.payload.dataRetentionDays;
        const maxHistoryItems = event.payload.maxHistoryItems;
        const timer = await DataClearService.getInstance();
        // 更新保留天数
        timer.setDataRetentionDays(dataRetentionDays);
        // 更新保留条数
        timer.setMaxHistoryItems(maxHistoryItems);
        // 停止定时任务
        timer.stopDataClear();
        if (dataRetentionDays > 0) {
            // 当保留天数大于0时，启动定时任务
            timer.startDataClear();
        }
    });
}

/**
 * 初始化剪贴板数据上下文
 * @param settings 程序配置信息
 */
export async function initializeClipboardDataContext(settings: Settings) {
    try {
        // 加载剪贴板项目列表
        await loadClipboardItems(true);

        // 启动剪贴板监听服务
        clipboardListener = await initClipboardListener();

        // 添加更新数据保留限制事件监听
        updateDataHistoryRestrictListener = await initUpdateDataHistoryRestrictListener();

        // 启动自动清理数据任务
        if (settings.dataRetentionDays > 0) {
            const clearTimer = await DataClearService.getInstance();
            clearTimer.startDataClear();
        }
    } catch (e) {
        console.error('初始化剪贴板数据失败:', e)
        error('初始化剪贴板数据失败:' + e);
    }
}

export async function destroyClipboardDataContext() {
    // 清除剪贴板监听服务
    if (clipboardListener) {
        clipboardListener();
    }

    // 清除更新数据保留限制事件监听
    if (updateDataHistoryRestrictListener) {
        updateDataHistoryRestrictListener();
    }

    // 清除数据清理定时器
    const clearTimer = await DataClearService.getInstance();
    clearTimer?.stopDataClear();
}