/**
 * 拖拽控制上下文
 */

import {bindTag, isItemTagged} from "./ClipboardDataContext.ts";
import {reactive} from "vue";

/**
 * 拖拽状态全局变量
 */
export const dragState = reactive({
    isDragging: false,
    dragItem: {} as ClipboardItem,
    dragIndex: -1,
    draggedOverTagId: -1
});

/**
 * 文本内容拖拽
 */
export function contentDragStart(item: ClipboardItem, event: DragEvent) {
    // 设置拖拽数据
    if (event.dataTransfer) {
        event.dataTransfer.setData('text/plain', item.content);
        event.dataTransfer.effectAllowed = 'link';
    }
}

/**
 * 处理拖拽开始事件
 * @param item 剪贴板内容
 * @param index 内容下标
 * @param event 拖拽事件
 */
export function handleDragStart(item: ClipboardItem, index: number, event: DragEvent) {
    dragState.isDragging = true;
    dragState.dragItem = item;
    dragState.dragIndex = index;

    // 设置拖拽数据
    if (event.dataTransfer) {
        event.dataTransfer.setData('text/plain', item.id.toString());
        event.dataTransfer.effectAllowed = 'link';
    }
}

/**
 * 处理拖拽结束事件
 */
export function handleDragEnd() {
    dragState.isDragging = false;
    dragState.dragItem = { } as ClipboardItem;
    dragState.dragIndex = -1;
    dragState.draggedOverTagId = -1;
}

/**
 * 处理拖拽进入标签事件
 * @param tagId 标签id
 */
export function handleDragEnterTag(tagId: number) {
    dragState.draggedOverTagId = tagId;
    // 如果项目已经绑定了该标签，则不允许再次绑定
    if (dragState.dragItem && isItemTagged(dragState.dragItem.id, tagId)) {
        return;
    }
}

/**
 * 处理拖拽离开标签事件
 * @param event 拖拽事件
 */
export function handleDragLeaveTag(event: DragEvent) {
    // 检查是否直接拖拽到了另一个标签上
    // 只有当不是拖拽到其他标签元素上时，才清除draggedOverTagId
    const relatedTarget = event.relatedTarget as HTMLElement;
    if (!relatedTarget || !relatedTarget.closest('.tag-item')) {
        dragState.draggedOverTagId = -1;
    }
}

/**
 * 处理拖拽放置事件
 * @param tag 标签
 */
export async function handleDropOnTag(tag: TagItem) {
    if (dragState.dragItem) {
        // 调用绑定标签接口
        await bindTag(dragState.dragItem, dragState.dragIndex, tag);
        // 重置拖拽状态
        handleDragEnd();
    }
}