/**
 * 文件数据控制上下文
 */

import {isImage, readImageAsBase64} from "../../../utils/fileSystem.ts";
import {error} from "@tauri-apps/plugin-log";
import {exists} from "@tauri-apps/plugin-fs";
import {reactive} from "vue";

/**
 * 图片缓存，用于存储图片的base64数据
 */
export const imageDataCache = reactive(new Map<string, string>());
/**
 * 文件是否存在缓存，用于存储文件是否存在状态
 */
export const fileExistCache = reactive(new Map<string, boolean>());

/**
 * 检查文件是否存在
 * @param filePaths 文件路径列表
 */
export function checkFileExist(filePaths: string) {
    if (!filePaths) {
        return;
    }
    const filePathList: Array<string> = JSON.parse(filePaths);
    if (filePathList.length === 1) {
        const filePath: string = filePathList[0];
        // 处理图片类型的文件
        if (isImage(filePath)) {
            // 这里不需要等待结果，避免处理慢
            loadImageFromPath(filePath);
        }
        loadFileExist(filePath);
    } else {
        filePathList.forEach(filePath => {
            // 处理图片类型的文件
            if (isImage(filePath)) {
                loadImageFromPath(filePath);
            }
            loadFileExist(filePath);
        });
    }
}

/**
 * 从文件系统加载图片
 * @param filePath 文件路径
 */
async function loadImageFromPath(filePath: string | null) {
    if (!filePath) {
        return;
    }
    if (imageDataCache.get(filePath)) {
        return;
    }
    try {
        const base64Image = await readImageAsBase64(filePath);
        if (base64Image) {
            imageDataCache.set(filePath, base64Image);
        }
    } catch (err) {
        error('加载图片失败:' + err);
        throw err;
    }
}

/**
 * 加载文件是否充值
 * @param filePath 文件路径
 */
async function loadFileExist(filePath: string) {
    if (!filePath) {
        return;
    }
    if (fileExistCache.get(filePath)) {
    }
    try {
        fileExistCache.set(filePath, await exists(filePath));
    } catch (e) {
        console.error('检查文件是否存在失败:', e);
    }
}