/**
 * 标签数据控制上下文
 */

import {reactive, ref} from "vue";
import ClipboardDBService from "../../../services/ClipboardDBService.ts";
import {loadClipboardItems} from "./ClipboardDataContext.ts";
import {error} from "@tauri-apps/plugin-log";
import {listen} from "@tauri-apps/api/event";

/**
 * 标签列表
 */
export const TagItems = ref<TagItem[]>();

/**
 * 标签选中状态
 */
export const selectedTagState = reactive({
    selectedTagId: undefined as number | undefined
})

/**
 * 标签设置
 */
export const tagSettingState = reactive({
    isShow: false,
    location: ''
});

/**
 * 加载所有标签
 */
export async function loadTags() {
    const db = await ClipboardDBService.getInstance();
    TagItems.value = await db.getAllTags();
    // todo 暂时先刷新列表重新取数据，避免标签删除了，但是列表数据没更新
    loadClipboardItems(true);
}

/**
 * 处理标签点击事件
 * @param {number} tagId - 标签ID
 */
export function handleTagClick(tagId: number) {
    // 如果点击的是当前已选中的标签，则取消选中
    if (selectedTagState.selectedTagId === tagId) {
        selectedTagState.selectedTagId = undefined;
    } else {
        // 否则选中该标签
        selectedTagState.selectedTagId = tagId;
    }

    // 根据选中的标签过滤剪贴板列表（重置列表）
    loadClipboardItems(true);
}

/**
 * 加载标签设置状态
 */
export async function loadTagSettingState(setting: Settings) {
    tagSettingState.isShow = setting.enableTag;
    tagSettingState.location = setting.bindTagBtnShowLocation;
}

/**
 * 初始化加载标签事件监听
 */
let loadTagsUnListener: any = null;

async function initLoadTagsListener() {
    return await listen('reload-tags', async (_event: any) => {
        await loadTags();
    });
}

/**
 * 初始化更新标签设置状态监听
 */
let updateTagSettingStateListener: any = null;

async function initUpdateTagSettingStateListener() {
    return await listen('update-tag-setting-state', async (event: any) => {
        tagSettingState.isShow = event.payload.isShow;
        tagSettingState.location = event.payload.location;
    });
}

/**
 * 初始化标签上下文
 * @param settings 程序配置信息
 */
export async function initializeTagContext(settings: Settings) {
    try {
        // 加载标签列表
        await loadTags();

        // 加载标签设置状态
        await loadTagSettingState(settings);

        // 添加更新标签设置状态事件监听
        updateTagSettingStateListener = await initUpdateTagSettingStateListener();

        // 添加加载标签事件监听
        loadTagsUnListener = await initLoadTagsListener();
    } catch (e) {
        console.error("标签上下文初始化失败", e)
        error("标签上下文初始化失败" + e);
    }
}

/**
 * 销毁标签上下文
 */
export function destroyTagContext() {
    // 清除加载标签事件监听
    if (loadTagsUnListener) {
        loadTagsUnListener();
    }
    // 清除更新标签设置状态事件监听
    if (updateTagSettingStateListener) {
        updateTagSettingStateListener();
    }

}