/**
 * 窗口控制上下文
 */
import {searchBoxState, toggleSearchBox} from "./ClipboardDataContext.ts";
import {getCurrentWebviewWindow} from "@tauri-apps/api/webviewWindow";
import {listFixedStore} from "../../../store/fixed.ts";
import {listen} from "@tauri-apps/api/event";
import {openAboutWindow, openSettingsWindow} from "../../../utils/window.ts";
import UpdaterService from "../../../services/UpdaterService.ts";
import {getShortcutKeys} from "../../../services/FileService.ts";
import {convertRegisterKey} from "../../../utils/ShortcutKeys.ts";
import {isRegistered, register, unregister} from "@tauri-apps/plugin-global-shortcut";
import {saveWakeUpRoutineKeyAvailable} from "../../../store/ShortcutKeyAvailableStatus.ts";
import {error} from "@tauri-apps/plugin-log";
import {sendNotification} from "@tauri-apps/plugin-notification";
import {currentMonitor, cursorPosition, PhysicalPosition} from "@tauri-apps/api/window";
import {LanguageConfig} from "../../../types/language.ts";
import {ref} from "vue";

// 监听当前窗口是否固定
const listFixedListen = listFixedStore();

// 自动隐藏窗口
export const isAutoHideWindow = ref(false);

/**
 * 隐藏当前窗口
 */
export async function hideWindow() {
    if (!listFixedListen.stateData()) {
        // 失焦隐藏窗口、搜索框显示时，需要隐藏搜索框
        if (searchBoxState.visible) {
            toggleSearchBox();
        }
        // 窗口未固定，触发失焦，隐藏窗口
        await getCurrentWebviewWindow().hide();
    }
}

/**
 * 注册打开当前窗口快捷键
 * @param shortcutKeys 快捷键
 */
async function registerOpenWindowKey(shortcutKeys: string) {
    await register(shortcutKeys, async (event) => {
        // 只监听按键按下，松开忽略
        if (event.state === 'Released') {
            return;
        }
        const win = getCurrentWebviewWindow();
        const visible = await win.isVisible();
        if (!visible) {
            const {x, y} = await cursorPosition();

            // 获取窗口大小
            const size = await win.outerSize();
            const monitor = await currentMonitor();

            // 计算窗口位置，确保不会超出屏幕边界
            let posX = x;
            let posY = y;

            if (monitor) {
                const {width: screenWidth, height: screenHeight} = monitor.size;

                // 如果窗口右边界超出屏幕，则向左偏移
                if (x + size.width > screenWidth) {
                    posX = screenWidth - size.width;
                }

                // 如果窗口下边界超出屏幕，则向上偏移
                if (y + size.height > screenHeight) {
                    posY = screenHeight - size.height;
                }
            }

            // 设置窗口位置并显示
            await win.setPosition(new PhysicalPosition(posX, posY));
            await win.show();
            await win.setFocus();
            window.focus();
            document.body.focus();
        } else {
            await hideWindow();
        }
    });
}

/**
 * 打开当前窗口快捷键注册
 */
async function registerShortcutKeysOpenWindow(language: LanguageConfig) {
    const keys = await getShortcutKeys();
    try {
        if (keys.wakeUpRoutine && keys.wakeUpRoutine.key && keys.wakeUpRoutine.key.length > 0) {
            const shortcutKeys = convertRegisterKey(keys.wakeUpRoutine.key);
            const registered = await isRegistered(shortcutKeys);
            if (registered) {
                console.log("快捷键已经注册，注销后重新注注册")
                await unregister(shortcutKeys);
            }
            await registerOpenWindowKey(shortcutKeys);
            saveWakeUpRoutineKeyAvailable(true);
        }
    } catch (ex) {
        error("打开程序快捷键注册失败，" + ex);
        console.error(ex);
        sendNotification({title: 'EasyPaste', body: language.pages.list.registerShortcutKeysErrorHint});
        saveWakeUpRoutineKeyAvailable(false);
    }
}

/**
 * 初始化窗口失焦定时任务
 * todo tauri://blur 只会生效一次，暂时只能用这种笨办法
 */
let focusState = false; // 窗口聚焦状态
let webFocusStatus = false; // web网页聚焦状态
let blurTimer: any = null;

function initBlurTimer() {
    return setInterval(async () => {
        window.onfocus = () => {
            webFocusStatus = true;
        }
        window.onblur = () => {
            webFocusStatus = false;
        }

        const win = getCurrentWebviewWindow();
        const focused = await win.isFocused();
        if (!webFocusStatus && ((focusState && !focused) || (!focusState && !focused))) {
            const visible = await win.isVisible();
            if (visible) {
                await hideWindow();
            }
        }
        if (focused !== focusState) {
            focusState = focused;
        }
    }, 200); // 时间可调整
}

/**
 * 初始化打开设置窗口事件监听
 */
let openSettingsListener: any = null;

function initOpenSettingsListener() {
    return listen('open-settings', (_event: any) => {
        openSettingsWindow();
    });
}

/**
 * 初始化打开关于窗口事件监听
 */
let openAboutListener: any = null;

function initOpenAboutListener() {
    return listen('open-about', (_event: any) => {
        openAboutWindow();
    });
}

/**
 * 初始化手动检查更新事件监听
 */
let checkUpdateUnListener: any = null;

async function initCheckUpdateListener() {
    return await listen('check-update', async (_event: any) => {
        const update = UpdaterService.getInstance();
        await update.checkForUpdates(true);
    });
}

/**
 * 初始化修改快捷键打开当前窗口监听
 */
let updateRegisterShortcutKeysOpenWindowListener: any = null;

async function initUpdateRegisterShortcutKeysOpenWindowListener() {
    return await listen('update-open-window-key', async (event: any) => {
        const keys: ShortcutKeys = event.payload.keys;
        if (keys.wakeUpRoutine && keys.wakeUpRoutine.key && keys.wakeUpRoutine.key.length > 0) {
            const shortcutKeys = convertRegisterKey(keys.wakeUpRoutine.key);
            await registerOpenWindowKey(shortcutKeys);
        }
    });
}

/**
 * 初始化更新自动隐藏窗口监听
 */
let updateAutoHideWindowListener: any = null;

async function initUpdateAutoHideWindowListener() {
    return await listen('update-auto-hide-window', async (event: any) => {
        const autoHideWindow = event.payload.isAutoHide;
        console.log('更新自动隐藏窗口状态', autoHideWindow)
        isAutoHideWindow.value = autoHideWindow;
        const window = getCurrentWebviewWindow();
        if (autoHideWindow) {
            if (blurTimer) {
                clearInterval(blurTimer);
            }
            // 添加监听窗口失焦自动隐藏定时任务
            blurTimer = initBlurTimer();
            await window.setSkipTaskbar(true);
        } else {
            // 清除窗口失焦定时任务
            if (blurTimer) {
                clearInterval(blurTimer);
            }
            await window.setSkipTaskbar(false);
        }
    });
}

/**
 * 初始化更新窗口置顶状态监听
 */
let updateAlwaysOnTopListener: any = null;

function initUpdateAlwaysOnTopListener() {
    return listen('update-always-on-top', async (event: any) => {
        const isTop = event.payload.isTop;
        const window = getCurrentWebviewWindow();
        await window.setAlwaysOnTop(isTop);
    });
}

/**
 * 初始化窗口上下文
 * @param settings 程序配置信息
 * @param language 语言信息
 */
export async function initializeWindowContext(settings: Settings, language: LanguageConfig) {
    try {
        // 添加监听窗口失焦自动隐藏定时任务
        isAutoHideWindow.value = settings.autoHideWindow;
        if (settings.autoHideWindow) {
            blurTimer = initBlurTimer();
        } else {
            const window = getCurrentWebviewWindow();
            await window.setSkipTaskbar(false);
        }
        // 窗口设置不置顶，取消置顶状态
        if (!settings.alwaysOnTop) {
            const window = getCurrentWebviewWindow();
            await window.setAlwaysOnTop(false);
        }

        // 添加打开设置窗口事件监听
        openSettingsListener = await initOpenSettingsListener();

        // 添加打开关于窗口事件监听
        openAboutListener = await initOpenAboutListener();

        // 添加检查更新事件监听
        checkUpdateUnListener = await initCheckUpdateListener();

        // 注册快捷键打开当前窗口
        await registerShortcutKeysOpenWindow(language);

        // 重新注册快捷键打开当前窗口
        updateRegisterShortcutKeysOpenWindowListener = await initUpdateRegisterShortcutKeysOpenWindowListener();

        // 添加更新自动隐藏窗口事件监听
        updateAutoHideWindowListener = await initUpdateAutoHideWindowListener();

        // 添加修改窗口置顶状态事件监听
        updateAlwaysOnTopListener = await initUpdateAlwaysOnTopListener();
    } catch (e) {
        console.error("窗口上下文初始化失败", e);
        error("窗口上下文初始化失败" + e);
    }
}

/**
 * 销毁窗口上下文
 */
export function destroyWindowContext() {
    // 清除窗口失焦定时任务
    if (blurTimer) {
        clearInterval(blurTimer);
    }
    // 清除打开设置窗口事件监听
    if (openSettingsListener) {
        openSettingsListener();
    }
    // 清除打开关于窗口事件监听
    if (openAboutListener) {
        openAboutListener();
    }
    // 清除检查更新事件监听
    if (checkUpdateUnListener) {
        checkUpdateUnListener();
    }
    // 清除重新注册快捷键打开当前窗口事件监听
    if (updateRegisterShortcutKeysOpenWindowListener) {
        updateRegisterShortcutKeysOpenWindowListener();
    }

    // 添加更新自动隐藏窗口事件监听
    if (updateAutoHideWindowListener) {
        updateAutoHideWindowListener();
    }

    // 清除修改窗口置顶状态事件监听
    if (updateAlwaysOnTopListener) {
        updateAlwaysOnTopListener();
    }
}