import Foundation
import TuistCore

/// A project mapper that auto-generates schemes for each of the targets of the graph
/// if the user hasn't already defined schemes for those.
public class AutogeneratedSchemesProjectMapper: ProjectMapping {
    // MARK: - Init

    public init() {}

    // MARK: - GraphMapping

    public func map(project: Project) throws -> (Project, [SideEffectDescriptor]) {
        let schemeNames = Set(project.schemes.map { $0.name })
        let schemes = project.schemes

        let autogeneratedSchemes = project.targets.compactMap { (target: Target) -> Scheme? in
            let scheme = self.createDefaultScheme(target: target,
                                                  project: project,
                                                  buildConfiguration: project.defaultDebugBuildConfigurationName)
            // The user has already defined a scheme with that name.
            if schemeNames.contains(scheme.name) { return nil }
            return scheme
        }

        return (project.with(schemes: schemes + autogeneratedSchemes), [])
    }

    // MARK: - Private

    private func createDefaultScheme(target: Target, project: Project, buildConfiguration: String) -> Scheme {
        let targetReference = TargetReference(projectPath: project.path, name: target.name)

        let buildTargets = buildableTargets(targetReference: targetReference, target: target, project: project)
        let testTargets = testableTargets(targetReference: targetReference, target: target, project: project)
        let executable = runnableExecutable(targetReference: targetReference, target: target, project: project)

        return Scheme(name: target.name,
                      shared: true,
                      buildAction: BuildAction(targets: buildTargets),
                      testAction: TestAction(targets: testTargets,
                                             arguments: nil,
                                             configurationName: buildConfiguration,
                                             coverage: false,
                                             codeCoverageTargets: [],
                                             preActions: [],
                                             postActions: [],
                                             diagnosticsOptions: Set()),
                      runAction: RunAction(configurationName: buildConfiguration,
                                           executable: executable,
                                           filePath: nil,
                                           arguments: Arguments(environment: target.environment),
                                           diagnosticsOptions: Set()))
    }

    private func buildableTargets(targetReference: TargetReference,
                                  target: Target,
                                  project: Project) -> [TargetReference] {
        switch target.product {
        case .appExtension, .messagesExtension:
            let hostAppTargets = hostAppTargetReferences(for: target, project: project)
            return [targetReference] + hostAppTargets
        default:
            return [targetReference]
        }
    }

    private func testableTargets(targetReference: TargetReference,
                                 target: Target,
                                 project: Project) -> [TestableTarget] {
        if target.product.testsBundle {
            return [TestableTarget(target: targetReference)]
        } else {
            // The test targets that are dependant on the given target.
            return project.targets
                .filter { $0.product.testsBundle && $0.dependencies.contains(.target(name: target.name)) }
                .sorted(by: { $0.name < $1.name })
                .map { TargetReference(projectPath: project.path, name: $0.name) }
                .map { TestableTarget(target: $0) }
        }
    }

    private func runnableExecutable(targetReference: TargetReference,
                                    target: Target,
                                    project: Project) -> TargetReference? {
        switch target.product {
        case .appExtension, .messagesExtension:
            return hostAppTargetReferences(for: target, project: project).first
        default:
            return targetReference
        }
    }

    private func hostAppTargetReferences(for target: Target,
                                         project: Project) -> [TargetReference] {
        project.targets
            .filter { $0.product == .app && $0.dependencies.contains(.target(name: target.name)) }
            .sorted(by: { $0.name < $1.name })
            .map { TargetReference(projectPath: project.path, name: $0.name) }
    }
}
