import Foundation
import TSCBasic

public enum FileHandlerError: FatalError, Equatable {
    case invalidTextEncoding(AbsolutePath)
    case writingError(AbsolutePath)
    case fileNotFound(AbsolutePath)
    case unreachableFileSize(AbsolutePath)
    case unconvertibleToData(AbsolutePath)
    case expectedAFile(AbsolutePath)

    public var description: String {
        switch self {
        case let .invalidTextEncoding(path):
            return "The file at \(path.pathString) is not a utf8 text file"
        case let .writingError(path):
            return "Couldn't write to the file \(path.pathString)"
        case let .fileNotFound(path):
            return "File not found at \(path.pathString)"
        case let .unreachableFileSize(path):
            return "Could not get the file size at path \(path.pathString)"
        case let .unconvertibleToData(path):
            return "Could not convert to Data the file content (at path \(path.pathString))"
        case let .expectedAFile(path):
            return "Could not find a file at path \(path.pathString))"
        }
    }

    public var type: ErrorType {
        switch self {
        case .invalidTextEncoding:
            return .bug
        case .writingError, .fileNotFound, .unreachableFileSize, .unconvertibleToData, .expectedAFile:
            return .abort
        }
    }
}

/// Protocol that defines the interface of an object that provides convenient
/// methods to interact with the system files and folders.
public protocol FileHandling: AnyObject {
    /// Returns the current path.
    var currentPath: AbsolutePath { get }

    /// Returns `AbsolutePath` to home directory
    var homeDirectory: AbsolutePath { get }

    /// Replaces a file/directory in a given path with another one.
    ///
    /// - Parameters:
    ///   - to: The file/directory to be replaced.
    ///   - with: The replacement file or directory.
    func replace(_ to: AbsolutePath, with: AbsolutePath) throws

    /// Returns true if there's a folder or file at the given path.
    ///
    /// - Parameter path: Path to check.
    /// - Returns: True if there's a folder or file at the given path.
    func exists(_ path: AbsolutePath) -> Bool

    /// Move a file from a location to another location
    ///
    /// - Parameters:
    ///   - from: File/Folder to be moved.
    ///   - to: Path where the file/folder will be moved.
    /// - Throws: An error if from doesn't exist or to does.
    func move(from: AbsolutePath, to: AbsolutePath) throws

    /// It copies a file or folder to another path.
    ///
    /// - Parameters:
    ///   - from: File/Folder to be copied.
    ///   - to: Path where the file/folder will be copied.
    /// - Throws: An error if from doesn't exist or to does.
    func copy(from: AbsolutePath, to: AbsolutePath) throws

    /// Reads a file at the given path and returns its data.
    ///
    /// - Parameter at: Path to the text file.
    /// - Returns: The content of the file.
    /// - Throws: An error if the file doesn't exist
    func readFile(_ at: AbsolutePath) throws -> Data

    /// Reads a text file at the given path and returns it.
    ///
    /// - Parameter at: Path to the text file.
    /// - Returns: The content of the text file.
    /// - Throws: An error if the file doesn't exist or it's not a valid text file.
    func readTextFile(_ at: AbsolutePath) throws -> String

    /// Reads a plist file at the given path and return decoded data
    ///
    /// - Parameter at: Path to the plist file.
    /// - Returns: The content of the plist file in data format
    /// - Throws: An error if the file doesn't exist or it's not a valid plist file.
    func readPlistFile<T: Decodable>(_ at: AbsolutePath) throws -> T

    /// Runs the given closure passing a temporary directory to it. When the closure
    /// finishes its execution, the temporary directory gets destroyed.
    ///
    /// - Parameter closure: Closure to be executed with the temporary directory.
    /// - Throws: An error if the temporary directory cannot be created or the closure throws.
    func inTemporaryDirectory(_ closure: (AbsolutePath) throws -> Void) throws

    /// Writes a string into the given path (using the utf8 encoding)
    ///
    /// - Parameters:
    ///   - content: Content to be written.
    ///   - path: Path where the content will be written into.
    ///   - atomically: Whether the content should be written atomically.
    /// - Throws: An error if the writing fails.
    func write(_ content: String, path: AbsolutePath, atomically: Bool) throws

    /// Traverses the parent directories until the given path is found.
    ///
    /// - Parameters:
    ///   - from: A path to a directory from which search the Config.swift.
    /// - Returns: The found path.
    func locateDirectoryTraversingParents(from: AbsolutePath, path: String) -> AbsolutePath?

    /// It traverses up the directories hierarchy appending the given path and returning the
    /// resulting path if it exists.
    /// - Parameters:
    ///   - path: Relative path to append to each path in the hierarchy.
    ///   - from: Path to traverse the hierarchy from.
    func locateDirectory(_ path: String, traversingFrom from: AbsolutePath) -> AbsolutePath?

    func glob(_ path: AbsolutePath, glob: String) -> [AbsolutePath]
    func throwingGlob(_ path: AbsolutePath, glob: String) throws -> [AbsolutePath]
    func linkFile(atPath: AbsolutePath, toPath: AbsolutePath) throws
    func createFolder(_ path: AbsolutePath) throws
    func delete(_ path: AbsolutePath) throws
    func isFolder(_ path: AbsolutePath) -> Bool
    func touch(_ path: AbsolutePath) throws
    func contentsOfDirectory(_ path: AbsolutePath) throws -> [AbsolutePath]

    /// Gives a md5 representation of the given file
    ///
    /// - Parameters:
    ///   - path: File to be assessed.
    /// - Returns: The file’s md5 as an utf8 encoded String.
    /// - Throws: An error if path's file data content can't be accessed.
    func md5(path: AbsolutePath) throws -> String

    /// Gives a base 64 md5 representation of the given file
    ///
    /// - Parameters:
    ///   - path: File to be assessed.
    /// - Returns: The file’s md5 as an utf8 base 64 encoded String.
    /// - Throws: An error if path's file data content can't be accessed.
    func base64MD5(path: AbsolutePath) throws -> String

    /// Gives the size of the given file, in bytes
    ///
    /// - Parameters:
    ///   - path: File to be assessed.
    /// - Returns: The file’s size in bytes.
    /// - Throws: An error if path's file size can't be retrieved.
    func fileSize(path: AbsolutePath) throws -> UInt64

    func changeExtension(path: AbsolutePath, to newExtension: String) throws -> AbsolutePath
}

public class FileHandler: FileHandling {
    // MARK: - Attributes

    public static var shared: FileHandling = FileHandler()
    private let fileManager: FileManager
    private let propertyListDecoder = PropertyListDecoder()

    /// Initializes the file handler with its attributes.
    ///
    /// - Parameter fileManager: File manager instance.
    public init(fileManager: FileManager = .default) {
        self.fileManager = fileManager
    }

    public var currentPath: AbsolutePath {
        AbsolutePath(fileManager.currentDirectoryPath)
    }

    public var homeDirectory: AbsolutePath {
        AbsolutePath(NSHomeDirectory())
    }

    public func replace(_ to: AbsolutePath, with: AbsolutePath) throws {
        // To support cases where the destination is on a different volume
        // we need to create a temporary directory that is suitable
        // for performing a `replaceItemAt`
        //
        // References:
        // - https://developer.apple.com/documentation/foundation/filemanager/2293212-replaceitemat
        // - https://developer.apple.com/documentation/foundation/filemanager/1407693-url
        // - https://openradar.appspot.com/50553219
        let rootTempDir = try fileManager.url(for: .itemReplacementDirectory,
                                              in: .userDomainMask,
                                              appropriateFor: to.url,
                                              create: true)
        let tempUrl = rootTempDir.appendingPathComponent("temp")
        defer { try? fileManager.removeItem(at: rootTempDir) }
        try fileManager.copyItem(at: with.url, to: tempUrl)
        _ = try fileManager.replaceItemAt(to.url, withItemAt: tempUrl)
    }

    public func inTemporaryDirectory(_ closure: (AbsolutePath) throws -> Void) throws {
        try withTemporaryDirectory(closure)
    }

    public func exists(_ path: AbsolutePath) -> Bool {
        let exists = fileManager.fileExists(atPath: path.pathString)
        logger.debug("Checking if \(path) exists... \(exists)")
        return exists
    }

    public func copy(from: AbsolutePath, to: AbsolutePath) throws {
        logger.debug("Copying file from \(from) to \(to)")
        try fileManager.copyItem(atPath: from.pathString, toPath: to.pathString)
    }

    public func move(from: AbsolutePath, to: AbsolutePath) throws {
        logger.debug("Moving file from \(from) to \(to)")
        try fileManager.moveItem(atPath: from.pathString, toPath: to.pathString)
    }

    public func readFile(_ at: AbsolutePath) throws -> Data {
        logger.debug("Reading contents of file at path \(at)")
        return try Data(contentsOf: at.url)
    }

    public func readTextFile(_ at: AbsolutePath) throws -> String {
        logger.debug("Reading contents of text file at path \(at)")
        let data = try Data(contentsOf: at.url)
        if let content = String(data: data, encoding: .utf8) {
            return content
        } else {
            throw FileHandlerError.invalidTextEncoding(at)
        }
    }

    public func readPlistFile<T: Decodable>(_ at: AbsolutePath) throws -> T {
        logger.debug("Reading contents of plist file at path \(at)")
        guard let data = fileManager.contents(atPath: at.pathString) else {
            throw FileHandlerError.fileNotFound(at)
        }
        return try propertyListDecoder.decode(T.self, from: data)
    }

    public func linkFile(atPath: AbsolutePath, toPath: AbsolutePath) throws {
        logger.debug("Creating a link from \(atPath) to \(toPath)")
        try fileManager.linkItem(atPath: atPath.pathString, toPath: toPath.pathString)
    }

    public func write(_ content: String, path: AbsolutePath, atomically: Bool) throws {
        logger.debug("Writing contents to file \(path) atomically \(atomically)")
        do {
            try content.write(to: path.url, atomically: atomically, encoding: .utf8)
        } catch {}
    }

    public func locateDirectory(_ path: String, traversingFrom from: AbsolutePath) -> AbsolutePath? {
        logger.debug("Traversing \(from) to locate \(path)")
        let extendedPath = from.appending(RelativePath(path))
        if exists(extendedPath) {
            return extendedPath
        } else if !from.isRoot {
            return locateDirectory(path, traversingFrom: from.parentDirectory)
        } else {
            return nil
        }
    }

    public func glob(_ path: AbsolutePath, glob: String) -> [AbsolutePath] {
        path.glob(glob)
    }

    public func throwingGlob(_ path: AbsolutePath, glob: String) throws -> [AbsolutePath] {
        try path.throwingGlob(glob)
    }

    public func createFolder(_ path: AbsolutePath) throws {
        logger.debug("Creating folder at path \(path)")
        try fileManager.createDirectory(at: path.url,
                                        withIntermediateDirectories: true,
                                        attributes: nil)
    }

    public func delete(_ path: AbsolutePath) throws {
        logger.debug("Deleting item at path \(path)")
        if exists(path) {
            try fileManager.removeItem(atPath: path.pathString)
        }
    }

    public func touch(_ path: AbsolutePath) throws {
        logger.debug("Touching \(path)")
        try fileManager.createDirectory(at: path.removingLastComponent().url,
                                        withIntermediateDirectories: true,
                                        attributes: nil)
        try Data().write(to: path.url)
    }

    public func isFolder(_ path: AbsolutePath) -> Bool {
        var isDirectory = ObjCBool(true)
        let exists = fileManager.fileExists(atPath: path.pathString, isDirectory: &isDirectory)
        return exists && isDirectory.boolValue
    }

    public func locateDirectoryTraversingParents(from: AbsolutePath, path: String) -> AbsolutePath? {
        logger.debug("Traversing \(from) to locate \(path)")

        let configPath = from.appending(component: path)

        if FileHandler.shared.exists(configPath) {
            return configPath
        } else if from == AbsolutePath("/") {
            return nil
        } else {
            return locateDirectoryTraversingParents(from: from.parentDirectory, path: path)
        }
    }

    public func contentsOfDirectory(_ path: AbsolutePath) throws -> [AbsolutePath] {
        try fileManager.contentsOfDirectory(atPath: path.pathString).map { AbsolutePath(path, $0) }
    }

    // MARK: - MD5

    public func md5(path: AbsolutePath) throws -> String {
        try Data(contentsOf: path.url).md5
    }

    public func base64MD5(path: AbsolutePath) throws -> String {
        guard let utf8str = try md5(path: path).data(using: .utf8) else {
            throw FileHandlerError.unconvertibleToData(path)
        }
        return utf8str.base64EncodedString()
    }

    // MARK: - File Attributes

    public func fileSize(path: AbsolutePath) throws -> UInt64 {
        let attr = try fileManager.attributesOfItem(atPath: path.pathString)
        guard let size = attr[FileAttributeKey.size] as? UInt64 else { throw FileHandlerError.unreachableFileSize(path) }
        return size
    }

    // MARK: - Extension

    public func changeExtension(path: AbsolutePath, to fileExtension: String) throws -> AbsolutePath {
        guard isFolder(path) == false else { throw FileHandlerError.expectedAFile(path) }
        let sanitizedExtension = fileExtension.starts(with: ".") ? String(fileExtension.dropFirst()) : fileExtension
        guard path.extension != sanitizedExtension else { return path }

        let newPath = path.removingLastComponent().appending(component: "\(path.basenameWithoutExt).\(sanitizedExtension)")
        try move(from: path, to: newPath)
        return newPath
    }
}
