import Foundation
import TSCBasic
import TuistCore
import XCTest

@testable import TuistGenerator
@testable import TuistSupport
@testable import TuistSupportTesting

final class AutogeneratedSchemesProjectMapperTests: TuistUnitTestCase {
    var subject: AutogeneratedSchemesProjectMapper!

    override func setUp() {
        super.setUp()
        subject = AutogeneratedSchemesProjectMapper()
    }

    override func tearDown() {
        super.tearDown()
        subject = nil
    }

    func test_map() throws {
        // Given
        let targetA = Target.test(name: "A")
        let targetB = Target.test(name: "B",
                                  product: .unitTests,
                                  dependencies: [.target(name: "A")])
        let project = Project.test(targets: [targetA, targetB])

        // When
        let (got, sideEffects) = try subject.map(project: project)

        // Then
        XCTAssertEmpty(sideEffects)
        XCTAssertEqual(got.schemes.count, 2)

        // Then: A
        let aScheme = got.schemes.first!
        XCTAssertEqual(aScheme.name, targetA.name)
        let aBuildAction = try XCTUnwrap(aScheme.buildAction)
        XCTAssertEqual(aBuildAction.targets.count, 1)
        let aBuildTargetReference = try XCTUnwrap(aBuildAction.targets.first)
        XCTAssertEqual(aBuildTargetReference.projectPath, project.path)
        XCTAssertEqual(aBuildTargetReference.name, targetA.name)

        // Then: A tests
        // Since B is a tests bundle with a dependency on A,
        // there should be a test action in A scheme to run the tests in B
        let aTestAction = try XCTUnwrap(aScheme.testAction)
        XCTAssertEqual(aTestAction.targets.count, 1)
        let aTestTargetReference = try XCTUnwrap(aTestAction.targets.first?.target)
        XCTAssertEqual(aTestTargetReference.projectPath, project.path)
        XCTAssertEqual(aTestTargetReference.name, targetB.name)

        // Then: B
        let bScheme = got.schemes.last!
        XCTAssertEqual(bScheme.name, targetB.name)
        let bBuildAction = try XCTUnwrap(bScheme.buildAction)
        XCTAssertEqual(bBuildAction.targets.count, 1)
        let bBuildTargetReference = try XCTUnwrap(bBuildAction.targets.first)
        XCTAssertEqual(bBuildTargetReference.projectPath, project.path)
        XCTAssertEqual(bBuildTargetReference.name, targetB.name)
        let bTestAction = try XCTUnwrap(bScheme.testAction)
        XCTAssertEqual(bTestAction.targets.count, 1)
        let bTestTargetReference = try XCTUnwrap(bTestAction.targets.first?.target)
        XCTAssertEqual(bTestTargetReference.projectPath, project.path)
        XCTAssertEqual(bTestTargetReference.name, targetB.name)
    }

    func test_map_doesnt_override_user_schemes() throws {
        // Given
        let targetA = Target.test(name: "A")
        let aScheme = Scheme.test(name: "A",
                                  shared: true,
                                  buildAction: nil,
                                  testAction: nil,
                                  runAction: nil,
                                  archiveAction: nil,
                                  profileAction: nil,
                                  analyzeAction: nil)
        let project = Project.test(targets: [targetA],
                                   schemes: [aScheme])

        // When
        let (got, sideEffects) = try subject.map(project: project)

        // Then
        XCTAssertEmpty(sideEffects)
        XCTAssertEqual(got.schemes.count, 1)

        // Then: A
        let gotAScheme = got.schemes.first!
        XCTAssertNil(gotAScheme.buildAction)
    }

    func test_map_appExtensions() throws {
        // Given
        let path = AbsolutePath("/test")
        let app = Target.test(name: "App",
                              product: .app,
                              dependencies: [
                                  .target(name: "AppExtension"),
                                  .target(name: "MessageExtension"),
                              ])
        let appExtension = Target.test(name: "AppExtension", product: .appExtension)
        let messageExtension = Target.test(name: "MessageExtension", product: .messagesExtension)

        let project = Project.test(path: path, targets: [app, appExtension, messageExtension])

        // When
        let (got, _) = try subject.map(project: project)

        // Then
        let buildActions = got.schemes.map(\.buildAction?.targets)
        XCTAssertEqual(buildActions, [
            [TargetReference(projectPath: path, name: "App")],
            [TargetReference(projectPath: path, name: "AppExtension"), TargetReference(projectPath: path, name: "App")],
            [TargetReference(projectPath: path, name: "MessageExtension"), TargetReference(projectPath: path, name: "App")],
        ])

        let runActions = got.schemes.map(\.runAction?.executable)
        XCTAssertEqual(runActions, [
            TargetReference(projectPath: path, name: "App"),
            TargetReference(projectPath: path, name: "App"), // Extensions set their host app as the runnable target
            TargetReference(projectPath: path, name: "App"), // Extensions set their host app as the runnable target
        ])
    }
}
