# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)
from localstack.utils.objects import keys_to_lower


class ApiGatewayApiKeyProperties(TypedDict):
    APIKeyId: str | None
    CustomerId: str | None
    Description: str | None
    Enabled: bool | None
    GenerateDistinctId: bool | None
    Name: str | None
    StageKeys: list[StageKey] | None
    Tags: list[Tag] | None
    Value: str | None


class StageKey(TypedDict):
    RestApiId: str | None
    StageName: str | None


class Tag(TypedDict):
    Key: str | None
    Value: str | None


REPEATED_INVOCATION = "repeated_invocation"


class ApiGatewayApiKeyProvider(ResourceProvider[ApiGatewayApiKeyProperties]):
    TYPE = "AWS::ApiGateway::ApiKey"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[ApiGatewayApiKeyProperties],
    ) -> ProgressEvent[ApiGatewayApiKeyProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/APIKeyId


        Create-only properties:
          - /properties/GenerateDistinctId
          - /properties/Name
          - /properties/Value

        Read-only properties:
          - /properties/APIKeyId

        IAM permissions required:
          - apigateway:POST
          - apigateway:GET

        """
        model = request.desired_state
        apigw = request.aws_client_factory.apigateway

        params = util.select_attributes(
            model, ["Description", "CustomerId", "Name", "Value", "Enabled", "StageKeys"]
        )
        params = keys_to_lower(params.copy())
        if "enabled" in params:
            params["enabled"] = bool(params["enabled"])

        if model.get("Tags"):
            params["tags"] = {tag["Key"]: tag["Value"] for tag in model["Tags"]}

        response = apigw.create_api_key(**params)
        model["APIKeyId"] = response["id"]
        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[ApiGatewayApiKeyProperties],
    ) -> ProgressEvent[ApiGatewayApiKeyProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - apigateway:GET
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[ApiGatewayApiKeyProperties],
    ) -> ProgressEvent[ApiGatewayApiKeyProperties]:
        """
        Delete a resource

        IAM permissions required:
          - apigateway:DELETE
        """
        model = request.desired_state
        apigw = request.aws_client_factory.apigateway

        apigw.delete_api_key(apiKey=model["APIKeyId"])
        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[ApiGatewayApiKeyProperties],
    ) -> ProgressEvent[ApiGatewayApiKeyProperties]:
        """
        Update a resource

        IAM permissions required:
          - apigateway:GET
          - apigateway:PATCH
          - apigateway:PUT
          - apigateway:DELETE
        """
        raise NotImplementedError
