# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class ApiGatewayBasePathMappingProperties(TypedDict):
    DomainName: str | None
    BasePath: str | None
    RestApiId: str | None
    Stage: str | None


REPEATED_INVOCATION = "repeated_invocation"


class ApiGatewayBasePathMappingProvider(ResourceProvider[ApiGatewayBasePathMappingProperties]):
    TYPE = "AWS::ApiGateway::BasePathMapping"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[ApiGatewayBasePathMappingProperties],
    ) -> ProgressEvent[ApiGatewayBasePathMappingProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/DomainName
          - /properties/BasePath

        Required properties:
          - DomainName

        Create-only properties:
          - /properties/DomainName
          - /properties/BasePath



        IAM permissions required:
          - apigateway:POST
          - apigateway:GET

        """

        # TODO we are using restApiId for PhysicalResourceId
        #  check if we need to change it
        model = request.desired_state
        apigw = request.aws_client_factory.apigateway

        params = {
            "domainName": model.get("DomainName"),
            "restApiId": model.get("RestApiId"),
            **({"basePath": model.get("BasePath")} if model.get("BasePath") else {}),
            **({"stage": model.get("Stage")} if model.get("Stage") else {}),
        }
        response = apigw.create_base_path_mapping(**params)
        model["RestApiId"] = response["restApiId"]
        # TODO: validations

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[ApiGatewayBasePathMappingProperties],
    ) -> ProgressEvent[ApiGatewayBasePathMappingProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - apigateway:GET
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[ApiGatewayBasePathMappingProperties],
    ) -> ProgressEvent[ApiGatewayBasePathMappingProperties]:
        """
        Delete a resource

        IAM permissions required:
          - apigateway:DELETE
        """
        model = request.desired_state
        apigw = request.aws_client_factory.apigateway

        apigw.delete_base_path_mapping(domainName=model["DomainName"], basePath=model["BasePath"])

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[ApiGatewayBasePathMappingProperties],
    ) -> ProgressEvent[ApiGatewayBasePathMappingProperties]:
        """
        Update a resource

        IAM permissions required:
          - apigateway:GET
          - apigateway:DELETE
          - apigateway:PATCH
        """
        raise NotImplementedError
