# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)
from localstack.utils.objects import keys_to_lower


class ApiGatewayGatewayResponseProperties(TypedDict):
    ResponseType: str | None
    RestApiId: str | None
    Id: str | None
    ResponseParameters: dict | None
    ResponseTemplates: dict | None
    StatusCode: str | None


REPEATED_INVOCATION = "repeated_invocation"


class ApiGatewayGatewayResponseProvider(ResourceProvider[ApiGatewayGatewayResponseProperties]):
    TYPE = "AWS::ApiGateway::GatewayResponse"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[ApiGatewayGatewayResponseProperties],
    ) -> ProgressEvent[ApiGatewayGatewayResponseProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Id

        Required properties:
          - ResponseType
          - RestApiId

        Create-only properties:
          - /properties/ResponseType
          - /properties/RestApiId

        Read-only properties:
          - /properties/Id

        IAM permissions required:
          - apigateway:PUT
          - apigateway:GET

        """
        model = request.desired_state
        api = request.aws_client_factory.apigateway
        # TODO: validations
        model["Id"] = util.generate_default_name_without_stack(request.logical_resource_id)

        params = util.select_attributes(
            model,
            ["RestApiId", "ResponseType", "StatusCode", "ResponseParameters", "ResponseTemplates"],
        )
        params = keys_to_lower(params.copy())

        api.put_gateway_response(**params)
        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[ApiGatewayGatewayResponseProperties],
    ) -> ProgressEvent[ApiGatewayGatewayResponseProperties]:
        """
        Fetch resource information


        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[ApiGatewayGatewayResponseProperties],
    ) -> ProgressEvent[ApiGatewayGatewayResponseProperties]:
        """
        Delete a resource

        IAM permissions required:
          - apigateway:GET
          - apigateway:DELETE
        """
        model = request.desired_state
        api = request.aws_client_factory.apigateway

        api.delete_gateway_response(
            restApiId=model["RestApiId"], responseType=model["ResponseType"]
        )
        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[ApiGatewayGatewayResponseProperties],
    ) -> ProgressEvent[ApiGatewayGatewayResponseProperties]:
        """
        Update a resource

        IAM permissions required:
          - apigateway:GET
          - apigateway:PUT
        """
        raise NotImplementedError
