# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class ApiGatewayRequestValidatorProperties(TypedDict):
    RestApiId: str | None
    Name: str | None
    RequestValidatorId: str | None
    ValidateRequestBody: bool | None
    ValidateRequestParameters: bool | None


REPEATED_INVOCATION = "repeated_invocation"


class ApiGatewayRequestValidatorProvider(ResourceProvider[ApiGatewayRequestValidatorProperties]):
    TYPE = "AWS::ApiGateway::RequestValidator"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[ApiGatewayRequestValidatorProperties],
    ) -> ProgressEvent[ApiGatewayRequestValidatorProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/RestApiId
          - /properties/RequestValidatorId

        Required properties:
          - RestApiId

        Create-only properties:
          - /properties/Name
          - /properties/RestApiId

        Read-only properties:
          - /properties/RequestValidatorId

        IAM permissions required:
          - apigateway:POST
          - apigateway:GET

        """
        model = request.desired_state
        api = request.aws_client_factory.apigateway

        if not model.get("Name"):
            model["Name"] = util.generate_default_name(
                request.stack_name, request.logical_resource_id
            )
        response = api.create_request_validator(
            name=model["Name"],
            restApiId=model["RestApiId"],
            validateRequestBody=model.get("ValidateRequestBody", False),
            validateRequestParameters=model.get("ValidateRequestParameters", False),
        )
        model["RequestValidatorId"] = response["id"]
        # FIXME error happens when other resources try to reference this one
        #  "An error occurred (BadRequestException) when calling the PutMethod operation:
        #  Invalid Request Validator identifier specified"
        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[ApiGatewayRequestValidatorProperties],
    ) -> ProgressEvent[ApiGatewayRequestValidatorProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - apigateway:GET
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[ApiGatewayRequestValidatorProperties],
    ) -> ProgressEvent[ApiGatewayRequestValidatorProperties]:
        """
        Delete a resource

        IAM permissions required:
          - apigateway:DELETE
        """
        model = request.desired_state
        api = request.aws_client_factory.apigateway

        api.delete_request_validator(
            restApiId=model["RestApiId"], requestValidatorId=model["RequestValidatorId"]
        )

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[ApiGatewayRequestValidatorProperties],
    ) -> ProgressEvent[ApiGatewayRequestValidatorProperties]:
        """
        Update a resource

        IAM permissions required:
          - apigateway:PATCH
        """
        raise NotImplementedError
