# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)
from localstack.utils.objects import keys_to_lower


class ApiGatewayUsagePlanKeyProperties(TypedDict):
    KeyId: str | None
    KeyType: str | None
    UsagePlanId: str | None
    Id: str | None


REPEATED_INVOCATION = "repeated_invocation"


class ApiGatewayUsagePlanKeyProvider(ResourceProvider[ApiGatewayUsagePlanKeyProperties]):
    TYPE = "AWS::ApiGateway::UsagePlanKey"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[ApiGatewayUsagePlanKeyProperties],
    ) -> ProgressEvent[ApiGatewayUsagePlanKeyProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Id

        Required properties:
          - KeyType
          - UsagePlanId
          - KeyId

        Create-only properties:
          - /properties/KeyId
          - /properties/UsagePlanId
          - /properties/KeyType

        Read-only properties:
          - /properties/Id

        IAM permissions required:
          - apigateway:POST
          - apigateway:GET

        """
        model = request.desired_state
        apigw = request.aws_client_factory.apigateway

        params = keys_to_lower(model.copy())
        result = apigw.create_usage_plan_key(**params)

        model["Id"] = result["id"]
        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[ApiGatewayUsagePlanKeyProperties],
    ) -> ProgressEvent[ApiGatewayUsagePlanKeyProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - apigateway:GET
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[ApiGatewayUsagePlanKeyProperties],
    ) -> ProgressEvent[ApiGatewayUsagePlanKeyProperties]:
        """
        Delete a resource

        IAM permissions required:
          - apigateway:DELETE
        """
        model = request.desired_state
        apigw = request.aws_client_factory.apigateway

        apigw.delete_usage_plan_key(usagePlanId=model["UsagePlanId"], keyId=model["KeyId"])

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[ApiGatewayUsagePlanKeyProperties],
    ) -> ProgressEvent[ApiGatewayUsagePlanKeyProperties]:
        """
        Update a resource


        """
        raise NotImplementedError
