# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class CertificateManagerCertificateProperties(TypedDict):
    DomainName: str | None
    CertificateAuthorityArn: str | None
    CertificateTransparencyLoggingPreference: str | None
    DomainValidationOptions: list[DomainValidationOption] | None
    Id: str | None
    SubjectAlternativeNames: list[str] | None
    Tags: list[Tag] | None
    ValidationMethod: str | None


class DomainValidationOption(TypedDict):
    DomainName: str | None
    HostedZoneId: str | None
    ValidationDomain: str | None


class Tag(TypedDict):
    Key: str | None
    Value: str | None


REPEATED_INVOCATION = "repeated_invocation"


class CertificateManagerCertificateProvider(
    ResourceProvider[CertificateManagerCertificateProperties]
):
    TYPE = "AWS::CertificateManager::Certificate"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[CertificateManagerCertificateProperties],
    ) -> ProgressEvent[CertificateManagerCertificateProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Id

        Required properties:
          - DomainName

        Create-only properties:
          - /properties/SubjectAlternativeNames
          - /properties/DomainValidationOptions
          - /properties/ValidationMethod
          - /properties/DomainName
          - /properties/CertificateAuthorityArn

        Read-only properties:
          - /properties/Id



        """
        model = request.desired_state
        acm = request.aws_client_factory.acm

        params = util.select_attributes(
            model,
            [
                "CertificateAuthorityArn",
                "DomainName",
                "DomainValidationOptions",
                "SubjectAlternativeNames",
                "Tags",
                "ValidationMethod",
            ],
        )
        # adjust domain validation options
        valid_opts = params.get("DomainValidationOptions")
        if valid_opts:

            def _convert(opt):
                res = util.select_attributes(opt, ["DomainName", "ValidationDomain"])
                res.setdefault("ValidationDomain", res["DomainName"])
                return res

            params["DomainValidationOptions"] = [_convert(opt) for opt in valid_opts]

        # adjust logging preferences
        logging_pref = params.get("CertificateTransparencyLoggingPreference")
        if logging_pref:
            params["Options"] = {"CertificateTransparencyLoggingPreference": logging_pref}

        response = acm.request_certificate(**params)
        model["Id"] = response["CertificateArn"]

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[CertificateManagerCertificateProperties],
    ) -> ProgressEvent[CertificateManagerCertificateProperties]:
        """
        Fetch resource information


        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[CertificateManagerCertificateProperties],
    ) -> ProgressEvent[CertificateManagerCertificateProperties]:
        """
        Delete a resource


        """
        model = request.desired_state
        acm = request.aws_client_factory.acm

        acm.delete_certificate(CertificateArn=model["Id"])
        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[CertificateManagerCertificateProperties],
    ) -> ProgressEvent[CertificateManagerCertificateProperties]:
        """
        Update a resource


        """
        raise NotImplementedError
