# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)
from localstack.services.cloudformation.stores import get_cloudformation_store


class CloudFormationMacroProperties(TypedDict):
    FunctionName: str | None
    Name: str | None
    Description: str | None
    Id: str | None
    LogGroupName: str | None
    LogRoleARN: str | None


REPEATED_INVOCATION = "repeated_invocation"


class CloudFormationMacroProvider(ResourceProvider[CloudFormationMacroProperties]):
    TYPE = "AWS::CloudFormation::Macro"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[CloudFormationMacroProperties],
    ) -> ProgressEvent[CloudFormationMacroProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Id

        Required properties:
          - FunctionName
          - Name

        Create-only properties:
          - /properties/Name

        Read-only properties:
          - /properties/Id



        """
        model = request.desired_state

        # TODO: fix or validate that we want to keep this here.
        #  AWS::CloudFormation:: resources need special handling since they seem to require access to internal APIs
        store = get_cloudformation_store(request.account_id, request.region_name)
        store.macros[model["Name"]] = model
        model["Id"] = model["Name"]

        return ProgressEvent(status=OperationStatus.SUCCESS, resource_model=model)

    def read(
        self,
        request: ResourceRequest[CloudFormationMacroProperties],
    ) -> ProgressEvent[CloudFormationMacroProperties]:
        """
        Fetch resource information


        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[CloudFormationMacroProperties],
    ) -> ProgressEvent[CloudFormationMacroProperties]:
        """
        Delete a resource


        """
        model = request.desired_state

        store = get_cloudformation_store(request.account_id, request.region_name)
        store.macros.pop(model["Name"], None)

        return ProgressEvent(status=OperationStatus.SUCCESS, resource_model=model)

    def update(
        self,
        request: ResourceRequest[CloudFormationMacroProperties],
    ) -> ProgressEvent[CloudFormationMacroProperties]:
        """
        Update a resource


        """
        raise NotImplementedError
