# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class EC2NetworkAclProperties(TypedDict):
    VpcId: str | None
    Id: str | None
    Tags: list[Tag] | None


class Tag(TypedDict):
    Key: str | None
    Value: str | None


REPEATED_INVOCATION = "repeated_invocation"


class EC2NetworkAclProvider(ResourceProvider[EC2NetworkAclProperties]):
    TYPE = "AWS::EC2::NetworkAcl"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[EC2NetworkAclProperties],
    ) -> ProgressEvent[EC2NetworkAclProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Id

        Required properties:
          - VpcId

        Create-only properties:
          - /properties/VpcId

        Read-only properties:
          - /properties/Id

        IAM permissions required:
          - ec2:CreateNetworkAcl
          - ec2:DescribeNetworkAcls

        """
        model = request.desired_state

        create_params = {
            "VpcId": model["VpcId"],
        }

        if model.get("Tags"):
            create_params["TagSpecifications"] = [
                {
                    "ResourceType": "network-acl",
                    "Tags": [{"Key": tag["Key"], "Value": tag["Value"]} for tag in model["Tags"]],
                }
            ]

        response = request.aws_client_factory.ec2.create_network_acl(**create_params)
        model["Id"] = response["NetworkAcl"]["NetworkAclId"]

        return ProgressEvent(status=OperationStatus.SUCCESS, resource_model=model)

    def read(
        self,
        request: ResourceRequest[EC2NetworkAclProperties],
    ) -> ProgressEvent[EC2NetworkAclProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - ec2:DescribeNetworkAcls
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[EC2NetworkAclProperties],
    ) -> ProgressEvent[EC2NetworkAclProperties]:
        """
        Delete a resource

        IAM permissions required:
          - ec2:DeleteNetworkAcl
          - ec2:DescribeNetworkAcls
        """
        model = request.desired_state
        request.aws_client_factory.ec2.delete_network_acl(NetworkAclId=model["Id"])
        return ProgressEvent(status=OperationStatus.SUCCESS, resource_model={})

    def update(
        self,
        request: ResourceRequest[EC2NetworkAclProperties],
    ) -> ProgressEvent[EC2NetworkAclProperties]:
        """
        Update a resource

        IAM permissions required:
          - ec2:DescribeNetworkAcls
          - ec2:DeleteTags
          - ec2:CreateTags
        """
        raise NotImplementedError
