# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class EC2RouteTableProperties(TypedDict):
    VpcId: str | None
    RouteTableId: str | None
    Tags: list[Tag] | None


class Tag(TypedDict):
    Key: str | None
    Value: str | None


REPEATED_INVOCATION = "repeated_invocation"


class EC2RouteTableProvider(ResourceProvider[EC2RouteTableProperties]):
    TYPE = "AWS::EC2::RouteTable"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[EC2RouteTableProperties],
    ) -> ProgressEvent[EC2RouteTableProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/RouteTableId

        Required properties:
          - VpcId

        Create-only properties:
          - /properties/VpcId

        Read-only properties:
          - /properties/RouteTableId

        IAM permissions required:
          - ec2:CreateRouteTable
          - ec2:CreateTags
          - ec2:DescribeRouteTables

        """
        model = request.desired_state
        ec2 = request.aws_client_factory.ec2
        # TODO: validations
        params = util.select_attributes(model, ["VpcId", "Tags"])

        tags = [{"ResourceType": "route-table", "Tags": params.get("Tags", [])}]

        response = ec2.create_route_table(VpcId=params["VpcId"], TagSpecifications=tags)
        model["RouteTableId"] = response["RouteTable"]["RouteTableId"]

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[EC2RouteTableProperties],
    ) -> ProgressEvent[EC2RouteTableProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - ec2:DescribeRouteTables
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[EC2RouteTableProperties],
    ) -> ProgressEvent[EC2RouteTableProperties]:
        """
        Delete a resource

        IAM permissions required:
          - ec2:DescribeRouteTables
          - ec2:DeleteRouteTable
        """
        model = request.desired_state
        ec2 = request.aws_client_factory.ec2
        try:
            ec2.delete_route_table(RouteTableId=model["RouteTableId"])
        except ec2.exceptions.ClientError:
            pass

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[EC2RouteTableProperties],
    ) -> ProgressEvent[EC2RouteTableProperties]:
        """
        Update a resource

        IAM permissions required:
          - ec2:CreateTags
          - ec2:DeleteTags
          - ec2:DescribeRouteTables
        """
        raise NotImplementedError
